//================================================================================
//================================================================================
//
//	Baseline.fsh
//  Facet Shader for general light and texturing
//
//  Various defines for enabling and disabling features at compile time.
//
//  NORMALISE_NORMALS			- normalises surface normals
//  DISCARD_ON_ALPHA				- discards pixels if the material's alpha is less than a threshold
//  TEXTURING_0					- enables Texturing unit 0
//  TEXTURING_1_MIX				- enables Texturing unit 1 with alpha merging
//  TEXTURING_1_OVERLAY			- enabled Texturing unit 1 with additive merging with texture_1 alpha.
//  DIRECTIONAL_LIGHT_0			- enables directional lighting
//  SPECULAR						- enables specular highlights
//  POINT_LIGHT_0				- enables point lighting
//  NORMAL_MAPPING				- enabled normal (bump) mapping
//  PARALLAX_MAPPING 
//
//================================================================================
//================================================================================

//#version 130

#if (defined POINT_LIGHT_0) || (defined DIRECTIONAL_LIGHT_0) || (defined POINT_LIGHT_1)
  #define LIGHTING_ENABLED
#endif

uniform mediump vec4 INPUT_SpecularColour;
uniform mediump float INPUT_Shininess;  // Must be highp else device and simulator show different results
uniform sampler2D SAMPLER_Texture1;

#ifdef NORMAL_MAPPING
  uniform lowp int INPUT_NormalMapTextureEnabled;		// Bump mapping
  uniform sampler2D SAMPLER_NormalMap;
#endif

uniform mediump float INPUT_Texture1Mix;			// Mix factor for Texture 1.  0.5 will be half and half.

uniform sampler2D SAMPLER_Texture0;
uniform mediump vec2  INPUT_Texture0Displacement;
uniform lowp int INPUT_Texture0Enabled;

#ifdef DIRECTIONAL_LIGHT_0
  uniform int INPUT_DirectionalLight0Enabled;
  uniform mediump vec3 INPUT_DirectionalLight0Direction;
  uniform mediump vec4 INPUT_DirectionalLight0Specular;
  uniform mediump vec4 INPUT_DirectionalLight0Colour;
#endif

#ifdef POINT_LIGHT_0
  uniform int INPUT_PointSpotLight0Enabled;
  uniform mediump vec4 INPUT_PointSpotLight0Specular;
  uniform mediump vec4 INPUT_PointSpotLight0Colour;
#endif

uniform mediump float INPUT_Reflectivity;				// Used for cube maps

//================================================================================
//
// Parameters passed from the Vertex Shader to the Facet Shader
//
//================================================================================

#ifndef NO_OBJECT_COLOUR
  varying mediump vec4 VtoF_Ambient;
  varying mediump vec4 VtoF_Colour;						// Colour is (vertex colour * diffuse)
#endif

#ifdef TEXTURING_0
  varying mediump vec2 VtoF_TextureCoord;
#endif

//================================================================================

#if (defined TEXTURING_1_SRCA_1MINUSSRCA) || (defined TEXTURING_1_1_SRCA) || (defined TEXTURING_1_CROSSFADE)
  varying mediump vec2 VtoF_SecondaryTextureCoord;
#endif 

//================================================================================

#ifdef CUBE_MAPPING
  varying mediump vec3 VtoF_CubeTextureCoord;
  uniform samplerCube SAMPLER_CubeMap;
#endif

//================================================================================

#ifdef LIGHTING_ENABLED
  varying mediump vec3 VtoF_ViewDirection;
  varying mediump vec3 VtoF_UntransformedViewDirection;
  varying mediump vec3 VtoF_Normal;
#endif

//================================================================================

#ifdef DIRECTIONAL_LIGHT_0
  varying mediump vec3 VtoF_DirectionalLight0_Dir;
  varying mediump   vec3 VtoF_DirectionalLight0_HalfVector;
#endif

//================================================================================

#ifdef POINT_LIGHT_0
  varying mediump vec3 VtoF_PointSpotLight0_Dir;
  varying mediump vec3 VtoF_PointSpotLight0_HalfVector;
#endif

//================================================================================

#ifdef POINT_LIGHT_1
  varying mediump vec3 VtoF_PointSpotLight1_Dir;
  varying mediump vec3 VtoF_PointSpotLight1_HalfVector;
#endif

//================================================================================
//================================================================================
//
//	Facet Shader code
//
//================================================================================
//================================================================================

void main()
{
	mediump vec3 n;
	mediump float NdotL,NdotHV;           
	mediump vec4 AccumulatedColour;
	mediump vec4 MaterialColour;
	mediump float Alpha;
	mediump vec4 DirectionalLight0Effect;
	mediump vec4 PointSpotLight0Effect;
	mediump vec4 PointSpotLight1Effect;
	mediump vec4 HighTemp;
	mediump vec2 newTexCoord;

//================================================================================

#ifdef TEXTURING_0
  #if (defined PARALLAX_MAPPING) && (defined NORMAL_MAPPING)
	// Experimental....
    mediump float height = texture2D(SAMPLER_NormalMap, VtoF_TextureCoord).w;
        
    height = height * 0.02; //0.007; //scale + bias;
    newTexCoord = VtoF_TextureCoord - (height * VtoF_UntransformedViewDirection.xy);
  #else  

    newTexCoord = VtoF_TextureCoord;

    #ifdef TEXTURE_DISPLACEMENTS
	  newTexCoord += INPUT_Texture0Displacement;
    #endif
    	
//    int DitherX = int(mod(newTexCoord.x * 512.0,2.0));
//    int DitherY = int(mod(newTexCoord.y * 512.0,2.0));
//    newTexCoord.y += float(DitherX) / 256.0;
//    newTexCoord.x += float(DitherY) / 256.0;
    	
  #endif    
#endif

//================================================================================

#ifdef LIGHTING_ENABLED
  #ifdef NORMAL_MAPPING
	//
	// When using Normal Mapping the Lighting positions and directions have been 
	// transformed in the into "Tangent Space" by the vertex shader.
	// In "Tangent Space" things are measured relative to the predominant surface tangent.
	// We then use the bumpy texture normals directly as the surface normals.
	//
	// Note that due to shortest-path problems in interpolated normals we MUST
	// normalize... This is still not perfect but generally good enough.
	// 
	n = normalize( texture2D(SAMPLER_NormalMap, newTexCoord).xyz * 2.0 - 1.0);
		
  #else
	//
	// When not using normal mapping the transformed normals are passed to us 
	// from the shader.
	//
	n = -normalize(VtoF_Normal);	// We need to invert here to make geometry be consistent with OpenGL 1
	
    #ifdef TWO_SIDED_LIGHTING
	  // Twizzle that deals with 2-sided lighting.
	  // It relies on a boolean being 0 for false and 1 for true.
	  n = (n * float(!gl_FrontFacing)) - (n * float(gl_FrontFacing));
    #endif

  #endif // !NORMAL_MAPPING
  	
#endif	// LIGHTING_ENABLED

//================================================================================

#ifdef TEXTURING_0	
  // Get the material's physical colour from diffuse and texture
	
  #ifdef NO_OBJECT_COLOUR
   	MaterialColour = texture2D(SAMPLER_Texture0,newTexCoord); 
  #else
   	MaterialColour = VtoF_Colour * texture2D(SAMPLER_Texture0,newTexCoord);
  #endif

#else
  MaterialColour = VtoF_Colour;
#endif    

//================================================================================

#ifdef TEXTURING_1_SRCA_1MINUSSRCA	
	// Get the material's physical colour from diffuse and texture
	// Blend according to secondary texture's alpha...
   	mediump vec4 Material2Colour = (VtoF_Colour * texture2D(SAMPLER_Texture1,VtoF_SecondaryTextureCoord));
	MaterialColour.xyz = mix(MaterialColour.xyz, Material2Colour.xyz, Material2Colour.w);
#endif    

#ifdef TEXTURING_1_1_SRCA	
	// Get the material's physical colour from diffuse and texture
	// Blend according to secondary texture's alpha...
   	mediump vec4 Material2Colour = (VtoF_Colour * texture2D(SAMPLER_Texture1,VtoF_SecondaryTextureCoord));
	MaterialColour = MaterialColour + (Material2Colour * Material2Colour.w);
#endif    

#ifdef TEXTURING_1_CROSSFADE	
	// Get the material's physical colour from diffuse and texture
	// Blend according to secondary texture's alpha...
   	mediump vec4 Material2Colour = (VtoF_Colour * texture2D(SAMPLER_Texture1,VtoF_SecondaryTextureCoord));
	MaterialColour = MaterialColour + (Material2Colour * INPUT_Texture1CrossFadeFactor);
#endif    

//================================================================================

#ifdef CUBE_MAPPING
	highp vec3 cube_color = textureCube(SAMPLER_CubeMap, VtoF_CubeTextureCoord.xyz).rgb;

	// Write the final pixel.
	MaterialColour.xyz = mix(MaterialColour.xyz, cube_color, INPUT_Reflectivity);
#endif    

//================================================================================
    
#ifdef DISCARD_ON_ALPHA    
    // This is really slow on most mobile devices...
    if (MaterialColour.a<0.02)		// Arbitrary cut off....  
    {
    	// Discard pixel if it is considered transparent enough to be not there at all;
    	// Importantly, we don't want to write to the depth buffer...
    	// This is important for textures with characters on to appear correctly invisible
    	// around the edges when rendered against other textures...
    	discard;					
    }
#endif    

//================================================================================

#ifdef LIGHTING_ENABLED

  #ifdef DIRECTIONAL_LIGHT_0
	// compute the dot product between normal and light direction... always >= 0 
	NdotL = max(dot(n,VtoF_DirectionalLight0_Dir),0.0);

	DirectionalLight0Effect = MaterialColour * NdotL * INPUT_DirectionalLight0Colour;
   
    #ifdef SPECULAR   
      // TODO: Why is this IF required?  Removal prevokes a GPU abort on the pow function...
	  if (NdotL > 0.0) 
	  {
        #ifdef NORMAL_MAPPING   		
		  mediump vec3 h = normalize(VtoF_DirectionalLight0_Dir + normalize(VtoF_ViewDirection));	
  		
		  NdotHV = max(dot(n,h),0.0);
        #else			
		  NdotHV = max(dot(n,normalize(VtoF_DirectionalLight0_HalfVector)),0.0);
        #endif // !NORMAL_MAPPING
        		
		HighTemp = INPUT_SpecularColour * INPUT_DirectionalLight0Specular * pow(NdotHV, INPUT_Shininess);
		DirectionalLight0Effect.xyz += HighTemp.xyz;        
	  }
    #endif // SPECULAR	

  #else	
	DirectionalLight0Effect = vec4(0,0,0,0);
  #endif  // else DIRECTIONAL_LIGHT_0

//================================================================================

  #ifdef POINT_LIGHT_0
	// Note we must normalise the light direction because it is interpolated.
    mediump vec3 NormalisedDirection = normalize(VtoF_PointSpotLight0_Dir); 
	NdotL = max(dot(n,NormalisedDirection),0.0);
	PointSpotLight0Effect = MaterialColour * NdotL * INPUT_PointSpotLight0Colour;
   
    #ifdef SPECULAR   
	  if (NdotL > 0.0) 
	  {
		// Compute the reflection vector between the light direction and the surface normal	
		highp vec3 R = reflect(-NormalisedDirection, n);
		//NdotHV = max(dot(n,VtoF_PointSpotLight0_HalfVector),0.0);
        
   		// Note the world is transformed into 'camera space' such that the camera position 
   		// is considered to be at (0,0,0) looking in z=-1 direction
   		NdotHV = max(dot(R, vec3(0,0,-1)), 0.0);
        	
		HighTemp = INPUT_SpecularColour * INPUT_PointSpotLight0Specular * pow(NdotHV,INPUT_Shininess);
		PointSpotLight0Effect += HighTemp;   
	  }     
    #endif // SPECULAR
		
  #else	
	PointSpotLight0Effect = vec4(0,0,0,0);
  #endif  // Else POINT_LIGHT_0

//================================================================================

  #ifdef POINT_LIGHT_1
	NdotL = max(dot(n,VtoF_PointSpotLight1_Dir),0.0);
	PointSpotLight1Effect = MaterialColour * NdotL * INPUT_PointSpotLight1Colour;
   dark
    #ifdef SPECULAR   
	  if (NdotL > 0.0) 
	  {
		highp vec3 R = reflect(-VtoF_PointSpotLight1_Dir, n);
        
   		// Note camera position is transformed to be at (0,0,0) looking in z=-1 direction
   		NdotHV = max(dot(R, vec3(0,0,-1)), 0.0);
        	
		HighTemp = INPUT_SpecularColour * INPUT_PointSpotLight1Specular * pow(NdotHV,INPUT_Shininess);
		PointSpotLight1Effect += HighTemp;
	  }        
    #endif // SPECULAR
		
  #else	
	PointSpotLight1Effect = vec4(0,0,0,0);
  #endif

#endif // LIGHTING_ENABLED

//================================================================================

#ifdef LIGHTING_ENABLED
	// The compiler will optimise out any add 0 operations...   
    AccumulatedColour.xyz = DirectionalLight0Effect.xyz + PointSpotLight0Effect.xyz;
#else 	
	// Fallback to use Materialcolour if no lighting is enabled...
	AccumulatedColour = MaterialColour;
#endif // else LIGHTING_ENABLED

#ifndef NO_OBJECT_COLOUR
//	AccumulatedColour.xyz += (VtoF_Ambient.xyz * MaterialColour.a);
	AccumulatedColour.xyz += VtoF_Ambient.xyz;
#endif
   	AccumulatedColour.a = MaterialColour.a;		// Preserve alpha channel - derived from diffuse and texture

//AccumulatedColour.xyz += (1.0/64.0);

//================================================================================

	gl_FragColor = AccumulatedColour;
	
//================================================================================
}
