class RequestVersionManager {
    constructor() {
        this.versionMap = new Map(); // key: apiName or resourceId, value: version
    }

    nextVersion(key) {
        const current = (this.versionMap.get(key) || 0) + 1;
        this.versionMap.set(key, current);
        return current;
    }

    isLatest(key, version) {
        return this.versionMap.get(key) === version;
    }
}

export class Util {
    static async create(gl) {
        const inst = new Util
        inst.gl = gl
        gl[inst.constructor.name.toLowerCase()] = inst
        const { api } = gl
        api.registerReceiver({ name: "util", client: inst })
        return inst
    }
    constructor() {
        this.rateLimitMap = new Map();
        this.resultCache = new Map();
        this.reqVersionMgr = new RequestVersionManager();
    }
    async onCall({ cmd, param }) {
        return this[cmd] ? await this[cmd](param) : null
    }
    isRateLimited({ fid, rule }) {
        const now = Date.now();
        if (!rule) return false
        const times = this.rateLimitMap.get(fid) || [];
        const recent = times.filter(t => now - t <= rule.interval);
        recent.push(now);
        this.rateLimitMap.set(fid, recent);

        return recent.length > rule.limit;
    }
    async mxFetch({ fid, timeout = 5000, url, method = "GET", headers = {}, body = {}, rateLimit = true, ...rest }) {
        let rule = null;
        let cacheKey = null;
        if (typeof body === "string" || typeof headers === "string") return { code: -1, msg: "请求参数错误", data: null }

        if (fid === 'weather') {
            cacheKey = `${url}-${JSON.stringify(body)}`;
            rule = { limit: 1, interval: 30000, action: "recent" };
        }
        return this._mxFetch({ fid, cacheKey, timeout, rule, url, method, headers, body, rateLimit, ...rest })
    }
    async _mxFetch({ fid, cacheKey, timeout = 5000, rule, url, method = "GET", headers = {}, body, rateLimit, ...rest }) {
        const controller = new AbortController();
        timeout = +timeout || 5000;
        const timer = timeout ? setTimeout(() => controller.abort("timeout"), timeout) : null;
        if (!cacheKey) cacheKey = `${method}-${url}`;

        try {
            console.log(`🔍[${fid}] 发起请求: ${method} ${url}`);
            if (rateLimit && rule && this.isRateLimited({ fid, rule })) {
                if (rule.action === 'recent') {
                    const result = this.resultCache.get(cacheKey);
                    if (result) {
                        console.log('use recent cache');
                        result._cached = true;
                        return result;
                    }
                }

                if (rule.action === 'error')
                    return {
                        code: -429,
                        data: null,
                        msg: `rate limit exceeded, please try again later`,
                    };
            }
            body = JSON.stringify(body);

            const res = await fetch(url, {
                method,
                headers,
                body,
                signal: controller.signal,
                ...rest
            });

            timer && clearTimeout(timer);

            if (!res.ok) {
                return { code: res.status, data: null, msg: `HTTP 错误 ${res.status}` };
            }

            const contentType = res.headers.get("content-type") || "";
            let response;
            if (contentType.includes("application/json")) {
                response = await res.json();
                if (response.code === undefined) {
                    response.code = 0
                }
            } else {
                const data = contentType.includes("text")
                    ? await res.text()
                    : await res.blob();
                response = { code: 0, data, msg: "OK" };
            }
            // 存入缓存
            this.resultCache.set(cacheKey, response);
            return response
        } catch (err) {
            timer && clearTimeout(timer);
            const msg = typeof err === "string" ? err : err.message;
            console.error(`❌[${fid}] 请求失败: ${msg}`);
            return { code: -1, data: null, msg };
        }
    }
    async mxFetchAll({ options = [] }) {
        if (!Array.isArray(options) || options.length === 0) {
            return { code: -1, msg: "请求列表为空", data: [] };
        }
        const tasks = options.map(opt => this.mxFetch(opt));
        const results = await Promise.allSettled(tasks);

        // 整理结果
        const data = results.map((res, index) => {
            if (res.status === "fulfilled") {
                return res.value;
            } else {
                return {
                    code: -1,
                    msg: res.reason?.msg || "request failed",
                    data: null,
                    fid: options[index].fid || `req-${index}`
                };
            }
        });

        return {
            code: 0,
            msg: "OK",
            data
        };
    }
}