local Types = require "venuscore.rtti.types"
local libvenuscore = require "libvenuscore"
local AE = require "apolloengine"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local BlueDynamicNode = require "bluecore.base.blue_dynamic_node"
local SetVariableRef = BlueDynamicNode:extend();

SetVariableRef:MemberRegister("refComp", Types.ReferenceType(AE.BlueprintComponent:RTTI()));
SetVariableRef:MemberRegister("isLocal");
SetVariableRef:MemberRegister("varName");

function SetVariableRef:new(graph)

  SetVariableRef.super.new(self, graph)

  -- 数据初始化
  self.varName = ""
  self.refComp = nil
  self.isLocal = nil -- if true, self.refComp is null
  self.refBluegraph = nil
end

function SetVariableRef:IsAllowDuplicate()
  return self.refBluegraph ~= nil and self.refBluegraph:GetVariableInfo(self.varName) ~= nil
end

function SetVariableRef:_OnDuplicate()

  local luaPath = self[BD.LUA_CLASS_PATH];
  local clazz = require(luaPath);
  local bluePrintNode = clazz(self.graph);
  bluePrintNode:EditorCreate(self.refComp, self.varName, self.isLocal);
  bluePrintNode[BD.LUA_CLASS_PATH] = luaPath;
  return bluePrintNode;

end


function SetVariableRef:_OnDeserializePost()

  SetVariableRef.super._OnDeserializePost(self)

  if not self.isLocal and self.refComp == nil then error ("SetVariableRef:_OnDeserializePost refComp is nil") end

  self:_Setup()

end


function SetVariableRef:_OnUpdate(args)
  self.refBluegraph[self.varName] = args[1]
  return self.refBluegraph[self.varName]
end

function SetVariableRef:_OnCompile()

  self.compileResult = nil

  if self.varName == nil then
    self.compileResult = "serialization fail";
    return false
  end

  if not self.isLocal and libvenuscore.isNil(self.refComp)  then
    self.compileResult = "component is lost";
    return false
  end

  if not self:_Setup() then
    return false
  end

  local varInfo = self.refBluegraph:GetVariableInfo(self.varName)
  if not varInfo then
    self.compileResult = string.format("variable %s is lost", tostring(self.varName) );
    ERROR("SetVariableRef "..self.compileResult)
    return false
  end

  local rtti = varInfo:GetRtti();
  local pinRtti = self[BD.OUTPUTS_INFO][1]:GetRtti();
  if not rtti:isType(pinRtti) then -- 如果编译时候不一致 需要打开蓝图同步
    self.compileResult = string.format("set variable %s bluenode inconsistent with variable, open and check it", tostring(self.varName));
    ERROR("GetVariableRef "..self.compileResult)
    return false ;
  end

  pinRtti = self[BD.OUTPUTS_INFO][1]:GetRtti();
  if not rtti:isType(pinRtti) then
    self.compileResult = string.format("set variable %s bluenode inconsistent with variable, open and check it", tostring(self.varName));
    ERROR("GetVariableRef "..self.compileResult)
    return false ;
  end

  return true ;
end

function SetVariableRef:SyncEd()

  if not self.isLocal and libvenuscore.isNil(self.refComp)  then
    self.compileResult = "component is lost";
    return
  end

  if not self:_Setup() then
    return false
  end

  local varInfo = self.refBluegraph:GetVariableInfo(self.varName)
  if not varInfo then
    self.compileResult = "varInfo "..tostring(self.varName).." is lost";
    return
  end

  local hostNodeName
  if self.isLocal then
    --local blueFunction =  self.graph ;
    hostNodeName = "function"
  else
    hostNodeName = self.refComp:GetHostNode():GetName()
  end

  local rtti = varInfo:GetRtti();
  local displayName = varInfo:GetDisplayName();
  assert(rtti, "varInfo rtti nil")

  self[BD.OUTPUTS_INFO][1]:ChangeNameAndHint(
          "result",
          "Reference of "..displayName.." in ".. hostNodeName)

  self[BD.INPUTS_INFO][1]:ChangeNameAndHint(
          displayName.." in ".. hostNodeName,
          "Reference of "..displayName.." in ".. hostNodeName)

  local originRtti = self[BD.INPUTS_INFO][1]:GetRtti();

  if not originRtti:isType(rtti) then --rtti变化才同步值
    local defaultValue
    if rtti:isArray() then
      defaultValue = {}
    else
      defaultValue = BU:GenerateLiteral(rtti, varInfo:GetDefault())
    end
    self[BD.OUTPUTS_INFO][1]:ChangeRtti(rtti, nil)
    self[BD.INPUTS_INFO][1]:ChangeRtti(rtti, defaultValue)
    self.inputs[1].literal = BU:GenerateLiteral(rtti, defaultValue)
  end

end

-- Setup the real reference blueGraph
function SetVariableRef:_Setup()

  if self.isLocal then
    self.refBluegraph = self.graph;
    -- 如果是局部变量,那么get节点和变量一定在同一个蓝图中的
  else
    local blueprint
    blueprint = BU.GetIns(self.refComp)
    if blueprint == nil then
      self.compileResult = "blueprint is deleted"
      ERROR("SetVariableRef "..self.compileResult);
      return false
    end
    self.refBluegraph =  blueprint;
  end

  if not self.refBluegraph then
    self.compileResult = "blueprint/bluefunction is deleted"
    ERROR("SetVariableRef "..self.compileResult);
    return false
  end
  return true
end

function SetVariableRef:GetVariableName()
  return self.varName -- BlueVariableInfo.name is Id name
end

function SetVariableRef:GetRefBluegraph()
  return self.refBluegraph
end

function SetVariableRef:IsVarArrayEd()
  local varInfo = self.refBluegraph:GetVariableInfo(self.varName)
  return varInfo:IsArray();
end

-- -- 编辑器运行时
if _KRATOSEDITOR then

  function SetVariableRef:_OnCreateWithEditor(refBlueprintComp, varName, isLocal)

    SetVariableRef.super._OnCreateWithEditor(self);

    -- 根据构建参数,初始化成员
    assert(isLocal or refBlueprintComp, "[SetVariableRef] refComp is nil ")
    assert(varName, "[SetVariableRef] varIdx is nil ")

    self.refComp = refBlueprintComp
    self.varName = varName
    self.isLocal = isLocal


    if self:_Setup() then

      local varInfo = self.refBluegraph:GetVariableInfo(varName);

      local rtti = varInfo:GetRtti()
      local displayName = varInfo:GetDisplayName();
      local defaultValue = BU:GenerateLiteral(rtti, varInfo:GetDefault())
      -- bluePinInfo的默认值 保持跟 字面值 一样 ，避免变量的初始化值是Node 会序列化

      local hostNodeName
      if self.isLocal then
        --local blueFunction =  self.graph ;
        hostNodeName = "function"
      else
        hostNodeName = self.refComp:GetHostNode():GetName()
      end
      self[BD.OUTPUTS_INFO][1]:ChangeRtti(rtti, nil)
      self[BD.OUTPUTS_INFO][1]:ChangeNameAndHint(
              "result",
              "Reference of "..displayName.." in ".. hostNodeName)

      self[BD.INPUTS_INFO][1]:ChangeRtti(rtti, defaultValue)
      self[BD.INPUTS_INFO][1]:ChangeNameAndHint(
              displayName.." in ".. hostNodeName,
              "Reference of "..displayName.." in ".. hostNodeName)

      self.inputs[1].literal = defaultValue

    else
      return  ;
    end
  end

end


SetVariableRef:SetFunctionType(BD.VARIABLE_NODE);
SetVariableRef:SetFunctionName("ref_variable")
SetVariableRef:RegisterInput (1, Types.AnyType, "n/a in n/a", "Reference of n/a", Types.AnyType(), false, nil);
SetVariableRef:RegisterOutput(1, Types.AnyType, 'result' , "result")
SetVariableRef:RegisterExecInput(1, "exec", "执行");
SetVariableRef:RegisterExecOutput(1, "exec", "执行");

return SetVariableRef