local venuscore = require "venuscore"
local mathfunction = require "mathfunction"
local apolloengine = require "apolloengine"
local apollonode = require "apolloutility.apollonode"
local renderqueue = require "apolloutility.renderqueue"
local defined = require "apolloutility.defiend"
local utils = require "posteffect_filter.utils"

local POSTEFFECT_PREFIX = "comm:script/apolloengine/posteffect/"

local statictexmain={};

-- the value should be consistent with posteffect/blendlayer.lua
local BlendMode = {
    Normal = 1,
    Add = 2,
    Lighten = 3 ,
    Multiply = 4 ,
    Overlay = 5,
    Screen = 6 ,
    Lighter = 7 ,
}

function statictexmain:Initialize(cameraLayer, config)
  local maincamera = renderqueue:GetCamera(cameraLayer);
  if maincamera == nil or
     config == nil or 
     config.posteffect == nil then
    ERROR("can't get maincamera or can't load config");
    return;
  end
  
  self.maincamera = maincamera;
                   
  local posteffectpathlist = {}
  local posteffectparameterlist = {}
  
  -- read script path and parameters from config file
  for i = 1, #config.posteffect, 1 do
    local posteffect = config.posteffect[i]
    for k, v in pairs(posteffect) do
      table.insert(posteffectpathlist, POSTEFFECT_PREFIX..k..".lua")
      table.insert(posteffectparameterlist, v)
    end
  end
  
  self.posteffectlist = self:_CreatePostEffectFromCamera(self.maincamera, posteffectpathlist, posteffectparameterlist)  
end

function statictexmain:Update(def)
  --self:SetShow(true)
end

function statictexmain:Exit()
  self:_Clear();
end

-- create post effect from parameter list
function statictexmain:_CreatePostEffectFromCamera(camera, posteffectpathlist,posteffectparameterlist)  
  if camera == nil or 
     posteffectpathlist == nil or 
     posteffectparameterlist == nil then
    return nil;
  end
  
  local posteffectlist = {};
  local posteffectnum = #posteffectpathlist;
  for i=1, posteffectnum, 1 do
    local posteffect = camera:CreatePostEffect();
    posteffect:CreateResource(posteffectpathlist[i])
    local posteffectparameter = posteffectparameterlist[i];
    for k, v in pairs(posteffectparameter) do 
      if "number" == type(v) then
        posteffect:SetScriptParameter(k, mathfunction.vector1(v))
      elseif "table" == type(v) then
        if #v == 2 then
          posteffect:SetScriptParameter(k, mathfunction.vector2(unpack(v)))
        elseif #v == 3 then
          posteffect:SetScriptParameter(k, mathfunction.vector3(unpack(v)))
        elseif #v == 4 then
          posteffect:SetScriptParameter(k, mathfunction.vector4(unpack(v)))
        end
      elseif utils:IsImage(v) then
        local tex = utils:LoadTexture(v)
        posteffect:SetScriptParameter(k, tex)
      elseif k == "BlendMode" then 
        if BlendMode[v] == nil then
          ERROR("Can not find blendMode "..v)
        else
          posteffect:SetScriptParameter(k, mathfunction.vector1(BlendMode[v]))   
        end
      end
    end
    posteffect:Disable();
    posteffect:SetQueue(i);
    table.insert(posteffectlist,posteffect);
  end

  return posteffectlist;
end

-- clear resource
function statictexmain:_Clear()
  if self.maincamera == nil then
    return
  end
  
  for i = 1, #self.posteffectlist, 1 do
    local posteffect = self.posteffectlist[i]
    self.maincamera:DeletePostEffect(posteffect)
    posteffect = nil
  end
  self.posteffectlist = nil;
end

-- show/hide the filter
function statictexmain:SetShow(show)
  if self.posteffectlist then
    for i = 1, #self.posteffectlist, 1 do
      local posteffect = self.posteffectlist[i]
      if show then
        posteffect:Enable()
      else
        posteffect:Disable()
      end
    end
  end
end

return statictexmain;