local Object        = require "classic"
local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local mathfunction  = require "mathfunction"
local defined       = require "facemorph.defined"
local RenderPassBase = require "facemorph.renderpass"

local MorphPass = RenderPassBase:extend();

function MorphPass:new()  
  MorphPass.super.new(self);

  self.texSlots = {};
  
  self:_GenerateRenderContexts();
end

function MorphPass:Clear()  
  MorphPass.super.Clear(self);
   
  self.texSlots = {};
end

-- Init buffer, generates morph vertices and textures
-- The vertices data will be normalized to [-1, 1]
-- The landmarks data will be normalized to [0, 1]
function MorphPass:ResetBuffer(index, tex, size, landmarks)
  self.points[index] = self:_GenerateVertices(size, landmarks);
  self.texCoords[index] = self:_GenerateTextureCoords(size, self.points[index]);   
  self:_NormalizeVertices( self.points[index], size);
  self:_UpdateTexture(index, self.texCoords[index], tex);
  self.texs[index] = tex;
end

function MorphPass:Update(alpha)
  self:UpdateBlendfactor(alpha);
  self:_UpdateVertices(self.points, alpha);
end

function MorphPass:SetFboSize(size)
   MorphPass.super.SetFboSize(self, size);
end

-- Generate Render Contexts
-- 1. Init vertex/indices stream
-- 2. Load material
-- 3. Create RenerNode
function MorphPass:_GenerateRenderContexts()    
  self.vtxStream, self.idxStream     =  self:_InitStreams();  
  
  self.texSlots[defined.SOURCE]     =  apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE0");                          
  self.texSlots[defined.TARGET]     =  apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE1");                                
  self.alphaSlot      =   apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ALPHA");
  
  self.renderNode     =  apollonode.RenderNode(); 
  local materialPath  = "docs:facemorph/material/facemorph.material";
  local renderMode    =  apolloengine.RenderComponent.RM_TRIANGLES ;
  self.renderNode:CreateResource(materialPath, renderMode, self.vtxStream, self.idxStream);    
  self.renderNode:SetSequence(defined.SEQUENCE);
  self.renderNode:SetShow(false);
end


-- Get texture slot and attribute based on index
function MorphPass:_GetTextureSlotAttr(index)
  if index < defined.SOURCE or index > defined.TARGET
  then
    return nil;
  end
  
  local slot = self.texSlots[index];  
  
  local attr = nil;
  if (index == defined.SOURCE)
  then
    attr = apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0;
  elseif(index == defined.TARGET)
  then
    attr = apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1;
  end
  return slot, attr;
end

-- Initialize the vertex stream and indicies stream
function MorphPass:_InitStreams()  
  local vtxstream     = apolloengine.VertexStream();
  
  -- set vertex format
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2);
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2);
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2);     

  for i = 1, defined.NUM_DELAUNAY_POINTS do 
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_POSITION, mathfunction.vector2(0, 0));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, mathfunction.vector2(0, 0));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1, mathfunction.vector2(0, 0));
  end
  
  local idxStream = apolloengine.IndicesStream();
  local indicesNum = #defined.DELAUNAY_INDICE_WITH_MOUTH;
  
  idxStream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
  idxStream:ReserveBuffer(indicesNum);

  for i = 1, indicesNum do
    idxStream:PushIndicesData(defined.DELAUNAY_INDICE_WITH_MOUTH[i]);
  end

  return vtxstream, idxStream;
end


-- normalize vertices to [-1, 1]
-- @ vertices:        input/output vertices
-- @ size:            input texturestream size
function MorphPass:_NormalizeVertices(vertices, size)
  if (vertices == nil)
  then
    return;
  end

  for i = 1, defined.NUM_DELAUNAY_POINTS do    
    local pnt = vertices:Get(i);
    --convert to position coordinate(-1, 1)
    local x = pnt:x() / size:x() * 2 - 1;
    local y = (1- pnt:y() / size:y()) * 2 - 1;
    local point = mathfunction.vector2(x, -y);
    vertices:Set(i, point);
  end
end

-- Update Blendfactor
function MorphPass:UpdateBlendfactor(alpha)
  self.renderNode:SetParameter(self.alphaSlot, mathfunction.vector1(alpha));
end

-- update morph vertices based on alpha 
function MorphPass:_UpdateVertices(pointsTbl, alpha)
  if (pointsTbl == nil or #pointsTbl ~= defined.TARGET) then
    return;
  end  
  
  local morphvertices = mathfunction.vector2array();
  if (alpha == 0)
  then
    morphvertices = pointsTbl[defined.SOURCE];
  elseif(alpha == 1)
  then
    morphvertices = pointsTbl[defined.TARGET];
  else
    local points = {};
    for i = 1, defined.NUM_DELAUNAY_POINTS do
      for j = 1, #pointsTbl do
        local pnt = pointsTbl[j]:Get(i);
        table.insert(points, j, pnt);
      end
      morphvertices:PushBack(points[defined.SOURCE] * (1 - alpha) + points[defined.TARGET] * alpha);
    end
  end
  self:_UpdateVertexStream(apolloengine.ShaderEntity.ATTRIBUTE_POSITION, morphvertices);  
end

-- Update texture 
-- @ index: which needs to be updated, valid input is SOURCE or TARGET
-- @ texcoords: texcoords data to be updated
-- @ tex:       texture to be updated
function MorphPass:_UpdateTexture(index, texcoords, tex)  
  if (texcoords == nil or tex == nil)
  then
    return;
  end
  
  local texSlot, texAttr = self:_GetTextureSlotAttr(index);
  self:_UpdateVertexStream(texAttr, texcoords); 
  --self.texs[index]:SubstituteTextureBuffer(ts);
  self.renderNode:SetParameter(texSlot, tex); 
end

-- Update vertex stream
-- @ attribute: indicates which needs to be updated
-- @ newpoints: new data
function MorphPass:_UpdateVertexStream(attribute, newpoints)
  local newpoint =  mathfunction.vector2(0, 0)
  local offset = self.vtxStream:GetAttributeIndex(attribute);
  for i = 1, newpoints:Size() do
    local point = newpoints:Get(i);
    newpoint:Set(point:x(), point:y());
    self.vtxStream:ChangeVertexDataWithAttributeFast(
                                  offset,
                                  i,
                                  newpoint);
  end
   
  self.vtxStream:SetReflushInterval(1, newpoints:Size());
  self.renderNode.render:ChangeVertexBuffer(self.vtxStream);
end

-- Generate initiate landmarks/border points
-- @ ts:              input TextureStream size
-- @ lmk:             input mathfunction.vector2array: face+foreheads;
function MorphPass:_GenerateVertices(size, lmk)
  if lmk == nil then
    return nil;
  end
  
  if #lmk ~= (defined.NUM_FACE_LAMDMARK + defined.NUM_FOREHEAD_LANDMARK) * 2 
  then
    return nil;
  end
  
  local points = mathfunction.vector2array();
  for i = 1, #lmk, 2 do
    points:PushBack(mathfunction.vector2(lmk[i], lmk[i+1]));
  end

  local col, row = 0, 0;
  local left, bot, width, height = 0, 0, size:x(), size:y();
  
  local NUM_DIVISION = 10;
  
  for cx = 1, NUM_DIVISION + 1 do
    col = math.min(left + width * (cx - 1) / NUM_DIVISION, left + width);
    for cy = 1, NUM_DIVISION + 1 do
      row =  math.min(bot + height * (cy - 1) / NUM_DIVISION, bot + height);
      if not (
        col ~= left and 
        col ~= (left + width) and
        row ~= bot and 
        row ~= (bot + height) ) then
        points:PushBack(mathfunction.vector2(col, row));
      end 
    end
  end
  
  return points;
end

-- Generate Texture Coordinates
-- @ ts     :         input TextureStream size
-- @ points  :        input mathfunction.vector2array
function MorphPass:_GenerateTextureCoords(size, points)
  if (points == nil ) then
    return nil;
  end
  
  local texCoords = mathfunction.vector2array();

  for j = 1, defined.NUM_DELAUNAY_POINTS do
    local point = points:Get(j); 
    -- normalize to [0, 1]
    texCoords:PushBack(point/size);     
  end
  return texCoords;
end

function MorphPass:ReleaseResource()
  MorphPass.super.ReleaseResource(self);
end



return MorphPass;
