local Object       = require "classic"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local apollonode   = require "apolloutility.apollonode"
local venuscore    = require "venuscore"
local defined      = require "apolloutility.defiend"
local PostEffect = require "apolloutility.apollonode.posteffect"

local ContourFeatherRender = Object:extend();

function ContourFeatherRender:new(scene, render_sequence, mask_layer, material_path, width, height)
    self.scene        = scene; 
    self.node         = self.scene:CreateNode(apolloengine.Node.CT_NODE);
    local rootnode = self.node:GetRoot();
    rootnode:DetachNode(self.node);
    self.render       = self.node:CreateComponent(apolloengine.Node.CT_RENDER);
    self.isshow = true;
    
    self.camera       = self.node:CreateComponent(apolloengine.Node.CT_CAMERA);
    self.frame_width  = width;
    self.frame_height = height;
    self.initialized  = false;

    local clear_color   = mathfunction.Color(0.0, 0.0, 0.0, 0.0);
    local fbo_size      = mathfunction.vector2(self.frame_width, self.frame_height);
    self.render_target  = apolloengine.RenderTargetEntity();
    self.render_target:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
    apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
    mathfunction.vector4(0, 0, self.frame_width, self.frame_height),
    fbo_size));

    self.image_gpu = self.render_target:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
    self.image_gpu:PushMetadata(apolloengine.TextureRenderMetadata(
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        fbo_size));

    local depth = self.render_target:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_DEPTH_STENCIL);
    depth:PushMetadata(
      apolloengine.DepthRenderBufferMetadata(
        apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
        fbo_size,
        apolloengine.TextureEntity.PF_DEPTH24_STENCIL8
      ));
    
    self.render_target:CreateResource();

    self.camera:SetSequenceCulling(true);
    self.camera:SetLayerMaskNothing();
    self.camera:AddLayerMask(mask_layer);
    self.camera:Activate();
    self.camera:SetSequence(render_sequence);
    self.camera:SetClearColor(clear_color);  
    self.camera:FixedResolution(fbo_size);
    self.camera:AttachRenderTarget(self.render_target);

    local material = venuscore.IFileSystem:PathAssembly(material_path);
    self.node:SetLayer(mask_layer);
    self.render:PushMetadata(
    apolloengine.RenderObjectMaterialMetadata(
    apolloengine.PathMetadata(material)));

    self.render:PushMetadata(
		apolloengine.RenderObjectMeshMetadate( 
			apolloengine.RenderComponent.RM_TRIANGLES,
      apolloengine.QuadVertexMetadata(true, false),
      apolloengine.QuadIndicesMetadata()));
    self.render:CreateResource();
    self.render:SetSequence(render_sequence);


    self.post = self.camera:CreatePostEffect();
    self.post:PushMetadata(
        apolloengine.PathMetadata(venuscore.IFileSystem:PathAssembly("comm:script/apolloengine/posteffect/contour_extract.lua")));
    self.post:CreateResource();  

    self.initialized   = true;
end

function ContourFeatherRender:SetShow(is_show)
    if self.isshow ~= is_show then
        self.isshow = is_show;
        if is_show then
          self.render:SetRenderProperty(apolloengine.RenderComponent.RP_SHOW);
        else
          self.render:EraseRenderProperty(apolloengine.RenderComponent.RP_SHOW);
        end  
    end
end

function ContourFeatherRender:PreUpdate(sequence, frame_width, frame_height)
    if not self.initialized then
        LOG("[ContourFeatherRender] preUpdate fail");
        return;
    end
    self.camera:SetSequence(sequence);
    self.render:SetSequence(sequence);
    self:SetShow(false);

    local fbo_size = mathfunction.vector2(frame_width, frame_height);
    self.camera:FixedResolution(fbo_size);
end


function ContourFeatherRender:SetParameter(key, value)
    if key == nil or value == nil then
        LOG("[ContourFeatherRender] SetParameter nil "..key);
        return;
    end

    if self.initialized then
        self.render:SetParameter(key, value);
    end
end

function ContourFeatherRender:SetPostParameter (key, value)
    if key == nil or value == nil then
        LOG("[ContourFeatherRender] SetParameter nil "..key);
        return;
    end

    if self.initialized then
        self.post:SetParameter(key, value);
    end
end

function ContourFeatherRender:GetResultImage()
    return self.image_gpu;
end

function ContourFeatherRender:Clear()
    if self.node ~= nil then
        local scene = self.scene;

        self.post:Disable();
        self.camera:DeletePostEffect(self.post);
        self.post = nil;

        scene:DeleteNode(self.node);

        self.camera = nil;
        self.render = nil;
        self.render_target = nil;
        self.image_gpu = nil;
        self.node = nil;
        self.scene = nil;
    end
end

return ContourFeatherRender;
