local PoseStablizer = {}

function PoseStablizer:Initialize(num_joints, noise_margin)
  --[[
      args: num_joints:   integer, denotes number of joints in total
            noise_margin: number or tabel, if tabel, the length must
                          equal to num_joints, denotes noise margin
                          in milimeter.
      e.g. PoseStablizer:Initialize(15, 0.15)
  ]]
  assert(type(num_joints) == "number")
  assert(num_joints > 0)
  self.num_joints = num_joints
  
  if type(noise_margin) == "tabel" then
    assert(#noise_margin == self.num_joints)
  end
  
    
  self.noise_margin = {}
  local m = {}
  for i = 1, self.num_joints do
    if type(noise_margin) == "number" then
      assert(noise_margin >= 0)
      self.noise_margin[i] = noise_margin
    else
      assert(noise_margin[i] >= 0)
      self.noise_margin[i] = noise_margin[i]
    end
  end
  
  self.cached_pose = {}
  for i = 1, self.num_joints do
    self.cached_pose[i] = {}
  end
  
end

function PoseStablizer:Update(pose3d)
  for i = 1, self.num_joints do
    --do not update for the 1st frame
    if #self.cached_pose[i] == 0 then
        self.cached_pose[i] = pose3d[i]
    -- 
    else
      if self:is_update(pose3d[i], self.cached_pose[i], self.noise_margin[i]) then
        for j = 1, 3 do
          self.cached_pose[i][j] = 0.35*pose3d[i][j] + 0.65*self.cached_pose[i][j]
          --self.cached_pose[i][j] = pose3d[i][j]
        end
      else
        for j = 1, 3 do
          self.cached_pose[i][j] = 0.1*pose3d[i][j] + 0.9*self.cached_pose[i][j]
          --self.cached_pose[i][j] = self.cached_pose[i][j]
        end
      end
    end
  end
  
  local cached_pose_copy = self:deep_copy(self.cached_pose)
  return cached_pose_copy
end

-- utils
function PoseStablizer:is_update(p_new, p_cache, margin)
  local is_update = false
  if #p_cache == 0 then
    is_update = true
  elseif self:dist(p_new, p_cache) > margin then
    is_update = true
  end
  return is_update
end

function PoseStablizer:dist(p1, p2)
  local sum = 0
  for i = 1, #p1 do
    sum = sum + (p1[i] - p2[i]) ^ 2
  end
  return math.sqrt(sum)
end

function PoseStablizer:deep_copy(orig)
  local copy
  if type(orig) == "table" then
    copy = {}
    for orig_key, orig_value in next, orig, nil do
      copy[self:deep_copy(orig_key)] = self:deep_copy(orig_value)
    end
    setmetatable(copy, self:deep_copy(getmetatable(orig)))
  else
    copy = orig
  end
  return copy
end


--[[
stb = PoseStablizer
stb:Initialize(1, 30)
p1 = {{0,0,0}}
p2 = {{10,100,100}}
pp1 = stb:Update(p1)
pp2 = stb:Update(p2)

print('hello')
]]
return PoseStablizer