local FourierFiltering = {}

function FourierFiltering:Initialize(...)
  --[[
      coeff     -   1-D lua table, coefficients of the filter
                    designed off-line.
  --]]
  local args = {...}
  if #args == 1 then
    if type(args[1]) == "table" then
      self.coeff = args[1]
    end
  else
    print("error: invalid method")
  end
  --
  self.coeff = {0.00061995524,0.0020032364,0.0040231175,0.0054718540,0.0041773110,
                -0.0020940411,-0.013936437,-0.028600523,-0.039476987,-0.037685376,
                -0.015674246,0.028546862,0.088698909,0.15086330,0.19778016,
                0.21524715,0.19778016,0.15086330,0.088698909,0.028546862,
                -0.015674246,-0.037685376,-0.039476987,-0.028600523,-0.013936437,
                -0.0020940411,0.0041773110,0.0054718540,0.0040231175,0.0020032364,
                0.00061995524}
  --]]
  
  --[[
  
  self.coeff = {0.00485388605797098,	0.0142668778964594,	0.000979102332543599,	-0.0539697665172143,	-0.0766382179224417,	0.0539234344426130,	0.300858879523048,	0.431527212114720,0.300858879523048,	0.0539234344426130,	-0.0766382179224417,	-0.0539697665172143,	0.000979102332543599,	0.0142668778964594,	0.00485388605797098}
  --]]
  --[[
  self.coeff = {-0.124256856098065,	0.0908306421793293,	0.219193357526355,	0.322879553390762,	0.322879553390762,	0.219193357526355,	0.0908306421793293,	-0.124256856098065}
  --]]
  --[[
  self.coeff = {0.0138409307475915,	0.122687416758707,	0.363471652493701,0.363471652493701,	0.122687416758707,	0.0138409307475915}
  --]]
  --[[
  self.coeff = {0.0318348336714106,	0.332059531171459,	0.392045728751517,	0.332059531171459,	0.0318348336714106}
  --]]
  self.len_q = #self.coeff
  self.coeff_inv = {}
  for i = 1, self.len_q do
   self.coeff_inv[self.len_q-i+1] = self.coeff[i]
  end
  self.mean = {}
  self.cache_demean = {}
  self.time = 0
  self.start_filtering = false

end

function FourierFiltering:Update(...)
  local args = {...}
  if self.time > 0 then
    -- number of input args must not be changed midway
    assert(#args == self.last_input_length)
  else
    -- init cache
    self.cache = {}
    for i = 1, #args do
      self.cache[i] = {}
    end
  end
  self.last_input_length = #args

  --push to cache
  for i = 1, #self.cache do
    self:tbl_push(self.cache[i], args[i])
  end

  --update time and flag  
  self.time = self.time + 1
  if self.time >= self.len_q then
    self.start_filtering = true
    for i = 1, #self.cache do
      self.cache_demean[i] = {}
      self.mean[i] = self:get_mean(self.cache[i])
      for j = 1, #self.cache[i] do
        self.cache_demean[i][j] = self:add(self.cache[i][j], self:mul(self.mean[i], -1))
      end
    end
  end
end

function FourierFiltering:Filter(...)
  local args = {...}
  self:Update(unpack(args))
  if self.start_filtering then
    local result = {}
    for i = 1, #args do
      result[i] = self:add(self:sum(self:mul(self.cache_demean[i], self.coeff_inv)), self.mean[i])
    end
    return unpack(result)
  else
    return unpack(args)
  end
end


--maths
function FourierFiltering:get_mean(var)
  local result = self:sum(var)
  result = self:mul(result, 1/#var)
  return result
end

function FourierFiltering:Zeros(size)
  --size    -   lua table
  local zero_size = self:deep_copy(size)
  table.remove(zero_size, 1) 
  local zero_like = {}
  for i = 1, size[1] do
    zero_like[i] = self:Zeros(zero_size)
  end
  return zero_like
end

function FourierFiltering:sum(var)
  if type(var) == "table" then
    local result = 0
    for i = 1, #var do
      result = self:add(result, var[i])
    end
    return result
  else
    print("error: unsupported dtype!")
    return nil
  end
end

function FourierFiltering:mul(var1, var2)
  return self:element_wise(var1, var2, "mul")
end

function FourierFiltering:add(var1, var2)
  return self:element_wise(var1, var2, "add")
end

function FourierFiltering:element_wise(var1, var2, op)
  -- op     -   operation, supported: "add", "mul"  
  if var1 == nil or var2 == nil then
    return nil
  elseif type(var1) == "number" and type(var2) == "number" then
    return self:operate_callback(var1, var2, op)
  elseif type(var1) == "table" and type(var2) == "number" then
    local result = {}
    for i = 1,#var1 do
      result[i] = self:element_wise(var1[i], var2, op)
    end
    return result
  elseif type(var1) == "number" and type(var2) == "table" then
    return self:element_wise(var2, var1, op)
  elseif type(var1) == "table" and type(var2) == "table" then
    assert(#var1 == #var2, "error: size must be agree!")
    local result = {}
    for i = 1,#var1 do
      result[i] = self:element_wise(var1[i], var2[i], op)
    end
    return result
  else
    print("error: unsupported dtype!")
    return nil
  end
end

function FourierFiltering:operate_callback(num1, num2, op)
  if op == "add" then
    return num1 + num2
  elseif op == "mul" then
    return num1 * num2
  else
    print("error: unsupported operation")
    return nil
  end
end

--utils
function FourierFiltering:deep_copy(orig)
  local copy
  if type(orig) == "table" then
    copy = {}
    for orig_key, orig_value in next, orig, nil do
      copy[self:deep_copy(orig_key)] = self:deep_copy(orig_value)
    end
    setmetatable(copy, self:deep_copy(getmetatable(orig)))
  else
    copy = orig
  end
  return copy
end

function FourierFiltering:tbl_push(tbl, val)
  table.insert(tbl, self:deep_copy(val))
  if #tbl > self.len_q then
    table.remove(tbl, 1)
  end
end


--test
--[[
s1 = {{1,1}, {-1,-1}, {2,2}, {-2,-2}, {3,3}, {-3,-3}}
s2 = {{2,2}, {-1,-1}, {2,2}, {-1,-1}, {2,2}, {-1,-1}}

lpf = FourierFiltering
lpf:Initialize({1,1,1})
for i = 1, #s1 do
  f1, f2 = lpf:Filter(s1[i], s2[i])
  print(f1[1], f1[2])
  print(f2[1], f2[2])
end
]]--
return FourierFiltering