local vc = require "venuscore"
local Fn = vc.Object:extend()

local function concat(tab1, tab2)
  local res = { unpack(tab1) }
  local count = #tab2
  for i = 1, count do
    table.insert(res, tab2[i])
  end
  return res
end

local function foreach(tab, func)
  local count = #tab
  for i = 1, count do
    func(tab[i])
  end
  return tab
end

local function map(tab, func)
  local count = #tab
  local res = {}
  for i = 1, count do
    res[i] = func(tab[i])
  end
  return res
end

local function filter(tab, func)
  local count = #tab
  local res = {}
  for i = 1, count do
    local item = tab[i]
    if func(item) then
      table.insert(res, item)
    end
  end
  return res
end

local function dedup(tab)
  local set = {}
  return filter(
    tab,
    function(item)
      if set[item] == nil then
        set[item] = true
        return true
      else
        return false
      end
    end
  )
end

local function reduce(tab, seed, func)
  local count = #tab
  local res = seed
  for i = 1, count do
    res = func(res, tab[i])
  end
  return res
end

local function flatMap(tab, func)
  return reduce(
    tab,
    {},
    function (res, item)
      return concat(res, func(item))
    end
  )
end

local function tableKeys(tab)
  local res = {}
  for k, _ in pairs(tab) do
    table.insert(res, k)
  end
  return res
end

local function tableValues(tab)
  local res = {}
  for _, v in pairs(tab) do
    table.insert(res, v)
  end
  return res
end

local function first(tab)
  return tab[1]
end

local function last(tab)
  return tab[#tab]
end

local function merge(tab1, tab2)
  local res = {}
  for k, v in pairs(tab1) do
    res[k] = v
  end
  for k, v in pairs(tab2) do
    res[k] = v
  end
  return res
end

function Fn:new(tab)
  self.tab = tab
end

function Fn.Just(...)
  return Fn({ ... })
end

function Fn:ForEach(func)
  return Fn(foreach(self.tab, func))
end

function Fn:Map(func)
  return Fn(map(self.tab, func))
end

function Fn:Reduce(seed, func)
  return Fn(reduce(self.tab, seed, func))
end

function Fn:Filter(func)
  return Fn(filter(self.tab, func))
end

function Fn:Dedup()
  return Fn(dedup(self.tab))
end

function Fn:Concat(tab)
  return Fn(concat(self.tab, tab))
end

function Fn:FlatMap(func)
  return Fn(flatMap(self.tab, func))
end

function Fn:TableKeys()
  return Fn(tableKeys(self.tab))
end

function Fn:TableValues()
  return Fn(tableValues(self.tab))
end

function Fn:First()
  return Fn(first(self.tab))
end

function Fn:Last()
  return Fn(last(self.tab))
end

function Fn:Wrap()
  return Fn({ self.tab })
end

function Fn:Merge(tab)
  return Fn(merge(self.tab, tab))
end

function Fn:Get()
  return self.tab
end

return Fn