local libvenuscore = require "libvenuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local Types = require "venuscore.rtti.types"
local BlueDefined = require "bluecore.bluedefined"
local BD = BlueDefined
local ConstantNode = require "bluecore.constantnode"
local BlueSelfNode = require "bluecore.blueselfnode"
local BlueContext = require "bluecore.bluecontext"
local BluePrintVariable = require "bluecore.bluevariableinfo"
local BlueUtility = require "bluecore.blueutility"
local BS = require "bluecore.bluesingle"
local EventDispatcher = require "bluecore.eventdispatcher"
local cv = require "computervisionfunction"
local BlueRtti = require "bluecore.core.blue_rtti"

-- 这个蓝图类 对应的是  Node
local BluePrint = libvenuscore.VenusBehavior:extend("BluePrint");

BluePrint:MemberRegister("_UID");
BluePrint:MemberRegister("bluePrintNodeList");
BluePrint:MemberRegister("bluePrintLinkList");
BluePrint:MemberRegister("bluePrintVariableList");
BluePrint:MemberRegister("_clsName");
BluePrint:MemberRegister("_clsId");
BluePrint:MemberRegister("_isP"); -- is prefab
BluePrint:MemberRegister("_version")
BluePrint:MemberRegister("blueFunctionList");
BluePrint:MemberRegister("_evDispatcherList");


-- 用于区分不同的BluePrintGraph实例与窗口
local gIndex = 1
-- 用于编译打印信息
local DEBUG_E_DOG = "-->"
local DEBUG_D_DOG = "|--"

function BluePrint:new()

  local baseClass = apolloengine.Node:RTTI() -- UE can base any Node/Component Class

  BS.init();

  self._version = BD.VERSION

  -- 文件是否被修改/需要重新编译
  self._isModify = false ;
  -- 编译结果
  self.compileFlag = true ;
  self.errorMessage = nil -- 类成员函数/变量/事件节点执行流编译错误
  -- 代表蓝图是否反序列化完成(Awake完成之后标记)
  self._isAwake = false ;

  LOG("BluePrint:new call");

  -- 用于区分不同的BluePrintGraph实例
  self.displayIndex = gIndex
  gIndex = gIndex + 1
  WARNING(" self.displayIndex --- "..tostring( self.displayIndex));

  if BD.LEAK_DEBUG then
    local index = self.displayIndex
    ERROR("^^^^^^^^^ BluePrint new called:" ..tostring(index));
    local prox = newproxy(true)
    getmetatable(prox).__gc = function() ERROR("^^^^^^^^^ BluePrint CG called:" ..tostring(index)); end
    self[prox] = true
  end


  self._UID = 0;                      -- NextID 先执行+1
  self.bluePrintNodeList = {}         -- 节点  用uid索引
  self.bluePrintLinkList = {}         -- 连线  用uid索引
  self.bluePrintVariableList = {};    -- 变量列表(由于prefab原因 改成map)
  self.blueFunctionList = {}          -- 成员函数列表(Local)
  self._evDispatcherList = {}         -- 事件分发器

  self._varUID = 0 ;
  self.bluePrintAllList = {};          -- 所有 节点+引脚  查找表   Awake补全
  self.bluePrintEventGraph = {};      -- 事件图表(首个节点) 用uid索引   Awake补全
  self.onTickCallbackList = {}        -- 每帧onTick的蓝图节点列表  goifalltriger需要一个条件触发回调
  self._customEventNodeArray = {}      -- 自定义事件节点数组
  
  --延迟节点
  self.DelayNodeList = {};
  self.deltaTime = 0.0;

  --执行流(非自定义事件分发器的执行流和普通事件节点的执行流  只能画完清除)
  self.execLinkList = {};
  --自定义事件节点的执行流
  self._isExecutingCustomEvent = false;
  self.currentEventName = nil;
  self.customEventExecLinkList = {};

  -- 是否prefab
  self._isP = false ;

  self._blueContext = BlueContext(self);   -- 当前蓝图上下文信息(e.g 动态创建的节点类) trigger BlueManager to scan files

  self._selfClass = baseClass;         --  当前蓝图类的基类 e.g apolloengine.Node:RTTI() C++ class BlueClass: public Node
  self._selfNode = BlueSelfNode(self);  -- 获取当前基类C++对象的引用                      C++  self 获取当前实例

  self.isrunning = false;

  self.contentStatusTable = {};

  self.isFirstUpdate = true;

  -- 编辑器蓝图面板blueEditor/反射面板blueprintComponent用到的数据的存放地方(不会序列化,端内没有)
  self.editorContext = nil;

end

function BluePrint:GetContext()
  return self._blueContext
end

function BluePrint:SetClassID(clsId, clsName)
  assert(clsId   ~= nil, "SetClassID clsId is nil");
  assert(clsName ~= nil, "SetClassID clsName is nil");
  self._clsId = clsId
  self._clsName = clsName
end

-- 编辑器 prefab实例再成为prefab类
function BluePrint:ClsIdChgEd(oldClsId, newClsId)

  -- 需要更新ClassID 并且 所有跟类相关的 都要 更新ClassId
  --
  -- 从 classInfo 遍历 当前node下的blueprint所有引用的 clsId -> ClassInfo(这是整个场景的)
  --    FuncCallNode(Call CustomEvent/EventDispatcher/LocalFunction)
  --    Bind/Unbind/UnbindAllEventNode
  --
  -- 自定义变量 不需要修改 还是原来的BlueprintType (与unreal一致)
  -- 函数参数  不需要修改 还是原来的BlueprintType  (与unreal一致)
  --
  -- 引用blueprintComp的不用,比如
  --    FuncInstanceNode

  if oldClsId ~= self:GetClassId() then
    error("ClsIdChgEd not match! oldClsId = "..tostring(oldClsId)..", current clsId = "..tostring(self:GetClassId()));
  end

  BlueRtti.UnRefBluePrintClass(self); -- 取消计数引用

  self._clsId = newClsId
  self._clsName = self._clsName .."_1";

  local nodes = self:GetBlueNodeList();
  for _, node in pairs(nodes) do
    if node.ClsIdChgEd then
      node:ClsIdChgEd(oldClsId, newClsId)
    end
  end

  local memberFtns = self:GetBlueFtnList()
  for _, ftn in pairs(memberFtns) do
    local ftnNodes = ftn:GetBlueNodeList()
    for _, node in pairs(ftnNodes) do
      if node.ClsIdChgEd then
        node:ClsIdChgEd(oldClsId, newClsId)
      end
    end
  end

  BlueRtti.RegisterClassAddRef(self);

end

function BluePrint:ClsIdSyncEd(oldClassId, newClassId)

  local nodes = self:GetBlueNodeList();
  for _, node in pairs(nodes) do
    if node.ClsIdChgEd then
      node:ClsIdChgEd(oldClassId, newClassId)
    end
  end

  local memberFtns = self:GetBlueFtnList()
  for _, ftn in pairs(memberFtns) do
    local ftnNodes = ftn:GetBlueNodeList()
    for _, node in pairs(ftnNodes) do
      if node.ClsIdChgEd then
        node:ClsIdChgEd(oldClassId, newClassId)
      end
    end
  end

end

function BluePrint:GetClassId()
  return self._clsId ;
end

function BluePrint:GetClassName()
  return self._clsName;
end

function BluePrint:SetClassNameEd(dispClsName)

  local classInfo = BlueRtti.GetClassesByIdEd(self._clsId)

  local result = classInfo:SetClassNameEd(dispClsName);
  if result then
    self._clsName = dispClsName
  else
    ERROR("SetClassNameEd fail "..self._clsName.."->"..dispClsName);
  end
  return result
end



-- 增加变量 内部根据RTTI字符串生成默认值 默认是变量(非数组)
function BluePrint:_AddVariableEd(rttiName)

  if self._varUID == 0 then -- first, check the maximum id in use
    for uid, _ in pairs(self.bluePrintVariableList) do
      if uid > self._varUID then
        self._varUID = uid
      end
    end
  end

  local nextID = self._varUID + 1

  local prefix = self:GetTypeName() == "BlueFunction" and BD.BPFUNCVAR_PREFIX or BD.BPCLASSVAR_PREFIX;
  local idName = prefix..tostring(nextID);
  local tryTime = 100
  for _, var in pairs(self.bluePrintVariableList) do
    if var.name == idName then
      tryTime = tryTime - 1
      if tryTime > 0 then
        nextID = nextID + 1
        idName = prefix..tostring(nextID);
      else
        ERROR("[_AddVariableEd] duplicate name, try but no work " .. tostring(idName));
        return
      end
    end
  end
  self._varUID = nextID
  LOG("[CreateVariable] id Name "..tostring(idName))


  local var = BluePrintVariable(self, rttiName, false, idName, true, false, false, "No Tips") ;
  var:EditorCreate();

  self.bluePrintVariableList[self._varUID] = var; -- 由于prefab不支持数组,index只能往前加,溢出会在上循环检查

  local clazz_of_set, clazz_of_get = self._blueContext:_SetupVariableNodeClass(var);
  var:Subscribe(clazz_of_set)
  var:Subscribe(clazz_of_get)
  var:Subscribe(self._blueContext); -- 蓝图本身需要监听类型的修改 用来更新上下文菜单

  return var
end

function BluePrint:SyncDefaultValueByVarUidEd(varUid)
  local varInfo = self.bluePrintVariableList[varUid];
  varInfo:SyncPrefabDefaultToOverrideEd();
end

function BluePrint:CreateVariable()
  return self:_AddVariableEd(mathfunction.vector3:GetTypeName());
end

-- Preset Event
function BluePrint:GetPresetEventGraphList()
  return self.bluePrintEventGraph;
end

--  Create Custom Event
function BluePrint:CreateCustomEventNodeEd(x, y, eventName)
  local node =  self:AddBluePrintNodeEd("bluecore.core.custom_event_node", x, y, eventName);
  table.insert(self._customEventNodeArray, node)
  return node
end

function BluePrint:AppendCustomEventNode(node)
  -- Call by CustomEventNode:_OnDeserialize
  table.insert(self._customEventNodeArray, node)
end

function BluePrint:_DeleteCustomEventNodeEd(nodeId)

  local node = self.bluePrintNodeList[nodeId]
  for idx, eventNode in ipairs(self._customEventNodeArray) do
    if node == eventNode then
      node = nil
      table.remove(self._customEventNodeArray, idx)
      break;
    end
  end

  if node~= nil then error("custom event "..tostring(nodeId).." not found"); end

  --self:DeleteBlueNode(nodeId);
end

function BluePrint:GetCustomEventNodes()
  return self._customEventNodeArray;
end

function BluePrint:AddArgForCustomEventEd(nodeId)
  local customEventNode = self.bluePrintNodeList[nodeId];
  local argId = customEventNode:AddArgumentEd();
  return argId;
end

function BluePrint:RemoveArgForCustomEventEd(nodeId, argId)
  local customEventNode = self.bluePrintNodeList[nodeId];
  customEventNode:RemoveArgumentEd(argId)
end

-- create Event Dispatcher
function BluePrint:CreateEventDispatcherEd()

  local BlueEventDispatcher = require "bluecore.core.blue_event_dispatcher"

  local uid = self:_NextID();
  local name = BD.DISPATHER_PREFIX..tostring(uid);

  local dispatcher = BlueEventDispatcher(self);
  dispatcher:EditorCreate(name, uid);

  self._evDispatcherList[uid] = dispatcher;

  return uid; -- for delete
end


function BluePrint:DestroyEventDispatcherEd(uid)

  local dispatcher =  self._evDispatcherList[uid]
  if dispatcher == nil then
    return
  end

  dispatcher:ToBeDeletedEd() -- 通知EventDispatcher即将被删除

  self._evDispatcherList[uid] = nil

end


function BluePrint:AddArgForEventDispatcherEd(uid)
  local dispatcher = self._evDispatcherList[uid];
  local argId = dispatcher:AddArgumentEd();
  return argId;
end


function BluePrint:RemoveArgForEventDispatcherEd(uid, argId)
  local dispatcher = self._evDispatcherList[uid];
  dispatcher:RemoveArgumentEd(argId)
end


function BluePrint:GetEventDispatchers()
  return self._evDispatcherList;
end


function BluePrint:GetEventDispatcherByUid(uid)
  return self._evDispatcherList[uid]
end


-- Create Member Function, self is blueprint or bluescene
function BluePrint:CreateFunction()

  local BlueFunction = require "bluecore.bluefunction"
  local funcIdx = self:_NextID();
  local defaultName = string.format(BD.FUNC_PREFIX.."%d", funcIdx);
  local blueFunc = BlueFunction(self, defaultName, funcIdx);
  blueFunc:EditorCreate();      -- like blueprint.EditorCreate()
  blueFunc:_OnAwake(self.Node); -- like blueprint._OnAwake()
  blueFunc:SetupPresetNode();   -- after EditorCreate+_OnAwake or _OnDeserialize+_OnAwake(_OnDeserializePos), it's allowed to create blue node
  self.blueFunctionList[funcIdx] = blueFunc;

  -- Editor's New = new EditorCreate _OnAwake
  -- OnDeserialize's New = new OnDeserialize _OnAwake

  -- BlueFunction:EditorCreate Add To ClassInfo
  -- Blueprint:_OnDeserialize Add To ClassInfo

  return blueFunc;
end


--blueprint调用该函数
--bluefunction调用则重写
--[[function BluePrint:CreateFunctionInstance(funcType, ftnUid, libUid, pos_x, pos_y) --要考虑函数模板引用了其他的函数
  local newNode
  if funcType == BlueDefined.FuncType.LOCAL then
    local comp = self.Node:GetComponent(apolloengine.Node.CT_BLUEPRINT);
    newNode = self:AddBluePrintNode("bluecore.functioncore.func_inst_node", pos_x, pos_y, funcType, ftnUid, libUid, comp);
  else
    --FIXME如果是全局函数？？
    local rootNode = self.Node:GetRoot();
    local comp = rootNode:GetComponent(apolloengine.Node.CT_BLUEPRINT);
    newNode = self:AddBluePrintNode("bluecore.functioncore.func_inst_node", pos_x, pos_y, funcType, ftnUid, libUid, comp);
  end
  return newNode;
end]]--

-- 删除指定成员函数
function BluePrint:DeleteFunctionByIndex(funcType, ftnUid, libUid)
  if funcType == BlueDefined.FuncType.LOCAL then
    local blueFunc = self.blueFunctionList[ftnUid] ;
    blueFunc:ToBeDeletedEd()
    self.blueFunctionList[ftnUid] = nil;
  end
  collectgarbage();
end

--
function BluePrint:CreateCallFtnNode(pos_x, pos_y, args)
  --args = {  -- 命令会pack,不能用nil,所以改成用字典
  --  funcType = ,
  --  clsUid = ,
  --  ftnUid = ,
  --  libUid = ,
  --  eventId =
  --  dispatcherId =
  --}
  local funcType = args.funcType

  local newNode
  if funcType == BlueDefined.FuncType.LOCAL then
    newNode = self:AddBluePrintNodeEd("bluecore.functioncore.func_call_node", pos_x, pos_y, funcType, args.clsUid, args.ftnUid, nil, args.eventId);
  elseif funcType == BlueDefined.FuncType.GLOBAL then
    newNode = self:AddBluePrintNodeEd("bluecore.functioncore.func_call_node", pos_x, pos_y, funcType, nil, args.ftnUid, args.libUid, args.eventId);
  elseif funcType == BlueDefined.FuncType.EVENT then
    newNode = self:AddBluePrintNodeEd("bluecore.functioncore.func_call_node", pos_x, pos_y, funcType, args.clsUid, nil, nil, args.eventId);
  elseif funcType == BlueDefined.FuncType.DISPATCHER then
    newNode = self:AddBluePrintNodeEd("bluecore.functioncore.func_call_node", pos_x, pos_y, funcType, args.clsUid, nil, nil, nil, args.dispatcherId);
  else

  end
  if newNode == nil then error("CreateCallFtnNode nil"); end
  return newNode;
end


-- 暂时认为 一个Node下面的Component的类型和名字是同名的(UE4可以指定Component的名字,类似定义了一个Component的成员变量)
-- 只有Node/Actor的蓝图类 才可以创建Component引用, Component蓝图不支持
function BluePrint:CreateCompRefNode(component, x , y )
  local rqrpath = nil;
  if component:isTypeOrDriverType(apolloengine.Node:RTTI()) then
    rqrpath = "bluecore.reference.blue_node_ref";
  else
    rqrpath = "bluecore.reference.blue_comp_ref"; 
  end

  local node = self:AddBluePrintNode(rqrpath, x, y, component)
  return node ;
end

function BluePrint:CreateBlueprintRefNodeEd(blueComp, x, y)
  local luaPath = "bluecore.reference.blueprint_ref"
  local node = self:AddBluePrintNodeEd(luaPath, x, y, blueComp)
  return node;
end

--ScriptCom的Instance的引用
function BluePrint:CreateScriptCompInsRefNode(scriptCom, ins, x, y)
  local rqrpath = "bluecore.reference.blue_scriptcomp_ins_ref";
  local node = self:AddBluePrintNode(rqrpath, x, y, scriptCom, ins)
  return node;
end

--ScriptCom的Instance的引用
function BluePrint:CreateCameraPostEffectRefNode(cameraCom, postEffectIdx, compName, x, y)
  local rqrpath = "bluecore.reference.blue_camera_posteffect_ref";
  --local strTable = string.split(ins, '/');
  --local postEffect = self.comp.PosteffectEntities[tonumber(strTable[2])];
  --local compName = strTable[1].."\n".."node: ".. self.comp:GetHostNode():GetName();
  local node = self:AddBluePrintNode(rqrpath, x, y, cameraCom, postEffectIdx, compName)
  return node;
end


-- 删除 变量节点/连线移除 变量节点类 上下文菜单
function BluePrint:DeleteVariableByIndex(varUid)

  local varinfo = self.bluePrintVariableList[varUid]
  assert(varinfo, "delete var but varinfo not found");

  -- 删除变量节点类 从上下文移除
  local clazz_set, clazz_get = self._blueContext:_RemoveVariableNodeClass(varinfo);

  -- 删除节点实例
  for _,node in pairs(self.bluePrintNodeList) do
    local clazz = getmetatable(node);
    if clazz == clazz_set or clazz == clazz_get  then
      WARNING("DeleteVariable node id "..tostring(node.uid));
      self:DeleteBlueNode(node.uid);
    end
  end

  -- 从 变量列表中移除 变量
  self.bluePrintVariableList[varUid] = nil

  -- 打印上下文信息
  --local BlueDumper = require "bluecore.bluedumper"
  --BlueDumper.dumpContext(self._blueContext);

  collectgarbage();

end

function BluePrint:SetModified()
  -- 目前没有任何操作 只是标记已修改,比如变量的修改导致上下文的修改
end

function BluePrint:GetVariableInfo(idName)
  for _, var in pairs(self.bluePrintVariableList) do
    if var.name == idName then
      return var
    end
  end

  if self._isAwake then
    ERROR(string.format("[GetVariableInfo] variable %s not found   ", tostring(idName)));
  end
end

function BluePrint:GetVariableIndexByName(idName)
  for varUid, var in pairs(self.bluePrintVariableList) do
    if var.name == idName then
      return varUid
    end
  end
  ERROR( "[GetVariableIndexByName] variable not found "..tostring(idName))
end


function BluePrint:GetVariableInfoByIndex(varUid)
  assert(type(varUid) == 'number',"[GetVariable] index not number")
  assert(self.bluePrintVariableList[varUid], "[GetVariable] index not found "..tostring(varUid))
  return self.bluePrintVariableList[varUid];
end

function BluePrint:GetVariableIndexByInfo(varInfo)
  for varUid, var in pairs(self.bluePrintVariableList) do
    if var == varInfo then
      return varUid
    end
  end
  error("GetVariableIndexByInfo not found");
end

function BluePrint:CreateVarRefNode(pos_x, pos_y, refBlueprintComp, varName, isSet, isLocal)
  local node
  if isSet then
    node = self:AddBluePrintNodeEd("bluecore.variable.setvariable_ref", pos_x, pos_y, refBlueprintComp, varName, isLocal)
  else
    node = self:AddBluePrintNodeEd("bluecore.variable.getvariable_ref", pos_x, pos_y, refBlueprintComp, varName, isLocal)
  end
  return node ;
end

function BluePrint:GetNodeClass(template_path, cls_spec_data)
  return self._blueContext:GetNodeClass(template_path,cls_spec_data)
end

function BluePrint:AddNodeClass(template_path, cls_spec_data, cls)
  self._blueContext:AddNodeClass(template_path, cls_spec_data, cls)
end


function BluePrint:GetPinInfoById(pidId)
  local pin = self.bluePrintAllList[pidId]
  local node = self.bluePrintAllList[pin.nodeUid]
  return node.infoTable[pin.pinType][pin.argId]
end

function BluePrint:GetPinInfo(pin)
  local node = self.bluePrintAllList[pin.nodeUid]
  return node.infoTable[pin.pinType][pin.argId]
end

function BluePrint:GetNodeByPin(pin)
  local node = self.bluePrintAllList[pin.nodeUid]
  return node
end

function BluePrint:GetPinById(pinUId)
  local pin = self.bluePrintAllList[pinUId];
  assert(pin, "GetPinById "..tostring(pinUId).." not found ")
  return pin
end

function BluePrint:GetNode(node_id)
  local node = self.bluePrintAllList[node_id]
  return node
end

-- 返回当前选定pin脚,rtti的类型
function BluePrint:TypeOfPin(pin_uid)

  local pin = self.bluePrintAllList[pin_uid]
  if pin == nil then
    ERROR("[TypeOfPin] pid uid is not valid " .. tostring(pin_uid));
    return
  end

  local pinInfo = self:GetPinInfo(pin)
  LOG("pinInfo: " .. tostring(pinInfo.name) .. "," .. tostring(pinInfo.argId) .. "(" .. self.bluePrintNodeList[pin.nodeUid]:GetName() .. ")")

  return pin.pinType, pinInfo.rtti
end


function BluePrint:AddBluePrintNode(luaPath, pos_x, pos_y, ...)

  local bluePrintNode = nil
  local clazz
  if type(luaPath) == 'table' then
    clazz = luaPath
  else
    clazz = require(luaPath)
  end
  bluePrintNode = clazz(self, ...);
  bluePrintNode[BlueDefined.LUA_CLASS_PATH] = luaPath
  bluePrintNode:EditorCreate();

  bluePrintNode.pos = mathfunction.vector2(pos_x,pos_y)

  self:InsertNodeInfo(bluePrintNode);
  return bluePrintNode;
end

--增加节点
-- luaPath string 代表是一个lua文件
-- luaPath table  代表是一个特化的模板节点类
-- ... 具体node构造函数参数
function BluePrint:AddBluePrintNodeEd(luaPath, pos_x, pos_y, ...)

  -- FIXME(hhl) self._UID dry up and self.bluePrintNodeList sparse

  local bluePrintNode = nil

  local clazz
  if type(luaPath) == 'table' then
    clazz = luaPath
  else
    clazz = require(luaPath)
  end
  bluePrintNode = clazz(self);
  bluePrintNode[BlueDefined.LUA_CLASS_PATH] = luaPath

  -- 区分反序列化和交互创建,在加入到list之前调用是为了避免编辑器渲染
  -- 参考 make_array
  bluePrintNode:EditorCreate(...);

  bluePrintNode.pos = mathfunction.vector2(pos_x,pos_y)

  self:InsertNodeInfo(bluePrintNode);

  return bluePrintNode;
end



function BluePrint:CopyBluePrintNode(nodeId, pos)
  local node = self.bluePrintNodeList[nodeId];
  return node:Duplicate(pos);
  -- body
end

function BluePrint:InsertNodeInfo(bluePrintNode)
  for _,v in pairs(bluePrintNode.inputs) do
    self.bluePrintAllList[v.uid] = v
  end
  
  for _,v in pairs(bluePrintNode.outputs) do
    self.bluePrintAllList[v.uid] = v
  end
  
  for _,v in pairs(bluePrintNode.execInputs) do
    self.bluePrintAllList[v.uid] = v
  end
  
  for _,v in pairs(bluePrintNode.execOutputs) do
    self.bluePrintAllList[v.uid] = v
  end

  self.bluePrintNodeList[bluePrintNode.uid] = bluePrintNode;

  self.bluePrintAllList[bluePrintNode.uid] = bluePrintNode;

  if bluePrintNode:GetFunctionType() == BlueDefined.EVENT_FUNCTION_NODE then
    self.bluePrintEventGraph[bluePrintNode.uid] = bluePrintNode;
  end

  if bluePrintNode.needDelay then
    self.DelayNodeList[bluePrintNode.uid] = bluePrintNode;
  end

  if bluePrintNode._OnTickCallBack then
    self.onTickCallbackList[bluePrintNode.uid] = bluePrintNode
  end

end




-- operator
function BluePrint:AddNodePin(node_uid, more_direction)

  local node = self.bluePrintNodeList[node_uid]
  local info
  if more_direction == BlueDefined.MORE_INPUT then
    if node:CanCreateMoreInput() then
      info = node:GetMoreInputsMethod()
    end
  elseif more_direction == BlueDefined.MORE_OUTPUT then
    if node:CanCreateMoreOutput() then
      info = node:GetMoreOutputsMethod()
    end
  elseif more_direction == BlueDefined.MORE_EXEC_INPUT then
    if node:CanCreateMoreExecInput() then
      info = node:GetMoreExecInputsMethod()
    end
  elseif more_direction == BlueDefined.MORE_EXEC_OUTPUT then
    if node:CanCreateMoreExecOutput() then
      info = node:GetMoreExecOutputsMethod()
    end
  end

  if info ~= nil and info.add ~= nil then
    LOG("operator "..tostring(info.add.opname))
    local func = info.add.onAdd;
    local pin = func(node)
    self.bluePrintAllList[pin.uid] = pin
    return pin
  else
    ERROR("[TDD] fail make_array not support add ")
  end

  return nil
end


function BluePrint:AddBlueNodePinToAll(pin)
  if pin.uid == nil or self.bluePrintAllList[pin.uid] ~= nil then
    error("AddBlueNodePinToAll exist pin to add :"..tostring(pin.uid));
  end
  self.bluePrintAllList[pin.uid] = pin;
end


function BluePrint:DeleteNodePin(pin_uid, more_direction)

  local result; --true or false bluedemo
  local tobe_deleted_pin = self.bluePrintAllList[pin_uid]
  if not tobe_deleted_pin then
    ERROR("[DeleteNodePin] pin not exist")
    return false;
  end

  local node = self.bluePrintNodeList[tobe_deleted_pin.nodeUid]
  --移除资源依赖~
  if tobe_deleted_pin.pinType == BlueDefined.PIN_DATA_INPUT 
    or tobe_deleted_pin.pinType == BlueDefined.MORE_INPUT then 
    self:RemoveInputDependency(node, tobe_deleted_pin);
  end

  -- 删除相关连线
  local tobe_deleted_linkIds = {}
  for _, lnk in pairs(tobe_deleted_pin.links) do
    table.insert(tobe_deleted_linkIds, lnk.linkUid)
  end

  for _, linkId in pairs(tobe_deleted_linkIds) do
    self:DeleteBlueLink(linkId);
  end

  -- 再从索引表移除
  self.bluePrintAllList[pin_uid] = nil

  if more_direction then
    local info
    if more_direction == BlueDefined.MORE_INPUT then
      if node:CanCreateMoreInput() then
        info = node:GetMoreInputsMethod()
      end
    elseif more_direction == BlueDefined.MORE_OUTPUT then
      if node:CanCreateMoreOutput() then
        info = node:GetMoreOutputsMethod()
      end
    elseif more_direction == BlueDefined.MORE_EXEC_INPUT then
      if node:CanCreateMoreExecInput() then
        info = node:GetMoreExecInputsMethod()
      end
    elseif more_direction == BlueDefined.MORE_EXEC_OUTPUT then
      if node:CanCreateMoreExecOutput() then
        info = node:GetMoreExecOutputsMethod()
      end
    end

    if info and info.del then
      LOG("operator "..tostring(info.del.opname))
      local operator = info.del.onDel;
      --result = operator(node, tobe_deleted_pin)
      operator(node, tobe_deleted_pin)
    end
  end

  collectgarbage();

  return result
end

function BluePrint:CanDelete(pin_uid)
  local pin = self.bluePrintAllList[pin_uid]
  return pin.delAble or false
end

--删除节点
function BluePrint:DeleteBlueNode(node_uid)

  local node = self.bluePrintNodeList[node_uid]
  if node == nil then
    ERROR("[DeleteBlueNode] node_uid not valid " .. tostring(node_uid))
    return
  end

  -- Step 0. 告知节点即将被删除
  node:DeleteEd();

  -- Step 1. 收集要删除节点的 引脚和连接线信息
  local toBeDeleteLinkId = {}
  local toBeDeletePinId = {}
  if node.inputs ~= nil then
    for _, input in pairs(node.inputs) do
      for _, lnk in pairs(input.links) do
        table.insert(toBeDeleteLinkId, lnk.linkUid)
      end
      table.insert(toBeDeletePinId, input.uid)
      self:RemoveInputDependency(node,input);
    end
  end

  if node.outputs ~= nil then
    for _, output in pairs(node.outputs) do
      for _, lnk in pairs(output.links) do
        table.insert(toBeDeleteLinkId, lnk.linkUid)
      end
      table.insert(toBeDeletePinId, output.uid)
    end
  end

  if node.execOutputs ~= nil then
    for _, execOutput in pairs(node.execOutputs) do
      for _, lnk in pairs(execOutput.links) do
        table.insert(toBeDeleteLinkId, lnk.linkUid)
      end
      table.insert(toBeDeletePinId, execOutput.uid)
    end
  end

  if node.execInputs ~= nil then
    for _, execInput in pairs(node.execInputs) do
      for _, lnk in pairs(execInput.links) do
        table.insert(toBeDeleteLinkId, lnk.linkUid)
      end
      table.insert(toBeDeletePinId, execInput.uid)
    end
  end

  -- Step.2 执行删除: 先删除连线,再删除引脚,再删除节点
  for _, linkId in pairs(toBeDeleteLinkId) do
    self:DeleteBlueLink(linkId);
  end

  for _, PinId in pairs(toBeDeletePinId) do
    self.bluePrintAllList[PinId] = nil;
  end

  self.bluePrintNodeList[node_uid] = nil

  -- 事件节点 从 bluePrintEventGraph 中移除
  if self.bluePrintEventGraph[node_uid] then
    self.bluePrintEventGraph[node_uid] = nil
  end

  if self.DelayNodeList[node_uid] then
    self.DelayNodeList[node_uid] = nil
  end

  if self.onTickCallbackList[node.uid] then
    self.onTickCallbackList[node.uid] = nil
  end

  if self.bluePrintAllList[node_uid] then
    self.bluePrintAllList[node_uid] = nil
  else
    ERROR("[DeleteBlueNode] node_uid in nodelist, but not alllist " .. tostring(node_uid));
  end

  collectgarbage();

end

-- For editor only
-- 特点类型的输入引脚 可能添加了依赖资源
-- 引脚的默认值 依赖的资源文件 都在venusroot 所以这里是否移除没有问题
--             另外目前添加依赖资源只会在 inputreflect.lua,创建蓝图节点是不会产生依赖的
function BluePrint:RemoveInputDependency(node, pin)

  local bluePinInfo = node:GetPinInfoByPin(pin)

  local toRemovePaths = {}
  if bluePinInfo.rtti == apolloengine.TextureEntity():RTTI() then
    if pin.literal ~= nil then
      table.insert(toRemovePaths, pin.literal)
    end
  --[[elseif bluePinInfo.rtti == apolloengine.MaterialEntity():RTTI() then
    if pin.literal ~= nil then
      table.insert(toRemovePaths, pin.literal)
    end]]--
  elseif bluePinInfo.rtti:isTypeOrDriverType(Types.PathType) then
    if pin.literal ~= nil then
      table.insert(toRemovePaths, pin.literal)
    end
  elseif bluePinInfo.rtti:isTypeOrDriverType(Types.DirectoryFileFilterType) then
    if pin.literal ~= nil then
      for _, path in pairs(pin.literal) do
        table.insert(toRemovePaths, path)
      end
    end
  end

  if #toRemovePaths ~= 0 then
    local comp = self.Node:GetComponent(apolloengine.Node.CT_BLUEPRINT)
    for _, path in ipairs(toRemovePaths) do
      WARNING("remove depenedency "..tostring(path))
      comp:RemoveSource(path);
    end
  end

  self:_RemoveInputDependency(toRemovePaths)

end

--bluefunction用
function BluePrint:_RemoveInputDependency(toRemovePaths)
  -- body
end

function BluePrint:TryPinToPin(pin_uid_a, pin_uid_b)

  local pinA = self.bluePrintAllList[pin_uid_a]
  local pinB = self.bluePrintAllList[pin_uid_b]

  if pinA == nil or pinB == nil then
    ERROR("[TryPinToPin] nil ? " .. " PinA = " .. tostring(pinA) .. " PinB = " .. tostring(pinB));
    return false;
  end

  if BlueUtility:IsPinSameDirection(pinA, pinB) then
    ERROR("[TryPinToPin] pin direction is equal ")
    return false;
  end

  if not BlueUtility:IsPinSameFlow(pinA, pinB)  then
    ERROR("[TryPinToPin] pin Flow does not match ")
    return false;
  end

  if BlueUtility:IsDataPin(pinA) then
    local paInfo = self:GetPinInfo(pinA)
    local pbInfo = self:GetPinInfo(pinB)

    -- make sure:
    -- paInfo is output
    -- pbInfo is input
    if paInfo.pinType == BlueDefined.PIN_DATA_INPUT then
      local tmp = paInfo  -- swap
      paInfo = pbInfo
      pbInfo = tmp
      tmp = pinA
      pinA = pinB
      pinB = tmp
    end

    local paIsArray = paInfo.rtti:isArray() or false
    local pbIsArray = pbInfo.rtti:isArray() or false

    if paIsArray ~= pbIsArray then

      if paIsArray then
        local nodeB = self:GetNodeByPin(pinB)
        local compatible = nodeB:IsInputArrayCompatible(pbInfo);
        if compatible then
          local elemType = paInfo.rtti:GetElementType();
          return elemType:isTypeOrDriverType(pbInfo.rtti) ;
        end
      end

      ERROR("[TryPinToPin] array and variable mismatch ")
      return false
    end

    if paIsArray then --都是数组
      if paInfo.rtti == Types.AnyArrayType or pbInfo.rtti == Types.AnyArrayType then
        LOG("[TryPinToPin] AnyArrayType match ")
        return true ;
      end
    else
      if paInfo.rtti == Types.AnyType or pbInfo.rtti == Types.AnyType then
        LOG("[TryPinToPin] AnyType match ")
        return true ;
      end
    end

    if paInfo.rtti:isTypeOrDriverType(pbInfo.rtti) then
      LOG("[TryPinToPin] isTypeOrDriverType match ")
      return true
    else
      return false
    end
  elseif BlueUtility:IsExecPin(pinA) then
      return true
  else
    return false
  end
end

-- 连线
function BluePrint:LinkPinToPin(pin_uid_a, pin_uid_b)
  -- 1.判断是否能够连接(考虑AnyType)
  -- 2.连线
  -- 3.通知两个节点已连接某个引脚(特化)

  if not self:TryPinToPin(pin_uid_a, pin_uid_b) then
    ERROR("[LinkPinToPin] not match");
    return nil
  end

  local pinA = self.bluePrintAllList[pin_uid_a]
  local pinB = self.bluePrintAllList[pin_uid_b]
  local nodeA = self.bluePrintAllList[pinA.nodeUid]
  local nodeB = self.bluePrintAllList[pinB.nodeUid]

  local inputPin = BlueUtility:IsInputPin(pinA) and pinA or pinB;
  local outputPin = BlueUtility:IsOutputPin(pinA) and pinA or pinB;

  if BlueUtility:IsDataPin(pinA) then

    local lnk = {[BD.LINK_UID] = self:_NextID(), [BD.LINK_START] = outputPin.uid, [BD.LINK_END] = inputPin.uid};
    self.bluePrintLinkList[lnk[BD.LINK_UID]] = lnk;

    if inputPin.links[1] ~= nil then
      local thePinOutputTo = self.bluePrintAllList[inputPin.links[1].pinUid]
      if thePinOutputTo then -- 原输入pin有链接某个node输出pin,从这个node的输出pin links中移除
        for i = #thePinOutputTo.links, 1, -1 do
          if thePinOutputTo.links[i].pinUid  == inputPin.uid then
            self:DeleteBlueLink(thePinOutputTo.links[i].linkUid);
          end
        end
      end
    end

    inputPin.useLiteral = false; -- 输入引脚已经连线 所以不使用输入的字面值
    inputPin.links[1] = {nodeUid = outputPin.nodeUid, pinUid = outputPin.uid, linkUid = lnk[BD.LINK_UID]};
    table.insert(outputPin.links, { nodeUid = inputPin.nodeUid, pinUid = inputPin.uid, linkUid = lnk[BD.LINK_UID] })

    nodeA:Linked(pinA, nodeB, pinB);
    nodeB:Linked(pinB, nodeA, pinA);

    return lnk;

  elseif BlueUtility:IsExecPin(pinA) then

    local lnk = {[BD.LINK_UID] = self:_NextID(), [BD.LINK_START] = outputPin.uid, [BD.LINK_END] = inputPin.uid};
    self.bluePrintLinkList[lnk[BD.LINK_UID]] = lnk;

    if outputPin.links[1] ~= nil then
      local thePinInputFrom = self.bluePrintAllList[outputPin.links[1].pinUid] -- 这个输出执行引脚 已经连接到某个输入执行引脚
      if thePinInputFrom then
        for i = #thePinInputFrom.links, 1, -1 do
          if thePinInputFrom.links[i].pinUid == outputPin.uid then
            self:DeleteBlueLink(thePinInputFrom.links[i].linkUid);
          end
        end
      end
    end

    outputPin.links[1] = {nodeUid = inputPin.nodeUid, pinUid = inputPin.uid, linkUid = lnk[BD.LINK_UID]}
    table.insert(inputPin.links, { nodeUid = outputPin.nodeUid, pinUid = outputPin.uid, linkUid = lnk[BD.LINK_UID] })

    nodeA:Linked(pinA, nodeB, pinB);
    nodeB:Linked(pinB, nodeA, pinA);

    return lnk;
  else
    ERROR("[LinkNodePinToNodePin] unknown type " .. tostring(pinA.pinType));
  end
  return nil;
end

--删除连线
function BluePrint:DeleteBlueLink(link_uid)

  -- Step.1 删除连线信息
  local link = self.bluePrintLinkList[link_uid];
  self.bluePrintLinkList[link_uid] = nil;

  if link == nil then
    ERROR("[DeleteLink] link_uid not valid " .. tostring(link_uid));
    return
  end

  local startPin = self.bluePrintAllList[link[BD.LINK_START]];
  local endPin = self.bluePrintAllList[link[BD.LINK_END]];
  local startNode = self.bluePrintAllList[startPin.nodeUid];
  local endNode = self.bluePrintAllList[endPin.nodeUid];

  if startPin == nil or endPin == nil then
    ERROR("[DeleteLink] pin is nil, " .. tostring(startPin) .. ", " .. tostring(endPin));
    return
  end

  if not BlueUtility:IsPinSameFlow(startPin, endPin) then
    ERROR("[DeleteLink] pin.isFlow is not match, ");
    --return -- 目前继续执行 但应该有异常 注意打印
  end

  -- Step.2 如果pin是数据输入,那么需要把 useLiteral 设置为 true
  local dataInputPin = nil
  if BlueUtility:IsDataPin(startPin) then
    if BlueUtility:IsInputPin(startPin) then
      dataInputPin = startPin;
    elseif BlueUtility:IsInputPin(endPin) then
      dataInputPin = endPin;
    else
      ERROR("[DeleteLink] two pin is data, but no one is input ");
      --return ;  -- 目前继续执行 但应该有异常 注意打印
    end
  end

  if dataInputPin ~= nil then
    dataInputPin.useLiteral = true
  end

  -- Step.3 把两个引脚上的连线信息给移除
  local removed = false
  for i = #startPin.links, 1, -1 do
    if startPin.links[i].pinUid == endPin.uid then
      table.remove(startPin.links, i)
      removed = true
    end
  end
  if not removed then  ERROR("[DeleteLink] endPin not found in startPin links's"); end

  removed = false
  for i = #endPin.links, 1, -1 do
    if endPin.links[i].pinUid == startPin.uid then
      table.remove(endPin.links, i)
      removed = true
    end
  end
  if not removed then ERROR("[DeleteLink] startPin not found in endPin links's"); end

  startNode:unLinked(startPin);
  endNode:unLinked(endPin);

end


function BluePrint:DeleteAllBlueLink()
	for k,_ in pairs(self.bluePrintLinkList) do
		self:DeleteBlueLink(k);
	end
end

function BluePrint:DeleteAllBlueNode()
	for k,_ in pairs(self.bluePrintNodeList) do
		self:DeleteBlueNode(k);
	end
  collectgarbage();
end

--设置输入引脚的常量值
--  只有在没有连接的情况下 才能设置常量值
--  即 没有连接 且 不是AnyType 才能设置字面值
--  目前移除RTTI检查 引脚有连接会崩
--  返回旧的常量值
function BluePrint:SetInputPinLiteral(pin_uid, literal)

  local pin = self.bluePrintAllList[pin_uid]

  -- 管脚有效
  if pin == nil then
    ERROR("[SetInputPinConstant] pin is missing " .. tostring(pin_uid));
    return false;
  end

  -- 是否数据节点
  if not BlueUtility:IsDataPin(pin) then
    ERROR("[SetInputPinConstant] pin is not data ");
    return false;
  end

  -- 是否输入
  if not BlueUtility:IsInputPin(pin) then
    ERROR("[SetInputPinConstant] pin is not input ");
    return false;
  end

  -- 字面值 与引脚类型 匹配 -- 目前认为没有必要做这个 因为是通过反射面板设置的 类型应该匹配输入引脚的rtti
  -- local rtti = BlueUtility:GetRttiByValue(literal)
  --assert(false, "NOT SUPPORT FUNCTION")
  --local pinInfo = self:GetPinInfo(pin);
  --if not rtti:isTypeOrDriverType(pinInfo.rtti) then
  --  ERROR("[SetInputPinConstant] type not match");
  --  return false;
  --end

  -- 保存常量
  pin.useLiteral = true
  local oldLiteral = pin.literal;
  pin.literal = literal

  if _KRATOSEDITOR then
    -- 删除原来的连线 (输入引脚 连线应该只有一个) -- 实际不支持这个
    local linkIds = {}
    local checkNum = 0
    for _, lnk in pairs(pin.links) do
      table.insert(linkIds, lnk.linkUid);
      checkNum = checkNum + 1
    end

    assert(checkNum == 0, "data pin found more than 1 input: " .. tostring(checkNum))
    --if checkNum > 1 then
    --  ERROR("[SetInputPinConstant] data pin found more than 1 input: " .. tostring(checkNum));
    --end
    --
    --for _, linkId in pairs(linkIds) do
    --  self:DeleteBlueLink(linkId);
    --end
  end
  return oldLiteral;
end

function BluePrint:RefreshDependencyResource(oldliteral, literal)

end

function BluePrint:SetCommentNodeContent(node_uid, str)
  self.bluePrintNodeList[node_uid].commname = str;
end

function BluePrint:_DoInputDependencyPhone(OneNode)
  --LOG("--_DoInputDependencyPhone--");
  -- 当前节点实例
  local instance = self.bluePrintNodeList[OneNode.uid]

  --当前节点输入依赖已经被解析过 不再重复解析
  if next(instance.dependInputs) then
    return true;
  end
  -- 当前处理节点的输入依赖
  local inputs = OneNode.inputs;

  for _, input in pairs(inputs) do

    local pinInfo = OneNode.infoTable[input.pinType][input.argId]

    if input.useLiteral then
      if BlueUtility:isBlueSelf(input.literal) then
        instance:SetDependencyInputsNode(input.argId, self._selfNode, 0);
      elseif BlueUtility:isBlueprintSelf(input.literal) then
        local value = BlueRtti.Create(pinInfo.rtti, self.Node)
        local inputInstance = ConstantNode(value);
        instance:SetDependencyInputsNode(input.argId, inputInstance, 0);
      else
        local literal = BlueRtti.Create(pinInfo.rtti, input.literal)
        local input_instance = ConstantNode(literal);
        instance:SetDependencyInputsNode(input.argId, input_instance, 0);
      end
      -- 字面值literal,不用解析输入依赖
    else
      -- 一个输入引脚只有一个输入连线
      -- 设置这个节点(方法)第index个参数, 从input_instance的第input_instance_arg_idx个输出
      local outputNode = self.bluePrintNodeList[input.links[1].nodeUid];
      local outputPin = self.bluePrintAllList[input.links[1].pinUid];

      outputNode:Compile();
      instance:SetDependencyInputsNode(input.argId, outputNode, outputPin.argId);
      -- 执行链上的节点(执行节点且执行输入或者执行输出有连线),不解析输入依赖
      local has_valid_exec_input = false
      if outputNode.execInputs ~= nil then
        for _, execInput in pairs(outputNode.execInputs) do
          if execInput.links ~= nil and next(execInput.links) then
            has_valid_exec_input = true;
            break ;
          end
        end
      end
      if not has_valid_exec_input and outputNode.execOutputs ~= nil then
        for _, execOutput in pairs(outputNode.execOutputs) do
          if execOutput.links ~= nil and next(execOutput.links) then
            has_valid_exec_input = true;
            break ;
          end
        end
      end
      if not has_valid_exec_input then
        -- 继续这个输出节点的输入依赖
        self:_DoInputDependencyPhone(outputNode)
      end

    end
  end

  return true;
end


-- 递归处理数据输入依赖
function BluePrint:_DoInputDependency(OneNode, debugDDog)

  -- FIXME(hhl) 需要处理执行引脚构成环(可能是正常的)  数据引脚构成环是错误的
  -- 当前处理结果
  local compileFlag = true ;

  -- 当前节点实例
  local instance = self.bluePrintNodeList[OneNode.uid]

  --当前节点输入依赖已经被解析过 不再重复解析
  if next(instance.dependInputs) then
    return true;
  end

  -- 当前处理节点的输入依赖
  local inputs = OneNode.inputs;

  -- 处理当前节点的输入依赖
  -- 输入依赖的两种终结者(input terminal),这类节点作为其他节点输入依赖的时候,不会递归再检查者这类节点的输入依赖
  -- 1. 执行链上的节点(执行输入引脚 或者 执行输出引脚 有连接)
  -- 2. 字面值literal (引脚没有连接 使用引脚上的值),会自动生成ConstantNode(不会有输入依赖,update无操作,只输出字面值)
  -- 3. 字面值Blueself(),会自动生成BlueSelfNode(不会有输入依赖,update无操作,输出返回的是当前蓝图类的基类对象,对应C++的self指针)

  for _, input in pairs(inputs) do

    local pinInfo = OneNode.infoTable[input.pinType][input.argId]

    if input.useLiteral then
      -- 如果输入常量是BlueSelf  这里需要判断这个引脚的类型和这个蓝图类(基类)是否一样
      if input.literal == nil then
        instance:AppendCompileResult(string.format("input %s use Nil Literal", tostring(pinInfo.argId)))
        self.compileFlag = false
        compileFlag = false ;
      elseif BlueUtility:isBlueSelf(input.literal) then
        --  当函数实参是self(C++)指针,需要检查形参类型和基类是否一致

        if not (pinInfo.rtti:isTypeOrDriverType(self._selfClass)
                or OneNode:LiteralMismatchOnCompile(input, pinInfo) ) then
          ERROR("[input] self not match base class, node " .. tostring(OneNode.luaPath) .. ", input " .. tostring(input.argId))
          instance:AppendCompileResult(
          string.format("Blueprint Self is %s, but argument (%s, %s) need %s ",
            tostring(self._selfClass:GetTypeName()) ,
            tostring(pinInfo.argId),
            tostring(pinInfo.name),
            tostring(pinInfo.rtti:GetTypeName())))

          self.compileFlag = false
          compileFlag = false ;
        end
        LOG("[input] " .. debugDDog .. "(" .. tostring(pinInfo.argId) .. "," .. tostring(pinInfo.name) .. "):" .. "SelfNode")

        -- 所有引脚是self的引脚 共用一个SelfNode实例 相当于self指针
        instance:SetDependencyInputsNode(input.argId, self._selfNode, 0);

      elseif BlueUtility:isBlueprintSelf(input.literal) then
        local blueprint = self:GetTypeName() == "BlueFunction" and self:GetBlueprint() or self ;
        if pinInfo.rtti:GetClassId() == blueprint:GetClassId() then
          local value = BlueRtti.Create(pinInfo.rtti, self.Node)
          local inputInstance = ConstantNode(value);
          instance:SetDependencyInputsNode(input.argId, inputInstance, 0);
        else
          local logStr = string.format("BlueprintClass not match, self is %s, but arg is %s ",
            tostring(blueprint:GetClassName()),
            tostring(pinInfo.rtti:GetDisplayName()))
          instance:AppendCompileResult(logStr)
          ERROR(logStr)

          self.compileFlag = false
          compileFlag = false ;
        end

      --elseif BlueUtility:isNone(input.literal) then -- deprecated

      --  instance:SetDependencyInputsNode(input.argId, BlueNoneNode(), 0);
      elseif pinInfo.rtti:isType(Types.AnyArrayType) then -- BLueRtti.Create -> IsArray 没有判断
        local logStr = string.format("AnyArrayType found, pin %s, %s ",
                                tostring(pinInfo.argId),  tostring(pinInfo.name))
        instance:AppendCompileResult(logStr)
        ERROR(logStr)
        self.compileFlag = false
        compileFlag = false ;
      else
        -- 使用常量值(节点引脚直接配置的值)
        LOG("[input] " .. debugDDog .. "(" .. tostring(pinInfo.argId) .. "," .. tostring(pinInfo.name) .. "):" .. "ConstantNode " .. tostring(input.literal))


        local literal = BlueRtti.Create(pinInfo.rtti, input.literal)
        if literal ~= nil then
          local input_instance = ConstantNode(literal);
          instance:SetDependencyInputsNode(input.argId, input_instance, 0);
        else

          local logStr = string.format("[input] create literal fail, pin %s , rtti:%s, node:%s",
            tostring(input.argId),
            tostring(BlueRtti.GetRealTypeName(pinInfo.rtti)),
            tostring(OneNode:GetName()))
          instance:AppendCompileResult(logStr)
          ERROR(logStr)
          self.compileFlag = false
          compileFlag = false ;

        end
      end
      -- 字面值literal,不用解析输入依赖
    else
      -- 一个输入引脚只有一个输入连线
      if input.links[1] == nil
              or self.bluePrintNodeList[input.links[1].nodeUid] == nil
              or self.bluePrintAllList[input.links[1].pinUid] == nil then
        -- 异常 存在输入引脚没有连接
        ERROR("[input] node " .. tostring(OneNode.luaPath) .. ", input " .. tostring(input.argId) .. " is missing")
        local logStr = "argument "..tostring(input.argId) .. " has no inputs";
        instance:AppendCompileResult(logStr);
        self.compileFlag = false
        compileFlag = false ;

      else
        -- 设置这个节点(方法)第index个参数, 从input_instance的第input_instance_arg_idx个输出
        local outputNode = self.bluePrintNodeList[input.links[1].nodeUid];
        local outputPin = self.bluePrintAllList[input.links[1].pinUid]

        if not outputNode:Compile() then
          -- 内部会标注失败原因
          self.compileFlag = false
          compileFlag = false;
        end


        local outputInfo = self:GetPinInfo(outputPin)
        LOG( string.format("[input] %s(%s,%s):%s,%s,%s", debugDDog,
                tostring(pinInfo.argId),
                tostring(pinInfo.name),
                tostring(outputNode.uid),
                tostring(outputNode:GetName()),
                tostring(outputInfo.argId)) )
        if not outputInfo.rtti:isTypeOrDriverType(pinInfo.rtti) then
          local outputRtti = outputInfo.rtti
          local flag = false
          if outputRtti:isArray() then
            local compatible = OneNode:IsInputArrayCompatible(pinInfo);
            if compatible then
              local elemType = outputRtti:GetElementType();
              flag = elemType:isTypeOrDriverType(pinInfo.rtti) ;
            end
          end
          if not flag then
            local logStr = "[input] "..tostring(input.argId).." type not match"
            instance:AppendCompileResult(logStr);
            ERROR(logStr);
            self.compileFlag = false
            compileFlag = false ;
          end
        end


        instance:SetDependencyInputsNode(input.argId, outputNode, outputPin.argId);

        -- 执行链上的节点,不解析输入依赖
        local has_valid_exec_input = false
        if outputNode.execInputs ~= nil then
          for _, execInput in pairs(outputNode.execInputs) do
            if execInput.links ~= nil and next(execInput.links) then
              has_valid_exec_input = true;
              break ;
            end
          end
        end

        if not has_valid_exec_input and outputNode.execOutputs ~= nil then
          for _, execOutput in pairs(outputNode.execOutputs) do
            if execOutput.links ~= nil and next(execOutput.links) then
              has_valid_exec_input = true;
              break ;
            end
          end
        end

        if has_valid_exec_input then
          LOG("[input] node has exec pin , skip " .. tostring(outputNode:GetName()))
        else
          -- 继续这个输出节点的输入依赖
          local result = self:_DoInputDependency(outputNode, debugDDog .. DEBUG_D_DOG)
          if not result then
            ERROR("[input] node " .. tostring(OneNode.luaPath) .. ", input " .. tostring(input.argId) .. " setup dependency fail")
            compileFlag = false
          end
        end
      end
    end
  end

  if instance.nodeType == BlueDefined.COMPONENT_NODE 
    or instance.nodeType == BlueDefined.COMPONENT_CONTENT_NODE  then
      if libvenuscore.isNil(instance.comp) then
        instance:AppendCompileResult("component is lost");
        self.compileFlag = false
        compileFlag = false ;
      elseif libvenuscore.isNil(instance:GetOutputByIndex()) then
        instance:AppendCompileResult("component content is lost");
        self.compileFlag = false
        compileFlag = false ;
      end
  end

  if instance.nodeType == BlueDefined.REF_NODE then
    if libvenuscore.isNil(instance.refNode) then
      instance:AppendCompileResult("Node is lost");
      self.compileFlag = false
      compileFlag = false ;
    end
  end

  -- 节点建立输入依赖成功
  return compileFlag;
end

-- 顺序处理执行输出链条
function BluePrint:_DoExecDependency(OneNode, debugEDog)

  --该节点往后的执行流已经被解析过,不重复解析
  if next(OneNode.nextExecNodes) then
    return true;
  end

  if _KRATOSEDITOR then ERROR("[exec] " .. debugEDog .. tostring(OneNode:GetName())); end

  if not OneNode:Compile() then
    self.compileFlag = false
    return false
  end

  local compileFlag
  if BD.STRICT_MODE then
    compileFlag = self:_DoInputDependency(OneNode, DEBUG_D_DOG);
  else
    compileFlag = self:_DoInputDependencyPhone(OneNode);
  end

  if not compileFlag then
    ERROR("[exec] setup input dependency for node fail " .. tostring(OneNode.luaPath));
    --  return false; -- 改成状态返回 继续执行编译 暴露全部问题
  end

  -- 当前处理节点的运行实例
  local instance = self.bluePrintNodeList[OneNode.uid]

  -- 该节点的输出执行引脚(多个输出执行引脚)
  local execOutputs = OneNode.execOutputs

  for idx, execOutput in pairs(execOutputs) do

    local pinInfo = self:GetPinInfo(execOutput)

    if execOutput.links[1] ~= nil then

      --ERROR("[exec] next " .. tostring(OneNode:GetName()).." > "..tostring(execOutput.links[1].node:GetName()));
      local nextExecInstance = self.bluePrintNodeList[execOutput.links[1].nodeUid];       -- 下个执行节点实例

      local pinNodeInfo = self:GetPinInfoById(execOutput.links[1].pinUid)
      local nextExecInstanceInputIdx = pinNodeInfo.argId;                                -- 下个执行节点引脚序号
      instance:setExecOutput(pinInfo.argId, nextExecInstance, nextExecInstanceInputIdx); -- 当前节点实例保存下一个要执行的节点实例引用

      -- 执行引脚的一个解析策略: 如果定义了abort,不会解析这个节点(输出执行引脚连接)的下一个执行节点
      if not pinNodeInfo.abort then
        local result
        result = self:_DoExecDependency(self.bluePrintNodeList[execOutput.links[1].nodeUid], debugEDog .. DEBUG_E_DOG);  -- 沿着执行引脚
        if not result then
          compileFlag = false ;
          ERROR("[exec] setup exec dependency fail " .. tostring(OneNode.luaPath));
        end
      else
        LOG("[exec] node " .. tostring(OneNode.luaPath)
                .. " depend on " .. tostring(self.bluePrintNodeList[self.bluePrintAllList[execOutput.links[1].pinUid].nodeUid]:GetName())
                .. ", " .. tostring(pinNodeInfo.name) .. " abort")
      end
    else
      -- 如果输出执行引脚没有配置,应该属于正常,比如节点是整个执行链的最后一个
      if _KRATOSEDITOR then LOG("[exec] node " .. tostring(OneNode.luaPath) .. ", exec out " .. tostring(idx) .. " is missing") end
    end
  end
  return compileFlag;
end

function BluePrint:_PreCompileCode()
  self.compileFlag = true
  self.errorMessage = nil
  if _KRATOSEDITOR then
    for _, node in pairs(self.bluePrintNodeList) do
      node.checkResult = nil
      node.compileResult = nil
      node.dependInputs = {} --输入依赖置空
      node.nextExecNodes = {} -- for multi-time compile
      node.nextExecIndex = {}
      node.nextExecNum = 0
    end
  end
end


-- useEventName = true, event is custom event node
function BluePrint:_CreateEventEntries(eventNodeList, useEventName) -- self.bluePrintEventGraph

  for _, event in pairs(eventNodeList) do
    -- 生成方法的名字
    local name = useEventName and event:GetEventName() or event:GetName();

    local ringExist = false;
    local ringNode;
    --if BD.STRICT_MODE then
    --  local BuEd = require "bluecore.editor.blueutility_ed"
    --  ringExist, ringNode= BuEd:CheckExecRing(event, self.bluePrintAllList);
    --end

    if ringExist then
      self.compileFlag = false;
      self.errorMessage = self.errorMessage and self.errorMessage.."\n\t\t\t\t"..name.." Event Flow Ring Finded" or name.." Event Flow Error Ring Finded";
      ringNode:AppendCompileResult("[Ring Check] This Node is a Ring Start Location!");
    else
      -- 生成方法的名字
      --local name = useEventName and event:GetEventName() or event:GetName();
      -- 一个蓝图允许有多个监听事件节点
      if event:GetEventType() == BlueDefined.EventType.RegisterCallback then
        name = name..event.uid;
      end

      -- 创建执行链和解析输入依赖
      local eventCompileResult = self:_DoExecDependency(event, DEBUG_E_DOG);
      if not eventCompileResult then
        ERROR("[CompileCode] fail to _DoExecDependency " .. tostring(name))
        self.errorMessage = self.errorMessage and self.errorMessage.."\n\t\t\t\t"..name.." Event Flow Error" or name.." Event Flow Error";
        if _KRATOSEDITOR then event:AppendCompileResult("fail to generate event function ");  end
        -- 当前事件节点处理函数生成失败,不影响下一个事件节点处理函数生成
      else

        local eventNode = self.bluePrintNodeList[event.uid] ;

        -- 记录每个事件入口信息
        self.EventEntries[name] = {}
        self.EventEntries[name].entry = eventNode

        local isNotBeginPlay = name ~= "_OnBeginPlay" and true or false

        -- 生成事件对应的函数/执行链
        self.EventEntries[name].func = function(self, args)
          -- from EventDispatcher, args is event.params (all EventNodes using EventDispatcher)
          -- from OnTick/OnBeginPlay, args is deltaTime
          -- from func_call_node to BlueFunction/CustomEventNode , args is table, args[1] args[2] .. args[n]

          -- 把函数的参数给到第一个节点(事件节点)
          if _KRATOSEDITOR then
            local BuEd = require "bluecore.editor.blueutility_ed"
            BuEd:BlueDebugYield(eventNode);
          end

          if self.isFirstUpdate and isNotBeginPlay then
            local skipTime = self.EventEntries[name].skipTime
            skipTime = skipTime and skipTime + 1 or 1
            self.EventEntries[name].skipTime = skipTime
            if skipTime % 20 == 1 then
              WARNING("blueprint Not Start, name "..tostring(name)..", "..tostring(self.Node.Name));
            end
            return
          end

          --更新事件节点的引脚输出值  事件节点在事件发生时候不用获取引脚参数
          eventNode.outputResults = {eventNode:_OnUpdate(args)};

          -- 由事件节点决定下一个执行的是哪一个node
          local nextIdx = eventNode:_OnNextBranch();

          -- 事件节点返回0 不执行后续节点
          if nextIdx == nil or nextIdx <= 0  then
            --ERROR("[Update] return; out of range, next = " .. tostring(nextIdx))
            return
          end

          -- 事件图表 事件节点后继的执行节点
          local lastNode = eventNode;
          local nextIndex = eventNode.nextExecIndex[nextIdx];
          local nextNode = eventNode.nextExecNodes[nextIdx];

          while nextNode ~= nil
          do
            if _KRATOSEDITOR then
              local linkUid = lastNode.execOutputs[nextIdx].links[1].linkUid;
              self:AddExecLinkList(linkUid);
            end
            nextIdx = nextNode:Update(nextIndex);
            if nextIdx == nil or nextIdx <= 0  then
              --ERROR("[Update] break; out of range, next = " .. tostring(nextIdx))
              break
            end
            lastNode = nextNode;
            nextIndex = nextNode.nextExecIndex[nextIdx];
            nextNode =  nextNode.nextExecNodes[nextIdx];
          end
        end -- self.EventEntries[name].func
      end
    end
  end

  if BD.STRICT_MODE then
    local BuEd = require "bluecore.editor.blueutility_ed"
    BuEd:ClearPinExtraLabel(self.bluePrintAllList);
  end


end

function BluePrint:_ClearEventEntries()
  if self.EventEntries ~= nil then
    for name, _ in pairs(self.EventEntries) do
      self[name] =  nil
    end
  end
  self.EventEntries = {}
end

-- 对蓝图进行编译(可重复执行)
-- 1. 索引->引用
-- 2. rtti检查
-- 3. 变量/node/component引用检查等
-- 4. 回调所有依赖节点的 _OnCompile 函数(返回false代表错误)
-- 5. 检查变量的值 isPrefabMode
--    对于复杂类型变量的处理
--    作为函数的参数(形参)  默认值应该是nil 但在产生的函数调用节点引脚字面值是NotNullType(必须链接)
--    作为函数的局部变量  不选择的话,运行时会产生nil
--    作为类的变量(目前无论是public还是private) 默认值都是nil,在prefab编辑器或者大场景都可在反射面板选择,但:
--          a. 在大场景中所有实例的变量都必须非nil(也就是已选择）
--          b. prefab编辑器中可不选择,但在大场景的prefab实例必须选择
--          c. 嵌套prefab实例 如果在prefab中没有选择,那么大场景中必须选择
function BluePrint:CompileCode(isPrefabMode)
  
  WARNING("starting to compile in "..(isPrefabMode and "prefab " or " big scene"))

  -- Step 0.清除之前的Compile信息
  self:_PreCompileCode();

  --编译局部函数
  for _, func in pairs(self.blueFunctionList) do

    if not func:CompileCode() then
      self.compileFlag = false
      local funcVarCompileError = func:GetErrorMessage();
      local errMess = funcVarCompileError and funcVarCompileError or "ExecFlow"
      self.errorMessage = "function "..tostring(func:GetName()).." compile error! ".."\n\t\t\t\t"..errMess;
      return false;
    end

  end

  -- Step 1.生成实例的成员方法
  -- 		事件节点 会给 BluePrintGraph对象 增加成员方法 比如 Update(deltaTime)
  -- 		每个事件节点的后继执行节点 由自己根据输入参数来决定 比如 KeyA(press,released)
  self.EventEntries = {}

  self:_CreateEventEntries(self.bluePrintEventGraph, false)  -- 预置事件节点
  self:_CreateEventEntries(self._customEventNodeArray, true)-- 自定义事件节点

  -- Step 2.生成实例的成员变量/注册变量
  -- 目前不作为ScriptCompoent导出的变量 重复MemberRegister会崩溃
  self:_SetupVariable(isPrefabMode);

  self._isModify = false ;
  WARNING("finish compiling, result: "..tostring(self.compileFlag ))
  return self.compileFlag ;
end

function BluePrint:_SetupVariable(isPrefabMode)

  if self.lastVarArray ~= nil then
    for _, varName in pairs(self.lastVarArray) do
      self[varName] = nil
    end
  end

  -- 用于每次编译都把上一次编译生成的变量从blueprint清除
  self.lastVarArray = {}
  for _, variable in pairs(self.bluePrintVariableList) do
    local rtti = variable:GetRtti()
    local name = variable:GetName()
    if variable.is_array then
      -- 数组元素为了可以直接反射,存放在ArrayItemWrap里面,所以这里需要'解封装'
      self[name] = {}
      rtti = rtti:GetElementType()
      for _, wrap in ipairs(variable:GetDefault()) do
        local value = BlueRtti.Create(rtti, wrap.value)
        table.insert(self[name], value)
      end
    else
      local value = BlueRtti.Create(rtti, variable:GetDefault())
      if value == nil and not isPrefabMode then
        self.compileFlag = false
        self.errorMessage = "variable " .. tostring(variable:GetDisplayName()) .." 's value is invalid"
        ERROR(self.errorMessage);
      end
      self[name] = value;
    end

    table.insert(self.lastVarArray, name);
  end
end

function BluePrint:IsModify()
  return self._isModify ;
end

function BluePrint:IsSync()
  return self._requestSync and true or false
end

function BluePrint:IsCompileSuccess()
  return self.compileFlag ;
end

function BluePrint:GetErrorMessage()
  return self.errorMessage;
end

function BluePrint:AddDebugPoint(node_uid)
  self.bluePrintNodeList[node_uid].debugStatus = true;
end

function BluePrint:DeleteDebugPoint(node_uid)
  self.bluePrintNodeList[node_uid].debugStatus = false;
end


-- 生成唯一ID
function BluePrint:_NextID()
  self._UID = self._UID + 1
  return self._UID
end

-- 提供给BlueNode获取当前pin和node实例
function BluePrint:_Get(index)
  return self.bluePrintAllList[index]
end

-- 运行时候,设置对应的事件处理函数到本blueprint对象
-- 对于监听事件处理函数
-- 注册 -- Run
-- 取消注册 -- 编辑器stop和删除BlueprintComponent(_OnDestroy) 端内unload素材(_OnDestroy)
function BluePrint:Run()
  self.isrunning = true;
  -- 记录蓝图修改过的Node
  self:_SaveRefNodeStatus();
  if self.EventEntries ~= nil then
    for name, eventEntry in pairs(self.EventEntries) do
      self[name] =  eventEntry.func
      local eventNode = eventEntry.entry
      if eventNode:GetEventType() == BlueDefined.EventType.RegisterCallback then
        assert(eventEntry.entry.RegisterTriggerFunc ~= nil, "[Run] not trigger")
        eventEntry.entry:RegisterTriggerFunc(eventEntry.func, self);
      end
    end
  end
  for _,func in pairs(self.blueFunctionList) do
    func:Run();
  end
  -- TODO RUN全局函数
end

function BluePrint:IsRunning()
  return self.isrunning
end

function BluePrint:Stop()

  if not self.isrunning  then LOG("BluePrint stop yet"); return ; end

  WARNING(" ---- -------------- ----- ")
  WARNING(" ---- BluePrint:Stop ----- ")
  WARNING(" ---- -------------- ----- ")

  self.isrunning = false;
  self.execLinkList = {};

  EventDispatcher:DeregisterCallback(self);

  if self.bluePrintNodeList ~= nil then
    for _,v in pairs(self.bluePrintNodeList) do
      v:Stop();
    end
  end

  self:_ClearEventEntries();
  self:_ClearRefNodeStatus();

  self.isFirstUpdate = true;

  if self.blueFunctionList ~= nil then
    for _,func in pairs(self.blueFunctionList) do
      func:Stop();
    end
  end
  --TODO STOP 全局函数

end

function BluePrint:GetVariableRttiByName(name)
  for _, var in pairs(self.bluePrintVariableList) do
    if var.name == name then
      assert(var.rtti, "[GetVariableRttiByName] "..tostring(name).." nil");
      return var.rtti
    end
  end
end

-- 蓝图reset
-- 1. 所有蓝图节点重置(清空参数和返回值)
-- 2. 重置变量值
function BluePrint:Reset()

  if self.bluePrintNodeList ~= nil then
    for _,v in pairs(self.bluePrintNodeList) do
      v:Reset();
    end
  end

  self:_ResetRefNodeStatus();
  self:_SetupVariable();

  for _, func in pairs(self.blueFunctionList) do
    func:Reset()
  end
  
end

function BluePrint:_TickOnNodes(def)
  -- call node's _OnTick
  for nodeId, node in pairs(self.onTickCallbackList) do
    node:_OnTickCallBack(def)
  end
end

function BluePrint:_RunDelayNodes()
  for _, obj in pairs(self.DelayNodeList) do
    obj:_OnUpdate();
  end
end

function BluePrint:AddExecLinkList(link_uid)
  if not self._isExecutingCustomEvent then
    table.insert(self.execLinkList, link_uid);
  else
    table.insert(self.customEventExecLinkList[self.currentEventName], link_uid);
  end
end

function BluePrint:SetCustomEventRunLabel(status, eventName)
  self._isExecutingCustomEvent = status;
  self.currentEventName = eventName;
  --同一个自定义事件，每一次进入都要把执行链清除掉
  if status then
    self.customEventExecLinkList[self.currentEventName] = {};
  end
end



--[[
BlueGraph实例创建
BlueNode Class创建(luaPath是个table 动态创建的节点类，比如设置变量节点类)
BlueNode Class内部成员反序列
BlueNode实例创建
BlueNode实例内部成员反序列(连接信息)
BlueNode实例 _OnDeserialize 被回调 (动态注册引脚信息，比如array节点类的输入和输出)

bluePrintVariableList 实例创建 (variable和nodes序列化不分先后)
--- 到这里BlueGraph中全部单体完成反序列化


BlueGraph:Awake() ?? -- BluePrintGraph:OnDeserializeDone
BlueGraphic AllList 重建 (到这里应该所有的)
BluePrintVariableList中的变量通过default得到rtti ???  基础类型 数组类型 复杂类型 ???
BlueNode Class的 _OnDeserializePost 被回调 -- 重置节点类的rtti和default值
BlueNode 的 _OnDeserializePost 被回调 (make array和for 会根据连接信息确认实例的引脚的rtti, 目前没有根据字面值来反推rtti)

]]--

function BluePrint:_OnUpdate(def)
  self.deltaTime = def

  --self.execLinkList = {};
  
  if self._OnTick then
    self:_OnTick(def)
  end
  self:_RunDelayNodes()
  self:_TickOnNodes(def)
end

function BluePrint:_OnStart()
  if self.isFirstUpdate and self._OnBeginPlay then
    self:_OnBeginPlay();
  end
  self.isFirstUpdate = false;
  WARNING("BluePrint onStart: "..tostring(self.Node and self.Node.Name or "n/a"));
end

function BluePrint:SetPrefab(flag)
  if self._isP and not flag then
    ERROR("Prefab should be Irreversible"); -- prefab不可逆
  end
  self._isP = flag ;
end

function BluePrint:IsPrefabNilEd()
  -- forward compatible
  return self._isP == nil and true or false
end

function BluePrint:IsPrefabEd()
  return self._isP and true or false
end

function BluePrint:_OnDeserialize()
  if self:GetClassId() ~= nil then
    -- 如果还没有ClassId 说明是旧的素材 延后到Awake注册
    -- 同时这类素材没有类的概念,也就没有引用这个类的地方,所以可延后,下次重新打开就会有类概念
    if _KRATOSEDITOR then
      local BuEd = require "bluecore.editor.blueutility_ed"
      BuEd:_CheckClassIdConflictEd(self); -- self.Node is nil
    end
    BlueRtti.RegisterClassAddRef(self);
  end
end

function BluePrint:EditorCreate()
  BlueRtti.RegisterClassAddRef(self, self.Node);
end

function BluePrint:_OnAwake()
  if _KRATOSEDITOR and self:GetClassId() == nil then -- 旧素材编辑器打开创建ClassId
    BlueRtti.RegisterClassAddRef(self, self.Node);
  end
  self:_AwakeProcess();
end

function BluePrint:_AwakeProcess()

  -- 记录当前blueprint所在的node
  LOG("[_OnAwake] self.Node " .. tostring(self.Node))
  self[BlueDefined.THIS] = self.Node;

  -- forward compatible
  self:SetPrefab(self.Node.PrefabPath ~= nil and self.Node.PrefabPath ~= "");

  -- 反序列化最后,完成依赖顺序的重建部分
  if next(self.blueFunctionList) ~= nil then
    --先让bluefunction _OnAwake
    for _, func in pairs(self.blueFunctionList) do
      func:_OnAwake(self.Node);
    end
  end

  self:OnDeserializeDone()

  -- 反序列化完成
  LOG("awake it "..tostring(self._isAwake)..","..tostring(self:GetObjectID()))
  self._isAwake = true ;
end

function BluePrint:_SaveRefNodeStatus()
  for k,v in pairs(self.bluePrintNodeList) do
    if v.nodeType == BlueDefined.REF_NODE and not libvenuscore.isNil(v:GetRef()) then
      local stID = v.refNode:GetStaticID();
      self.contentStatusTable[stID] = self.contentStatusTable[stID] or v.refNode.Active;
    end
    if v.nodeType == BlueDefined.COMPONENT_NODE and not libvenuscore.isNil(v:GetRef()) then
      local stID = v.comp:GetStaticID();
      local status = v.comp:isTypeOrDriverType(cv.IVisionComponent:RTTI()) and v.comp.Enable or v.comp.Active;
      self.contentStatusTable[stID] = self.contentStatusTable[stID] or status;
    end
  end

  for varUid, varInfo in pairs(self.bluePrintVariableList) do
    local isArray = varInfo:IsArray();
    local default = varInfo:GetDefault();
    if not isArray then
      local typeInfo = varInfo:GetTypeInfomation("default_value")
      if typeInfo:GetTypeName() == "ReferenceReflector" then
        local stID = default:GetStaticID();
        self.contentStatusTable[stID] =  default.Active;
      end
    else
      if next(default) ~= nil then
        local arrayItemWrap1 = default[1]
        local typeInfo = arrayItemWrap1:GetTypeInfo()
        if typeInfo:GetTypeName() == "ReferenceReflector" then
          -- typeInfo:isType(Types.ReferenceType) is wrong, because ReferenceType:isType is compare reference_type
          -- that is, all element is reference type/C++/Node/Component
          for arrayIdx, def in ipairs(default) do
            local value = def:Get();
            local stID = value:GetStaticID();
            self.contentStatusTable[stID] =  value.Active;
          end
        end
      end
    end
  end
end

function BluePrint:_ClearRefNodeStatus()
  self.contentStatusTable = {}
end

function BluePrint:_ResetRefNodeStatus()
  --node component的激活状态重置
  for k,v in pairs(self.bluePrintNodeList) do
    if v.nodeType == BlueDefined.REF_NODE then
      local stID = v.refNode:GetStaticID();
      v.refNode.Active = self.contentStatusTable[stID];
    end
    if v.nodeType == BlueDefined.COMPONENT_NODE then
      local stID = v.comp:GetStaticID();
      if v.comp:isTypeOrDriverType(cv.IVisionComponent:RTTI()) then
        v.comp.Enable = self.contentStatusTable[stID];
      else
        v.comp.Active = self.contentStatusTable[stID];
      end
    end
  end

  for varUid, varInfo in pairs(self.bluePrintVariableList) do
    local isArray = varInfo:IsArray();
    local default = varInfo:GetDefault();
    if not isArray then
      local typeInfo = varInfo:GetTypeInfomation("default_value")
      if typeInfo:GetTypeName() == "ReferenceReflector" then
        local stID = default:GetStaticID();
        default.Active = self.contentStatusTable[stID]
      end
    else
      if next(default) ~= nil then
        local arrayItemWrap1 = default[1]
        local typeInfo = arrayItemWrap1:GetTypeInfo()
        if typeInfo:GetTypeName() == "ReferenceReflector" then
          -- typeInfo:isType(Types.ReferenceType) is wrong, because ReferenceType:isType is compare reference_type
          -- that is, all element is reference type/C++/Node/Component
          for arrayIdx, def in ipairs(default) do
            local value = def:Get();
            local stID = value:GetStaticID();
            value.Active = self.contentStatusTable[stID]
          end
        end
      end
    end
  end
end

function BluePrint:_OnDestroy()
  -- 编辑器--蓝图Component删除 ->  _OnDestroy
  -- 端内 --素材释放 -> _OnDestroy
  self:Stop();

  if _KRATOSEDITOR then

    BlueRtti.UnRefBluePrintClass(self);
    -- 打开Prefab编辑器 只有TopNode的GetHostPrefabPath是true 子节点都是false
    local isPrefab =  self.Node.HostPrefabPath ~= "";

    ERROR("------------ Node "..tostring(self.Node)..".. isPrefab:"..tostring(isPrefab))
    if not isPrefab  then -- TODO(hhl) 把大场景的蓝图类如果删除的话,需要从ClassDeclareMap中删除
      ERROR("[TODO] remove blueprint in FullScene from  ClassDeclareMap ");
    end

  end
end


function BluePrint:OnDeserializeDone()

  if next(self.bluePrintNodeList) ~= nil then

    for _, node in pairs(self.bluePrintNodeList) do

      self.bluePrintAllList[node.uid] = node

      if node:IsNodeType(BlueDefined.EVENT_FUNCTION_NODE) then
        LOG("Found Event Node !")
        self.bluePrintEventGraph[node.uid] = node
      --elseif node:IsNodeType(BlueDefined.CUSTOM_EVENT_NODE) then
      --  LOG("Found Custom Event Node !")
      --  table.insert(self._customEventNodeArray, node)
      -- CustomEventNode:OnDeserialize add itself to _customEventNodeArray
      end

      if node._OnTickCallBack then
        self.onTickCallbackList[node.uid] = node
      end

      if node.needDelay then
        LOG("Found Delay Type Node !")
        self.DelayNodeList[node.uid] = node
      end

      for _, pin in pairs(node.inputs) do
        self.bluePrintAllList[pin.uid] = pin
      end

      for _, pin in pairs(node.outputs) do
        self.bluePrintAllList[pin.uid] = pin
      end

      for _, pin in pairs(node.execInputs) do
        self.bluePrintAllList[pin.uid] = pin
      end

      for _, pin in pairs(node.execOutputs) do
        self.bluePrintAllList[pin.uid] = pin
      end
    end
  end

  for _, variable in pairs(self.bluePrintVariableList) do
    variable:_OnDeserializePost()
    if _KRATOSEDITOR then --非编辑器不用恢复上下文同步
      -- 变量节点类
      local cls_set,cls_get = self._blueContext:_RegisterVariableNodeClass(variable);
        variable:Subscribe(cls_set)
        variable:Subscribe(cls_get)
        variable:Subscribe(self._blueContext);
    end
  end
  -- 变量节点类恢复
  self._blueContext:_OnDeserializePost()

  for _, node in pairs(self.bluePrintNodeList) do
    if node._OnDeserializePost then
      node:_OnDeserializePost()
    end
  end

  --local BlueDumper = require "bluecore.bluedumper"
  --BlueDumper.dump(self);

  LOG("Done");
end

function BluePrint:GetBlueNodeList()
  return self.bluePrintNodeList
end

function BluePrint:GetBlueFtnList()
  return self.blueFunctionList;
end

function BluePrint:GetBlueFtnById(funcId)
  local blueFunc = self.blueFunctionList[funcId]
  if blueFunc then
    error("GetBlueFtnById not found "..tostring(funcId));
  end
  return blueFunc
end


function BluePrint:GetBlueNodeList()
  return self.bluePrintNodeList;
end

-- 每次显示蓝图的时候 都要同步一下(func_call_node函数调用节点)
function BluePrint:SyncEd()

  WARNING("Sync >>>>> ")
  WARNING(string.format("Sync ---------- Node:%s;Class:%s;", self.Node and self.Node:GetName() or "nil", self:GetClassName() or "nil") )
  WARNING("Sync <<<<< ")

  self._requestSync = false

  if self.bluePrintNodeList ~= nil then
    for _, node in pairs(self.bluePrintNodeList) do
      if node.SyncEd then node:SyncEd() end
    end
  end

  if self.blueFunctionList ~= nil then
    for _, func in  pairs(self.blueFunctionList) do
      func:SyncEd();
    end
  end
end


-- 1: 准备切换成prefab实例(针对prefab实例再成prefab文件)
-- 2: 从普通节点切换到prefab实例
function BluePrint:ChangeToPrefabEd(state)
  if state == 1 then
    for _, varInfo in pairs(self.bluePrintVariableList) do
      varInfo:PrepareToPrefabEd()
    end
  elseif state == 2 then
    for _, varInfo in pairs(self.bluePrintVariableList) do
      varInfo:PostToPrefabEd()
    end
  else
    error(string.format("ChangeToPrefabEd error state %s", tostring(state)));
  end
end

-- Hook ! 必须放在最后,当编辑状态监听是否修改过
if _KRATOSEDITOR then

  local isConstFunction = function(name)
    local result = string.startsWith(name,'__' ) -- FIXME(hhl) 需要重新考虑一下监听函数的名字
    result = result or string.startsWith(name,'_')
    result = result or string.startsWith(name,'Try');
    result = result or string.startsWith(name,'Get');
    result = result or string.startsWith(name,'Is');
    result = result or string.startsWith(name,'TypeOfPin');
    result = result or string.startsWith(name,'CanDelete');
    result = result or string.startsWith(name,'Run');
    result = result or string.startsWith(name,'Stop');
    result = result or string.startsWith(name,'AddDebugPoint');  
    result = result or string.startsWith(name,'DeleteDebugPoint'); 
    --增加删除断点不应该触发stop重新编译 导致yield住蓝图stop掉
    result = result or string.startsWith(name,'AddExecLinkList');
    result = result or string.startsWith(name,'SetCustomEventRunLabel');
    result = result or string.startsWith(name,'SetCommentNodeContent');
    result = result or string.startsWith(name,'Reset');
    result = result or string.startsWith(name,"EditorCreate")
    result = result or string.startsWith(name,"new")
    result = result or string.startsWith(name,"CompileCode")
    return result
  end

  WARNING("------------------");
  for name, member in pairs(BluePrint) do
    if type(member) == 'function'  and not isConstFunction(name) then
      WARNING("Blue: "..name.." is non-const function");
      local packagedFunc = member ;
      --local useAutoSync = autoSyncFunctions[name] or false
      local hook = function(self, ...)
        LOG("----------------------------------------------------------")
        LOG("Modify by "..tostring(name)..", node:"..tostring(self.Node))
        LOG("----------------------------------------------------------")
        self._isModify = true ; -- control compile button
        self._requestSync = true ; -- request auto Sync
        self:Stop();
        local result = packagedFunc(self, ...)
        --if useAutoSync then self:Sync(); end
        return result
      end
      BluePrint[name] = hook --针对原函数多包装了一层进行封装 以后访问函数就从hook开始
    end
  end
end

return BluePrint

