local vc = require "venuscore"
local ae = require "apolloengine"
local mf = require "mathfunction"
local cv = require "computervisionfunction"

local FeedFaceSkinBehavior = vc.VenusBehavior:extend("FeedFaceSkinBehavior"); 

local dtSize = cv.GPUSegmentComponent.mlRectSize
local dtPosition = cv.GPUSegmentComponent.mlRectPosition
local dtPivot = cv.GPUSegmentComponent.mlRectPivot
local dtRoll = cv.GPUSegmentComponent.mlRectRoll;

local anchorsTable =
{
  [cv.GPUSegmentComponent.cvGPU_Mask_Left_Eye]   =
  {
    mf.vector3(0.1, 1.0, 0.2),
    mf.vector3(0.0, 1.0, 1.15),
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Right_Eye]  =
  {
    mf.vector3(0.0, 0.9, 0.2),
    mf.vector3(0.0, 1.0, 1.15),
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Nose]       =
  {
    mf.vector3(0.4, 0.6, 0.35),
    mf.vector3(0.3, 0.7, 0.65),
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Mouth]      =
  {
    mf.vector3(0.15, 0.85, -0.1),
    mf.vector3(0.15, 0.85, 0.9),
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Left_Brow]  = 
  {
    mf.vector3(0.25, 1.0, 0.0),
    mf.vector3(0.2, 1.0, 1.0),
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Right_Brow] =
  {
    mf.vector3(0.0, 0.75, 0.0),
    mf.vector3(0.0, 0.8, 1.0),
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_FullFace]   =
  {
    mf.vector3(0.25, 0.75, 0.2),
    mf.vector3(0.25, 0.75, 0.8),
  },
}
local alphaAdjust = 4.0

local anchorsTable2 =
{
  [cv.GPUSegmentComponent.cvGPU_Mask_Left_Eye]   =
  {
    { 52, 64, 0.5 },
    { 52,  4, 0.3 },
    { 55, 80, 0.3 },
    { 55, 67, 0.5 },
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Right_Eye]   =
  {
    { 61, 71, 0.5 },
    { 61, 26, 0.3 },
    { 58, 81, 0.3 },
    { 58, 68, 0.5 },
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Nose]   =
  {
    { 78, 44, 0.5 },
    { 79, 44, 0.5 },
    { 83, 46, 0.5 },
    { 82, 46, 0.5 },
    --{ 37, 38, 0.5 },
    --{ 82,  5, 0.6 },
    --{ 46, 49, 0.5 },
    --{ 83, 27, 0.6 },
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Mouth]   =
  {
    { 85, 82, 0.5 },
    { 95, 14, 0.5 },
    { 91, 18, 0.5 },
    { 89, 83, 0.5 },
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Left_Brow]   =
  {
    { 65, 35, 2.0 },
    { 37, 38, 0.3 },
    { 67, 68, 0.3 },
    { 64, 52, 0.5 },
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_Right_Brow]   =
  {
    { 70, 40, 2.0 },
    { 38, 37, 0.3 },
    { 68, 67, 0.3 },
    { 71, 61, 0.5 },
  },
  [cv.GPUSegmentComponent.cvGPU_Mask_FullFace]   =
  {
    { 66, 36, 2.0 },
    { 69, 39, 2.0 },
    --{ 90, 19, 0.5 },
    --{ 84, 13, 0.5 },
    { 83, 26, 0.1 },
    { 82,  6, 0.1 },
  },
}

function FeedFaceSkinBehavior:new()
  self.rollMode = 0;
  self.FixedHorizontal = 0;
  self.skin = nil;
  self.FaceID = 1;
  self.feathering = 0;
  self.MaskContourVal = 0;
  --self.TexturePath = "DEVICE_CAPTURE";
  --self.Texture = nil;
  self.RenderOrder = 0;
end

function FeedFaceSkinBehavior:_OnUpdate(delta)

-----------------------------------------------------------------------------------------------
-- set parameter for GPUSegment component
-----------------------------------------------------------------------------------------------
  local gpuSeg = self.Node:GetComponent(ae.Node.CT_CV_GPUSEGMENT);
  if gpuSeg == nil then
    return
  end
  
  gpuSeg.FaceID = self.FaceID;
  gpuSeg.Feathering = self.feathering;
  gpuSeg.MaskContourVal = self.MaskContourVal;

  local face106 = gpuSeg:Face106P()
  if face106 == nil then
    return
  end
  --local p82 = face106:Get(82+1)
  --WARNING(string.format("SUNTYLOG: p82 (%f %f)", p82:x(), p82:y()))
 
  local cpuSeg = self.Node:GetComponent(ae.Node.CT_CV_SEGMENT);
  if cpuSeg ~= nil then
    local maskTex = cpuSeg:GetMask();
    local rectsize = cpuSeg:GetData(dtSize)[dtSize];
    local rectpos = cpuSeg:GetData(dtPosition)[dtPosition];
    if rectsize and rectpos then
      gpuSeg:SetRectInfo(mf.vector2(rectpos[1], rectpos[2]), mf.vector2(rectsize[1], rectsize[2]));
    else
      gpuSeg:SetRectInfo(mf.vector2(0,0), mf.vector2(0,0));
    end
    if maskTex then
      gpuSeg:SetMaskTexture(maskTex);
    end
    cpuSeg.FaceID = self.FaceID;
  end
  
  local fgTex = gpuSeg:GetMask()
  local data = gpuSeg:GetData();
  local bgTex = gpuSeg:GetTexture()
  local bgSize = gpuSeg:GetDetectSize()
  if fgTex == nil or data == nil or bgTex == nil or bgSize == nil then
    return
  end
  local texPos = data[dtPosition];
  local texSize = data[dtSize];
  local texPivot = data[dtPivot];
  local texRoll = data[dtRoll];

  if texPos == nil or texSize == nil or texPivot == nil or texRoll == nil then
    return
  end

  --WARNING(string.format("SUNTYLOG: texSize %d %d", texSize[1], texSize[2]))

-----------------------------------------------------------------------------------------------
-- set parameter for child skin
-----------------------------------------------------------------------------------------------
  if self.skin == nil then
    return;
  end
  local skinrender = self.skin:GetComponent(ae.Node.CT_RENDER);
  if skinrender == nil then
    return;
  end

  local currType = gpuSeg.Type
  local chosenAnchors = anchorsTable2[currType]

  --coordinate origin of gpu-seg-data is at left-top corner
  --this matrix convert from fg coord to bg coord, both origins of which are at left-bottom corner
  local scale1 = mf.Matrix33:CreateScaleMatrix(mf.vector2(texSize[1], -texSize[2]))
  local movePivot1 = mf.Matrix33:CreateTranslateMatrix(mf.vector2(-texPivot[1], texPivot[2]))
  local rotate = mf.Matrix33:CreateRotateMatrixZ(texRoll[1])
  local movePivot2 = mf.Matrix33:CreateTranslateMatrix(mf.vector2(texPivot[1], -texPivot[2]))
  local movePosition = mf.Matrix33:CreateTranslateMatrix(mf.vector2(texPos[1], bgSize:y() - texPos[2]))
  local scale2 = mf.Matrix33:CreateScaleMatrix(mf.vector2(1.0 / bgSize:x(), 1.0 / bgSize:y()))

  local matFgToBg = scale1 * movePivot1 * rotate * movePivot2 * movePosition * scale2

  skinrender:SetParameter("_FgTex", fgTex)
  skinrender:SetParameter("_BgTex", bgTex)
  skinrender:SetParameter("_MatFgToBg1", matFgToBg:Row(0))
  skinrender:SetParameter("_MatFgToBg2", matFgToBg:Row(1))
  skinrender:SetParameter("_MatFgToBg3", matFgToBg:Row(2))
  if chosenAnchors then
    for k = 1, 4 do
      local item = chosenAnchors[k]
      local p1 = face106:Get(item[1] + 1)
      local p2 = face106:Get(item[2] + 1)
      local p = p1 * (1.0 - item[3]) + p2 * item[3]
      skinrender:SetParameter("_BgAnchor" .. tostring(k), p)
    end
  end
  skinrender:SetParameter("_BgSize", bgSize)
  skinrender:SetParameter("_AlphaAdjust", mf.vector1(alphaAdjust))

  -- set render order for sticker node
  skinrender:SetRenderOrder(self.RenderOrder);


-----------------------------------------------------------------------------------------------
-- set layer mask for both sticker and skin based on the layer value of current node(father)
-----------------------------------------------------------------------------------------------
  local layerItems = ae.LayerMask:GetLayerNames();
  for k, v in pairs(layerItems) do
    if v ~= ae.LayerMask.MASK_EDITOR_SCENE_LAYER_NAME and 
      v ~= ae.LayerMask.MASK_EDITOR_UI_LAYER_NAME then
      local islayer = self.Node:isLayer(v);
      if islayer == true then
        self.skin:SetLayer(v);
        break;
      end
    end
  end
end

-----------------------------------------------------------------------------------------------
-- reigistered member set/get function
-----------------------------------------------------------------------------------------------

function FeedFaceSkinBehavior:GetSkin()
  return  self.skin;
end

function FeedFaceSkinBehavior:SetSkin(in_skin)
  self.skin = in_skin;
end

--[[
function FeedFaceSkinBehavior:LoadTexture(TexturePath)
  
  local extension = filelfs:getExtension(TexturePath);
  extension  = extension or defined.TextureType.ReFS;
  local Texture = ae.TextureEntity();

  if extension == defined.TextureType.ReFS then
    Texture:PushMetadata(ae.TextureReferenceMetadata(TexturePath))
    Texture:CreateResource();
  else
    Texture:PushMetadata(ae.TextureFileMetadata(
      ae.TextureEntity.TU_STATIC,
      ae.TextureEntity.PF_AUTO,
      1, false,
      ae.TextureEntity.TW_CLAMP_TO_EDGE,
      ae.TextureEntity.TW_CLAMP_TO_EDGE,
      ae.TextureEntity.TF_LINEAR,
      ae.TextureEntity.TF_LINEAR,
      vc.IFileSystem:PathAssembly(TexturePath)));
    
    Texture:SetKeepSource(true);
    Texture:CreateResource();
  end
  local texStream = Texture:GetSourceStream();
  if texStream == nil then
    return nil;
  else
    return Texture;
  end
end
]]

function FeedFaceSkinBehavior:GetRenderOrder()
  return self.RenderOrder;
end

function FeedFaceSkinBehavior:SetRenderOrder(in_RenderOrder)
  self.RenderOrder = in_RenderOrder;
end

function FeedFaceSkinBehavior:GetFaceID()
  return self.FaceID;
end

function FeedFaceSkinBehavior:SetFaceID(in_FaceID)
  self.FaceID = in_FaceID;
end

-- real feathering value range[0, 60] normalized to [0, 20] when displayed
function FeedFaceSkinBehavior:SetFeathering(in_feathering)
  self.feathering = in_feathering;--/20.0*60;
end

function FeedFaceSkinBehavior:GetFeathering()
  local NormFeathering = self.feathering;--/60.0*20;
  return NormFeathering;
end

function FeedFaceSkinBehavior:SetMaskContourVal(in_MaskContourVal)
  self.MaskContourVal = in_MaskContourVal;
end

function FeedFaceSkinBehavior:GetMaskContourVal()
  return self.MaskContourVal;
end

--[[
function FeedFaceSkinBehavior:GetTexturePath()
  return self.TexturePath;
end

function FeedFaceSkinBehavior:SetTexturePath(in_texturepath)
  self.TexturePath = in_texturepath;
  self.Texture = self:LoadTexture(self.TexturePath) or self.Texture;
end
]]

-----------------------------------------------------------------------------------------------
-- reigistered member(UI)
-----------------------------------------------------------------------------------------------
-- hide
FeedFaceSkinBehavior:MemberRegister(
    "skin",
    vc.ScriptTypes.ReferenceType(
      ae.Node:RTTI(),
      FeedFaceSkinBehavior.GetSkin,
      FeedFaceSkinBehavior.SetSkin
    ),
"none");

FeedFaceSkinBehavior:MemberRegister(
  "Render Order",
  vc.ScriptTypes.IntType(
    nil, nil,
    FeedFaceSkinBehavior.GetRenderOrder,
    FeedFaceSkinBehavior.SetRenderOrder
  )
);

-- add interface to GPUsegment component FaceID 
FeedFaceSkinBehavior:MemberRegister(
    "FaceID",
    vc.ScriptTypes.ComboType(
      {
        {
          key = "0",
          value = 1
        },
        {
          key = "1",
          value = 2
        },
        {
          key = "2",
          value = 3
        },
      },
      FeedFaceSkinBehavior.GetFaceID,
      FeedFaceSkinBehavior.SetFaceID
));

--[[
FeedFaceSkinBehavior:MemberRegister("TexturePath",
  vc.ScriptTypes.FilePathType(
    ae.TextureEntity:RTTI(),
    FeedFaceSkinBehavior.GetTexturePath,
    FeedFaceSkinBehavior.SetTexturePath
),
"none");]]


-- add interface to GPUsegment component MaskContour value
FeedFaceSkinBehavior:MemberRegister(
     "Expand/Shrink",
     vc.ScriptTypes.FloatType(
       0, 1,
       FeedFaceSkinBehavior.GetMaskContourVal,
       FeedFaceSkinBehavior.SetMaskContourVal
));

-- add interface to GPUsegment component feathering value
FeedFaceSkinBehavior:MemberRegister(
     "Feathering",
     vc.ScriptTypes.FloatType(
       0, 1,
       FeedFaceSkinBehavior.GetFeathering,
       FeedFaceSkinBehavior.SetFeathering
));

return FeedFaceSkinBehavior;