local venuscore = require "venuscore"
local venusjson = require "venusjson"
local apolloengine = require "apolloengine"
local defined = require "behavior.cartoon_behavior.defined"
local utils = require "behavior.cartoon_behavior.utils"

local ExpressionAnimationPlayBehavior = venuscore.VenusBehavior:extend("ExpressionAnimationPlayBehavior")
local CONTROLLER_SCRIPT = "load_behavior.lua"
local MORPHER_SCRIPT = "morpher_control_behavior.lua"

function ExpressionAnimationPlayBehavior:new()
  self.morpher_list = {}
  self.ExpressionFile = ""
  self.expression_finished = true
end

function ExpressionAnimationPlayBehavior:_OnAwake()
  if self.ExpressionFile ~= "" then
    self:LoadExpressionAnimation()
  end
  if self.Controller ~= nil then
    self:RegisterNode()
  end
end

function ExpressionAnimationPlayBehavior:_OnUpdate(delta)
  if self.ExpressionFile == nil or self.expression_finished then
    return
  end
  self:PlayExpression(self:GetExpression(delta))
end

function ExpressionAnimationPlayBehavior:Play()
  self.current_frame = 1
  self.current_time = self.expression_frames[1].time
  self.expression_finished = false
end

function ExpressionAnimationPlayBehavior:GetStatus()
  return defined.animation_status.DONE
end

function ExpressionAnimationPlayBehavior:RegisterNode()
  local controller_behavior = utils:GetBehavior(self.Controller, CONTROLLER_SCRIPT)
  if controller_behavior ~= nil then
    controller_behavior:RegisterAnimationNode(self.Node)
  end
end

function ExpressionAnimationPlayBehavior:RegisterMorpher(node)
  local behavior = utils:GetBehavior(node, MORPHER_SCRIPT)
  if behavior then
    table.insert(self.morpher_list, behavior)
  end
end

function ExpressionAnimationPlayBehavior:GetExpressionFile()
  return self.ExpressionFile
end

function ExpressionAnimationPlayBehavior:SetExpressionFile(file)
  self.ExpressionFile = file
  self:LoadExpressionAnimation()
end


function ExpressionAnimationPlayBehavior:LoadExpressionAnimation()
  local config = venusjson.LaodJsonFile(self.ExpressionFile)
  if config.frames == nil then
    return false
  end
  self.expression_frames = config.frames
  if #self.expression_frames == 0 then
    return false
  end
end

function ExpressionAnimationPlayBehavior:GetExpression(delta)
  local next_time = self.current_time + delta
  while self.current_frame < #self.expression_frames do
    local frame = self.expression_frames[self.current_frame + 1]
    if frame.time > next_time then
      break
    end
    self.current_frame = self.current_frame + 1
  end
  if self.current_frame == #self.expression_frames then
    self.expression_finished = true
    return self:GetWeightsAt(self.current_frame)
  end
  self.current_time = next_time
  return self:GetWeightsFrom(next_time, self.current_frame)
end

function ExpressionAnimationPlayBehavior:PlayExpression(tracker_weight)
  for _, behavior in ipairs(self.morpher_list) do
    behavior:Play(tracker_weight)
  end
end

function ExpressionAnimationPlayBehavior:GetWeightsAt(from_frame)
  local result = {}
  for name, weight in pairs(self.expression_frames[from_frame].weights) do    
    result[name] = weight
  end
  return result
end

function ExpressionAnimationPlayBehavior:GetWeightsFrom(time, from_frame)
  local rate = (time - self.expression_frames[from_frame].time) / (self.expression_frames[from_frame + 1].time - self.expression_frames[from_frame].time)
  local result = {}
  
  local to_frame = from_frame + 1
  for name, start_weight in pairs(self.expression_frames[from_frame].weights) do    
    local next_weight = self.expression_frames[to_frame].weights[name] or 0
    result[name] = start_weight * (1- rate) + next_weight * rate
  end
  for name, next_weight in pairs(self.expression_frames[to_frame].weights) do
    if result[name] == nil then
      local start_weight = self.expression_frames[from_frame].weights[name] or 0
      result[name] = start_weight * (1- rate) + next_weight * rate
    end
  end
  return result
end


ExpressionAnimationPlayBehavior:MemberRegister(
  "ExpressionFile",
  venuscore.ScriptTypes.FilePathType(
    {"json"},
    ExpressionAnimationPlayBehavior.GetExpressionFile,
    ExpressionAnimationPlayBehavior.SetExpressionFile
  )
)

ExpressionAnimationPlayBehavior:MemberRegister(
  "Controller",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.Node:RTTI(),    
    nil,
    function(thiz, value)
      thiz.Controller = value
      thiz:RegisterNode()
    end
  )
)

return ExpressionAnimationPlayBehavior