require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local OuterCanthus = {}

function OuterCanthus:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function OuterCanthus:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function OuterCanthus:_InitParams(defined)
  beautydefined = defined
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_PTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_OUTERCANTHUS_PTS");
  apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_COEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_OUTERCANTHUS_COEF");
  apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_FALLOFF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_OUTERCANTHUS_FALLOFF");
  apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_REF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_OUTERCANTHUS_REF");
  apolloengine.ShaderEntity.OUTERCANTHUS_PARAM = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"OUTERCANTHUS_PARAM");
  apolloengine.ShaderEntity.OUTERCANTHUS_PARAMRADIUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"OUTERCANTHUS_PARAMRADIUS");
end

function OuterCanthus:SetParams(config, shaderParams)
  --shaderParams.outercanthusPointArray = mathfunction.vector4array();
  shaderParams.outercanthusParamArray = mathfunction.vector2array();
  --shaderParams.outercanthusRefArray = mathfunction.vector2array();
  shaderParams.outercanthusParamRadiusArray = mathfunction.vector2array();

  for i = 1, #beautydefined.OutercanthusKey + 1, 1 do
    shaderParams.outercanthusParamArray:PushBack(beautyutils._ArrayToVec2(config.OutercanthusParam[i]));
    shaderParams.outercanthusParamArray:PushBack(beautyutils._ArrayToVec2(config.OutercanthusParam[i]));
    
    shaderParams.outercanthusParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.OutercanthusParamRadius[i]));
    shaderParams.outercanthusParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.OutercanthusParamRadius[i]));
  end
  
  self:_InitLens(config)
end

function OuterCanthus:_InitLens(config)
  self.lens = {}
  for i = 1, #beautydefined.OutercanthusKey do
    local index1 = beautydefined.OutercanthusKey[i][1]+1;
    local index2 = beautydefined.OutercanthusDir[i][1]+1;
    self.lens[#self.lens + 1] = Len(index1, index2, -config.OutercanthusParam[i][1], -config.OutercanthusParam[i][2] * 5, config.OutercanthusParamRadius[i][1])
    
    index1 = beautydefined.OutercanthusKey[i][2]+1;
    index2 = beautydefined.OutercanthusDir[i][2]+1;
    self.lens[#self.lens + 1] = Len(index1, index2, -config.OutercanthusParam[i][1], -config.OutercanthusParam[i][2] * 5, config.OutercanthusParamRadius[i][1])
  end
end

function OuterCanthus:_InitQuad(quadnode, shaderParams)
  --quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_PTS, shaderParams.outercanthusPointArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.OUTERCANTHUS_PARAM, shaderParams.outercanthusParamArray);
  --quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_REF, shaderParams.outercanthusRefArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.OUTERCANTHUS_PARAMRADIUS, shaderParams.outercanthusParamRadiusArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_COEF, mathfunction.vector1(0.0));
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_FALLOFF, mathfunction.vector1(1.0));
end

function OuterCanthus:_DrawQuad(quadnode, keypoints, rate, config)
  -- param: args = {rate, config}
  local outercanthusPointArray = mathfunction.vector4array();

  local center = {keypoints[beautydefined.OutercanthusDir[1][1] + 1], keypoints[beautydefined.OutercanthusDir[1][2] + 1]}
  local alpha = config.OutercanthusDistAlphaParam;

  for i = 1, #beautydefined.OutercanthusKey, 1 do
    for j = 1, 2 do
      local index1 = beautydefined.OutercanthusKey[i][j]+1;
      local index2 = beautydefined.OutercanthusDir[i][j]+1;
      local point = {keypoints[index1][1], (1+alpha) * keypoints[index1][2] - alpha * center[j][2]}
      local dirPoint = { keypoints[index2][1], keypoints[index2][2] }
      outercanthusPointArray:PushBack(beautyutils._ArrayToVec4(point, dirPoint));
    end
  end
    
  for j = 1, 2 do
    local point = {
      (keypoints[beautydefined.OutercanthusKey[1][j]+1][1] + keypoints[beautydefined.OutercanthusKey[2][j]+1][1]) / 2,
      (keypoints[beautydefined.OutercanthusKey[1][j]+1][2] + keypoints[beautydefined.OutercanthusKey[2][j]+1][2]) / 2
    }
    point = {point[1], (1+alpha) * point[2] - alpha * center[j][2]}
    local index = beautydefined.OutercanthusDir[1][j]+1
    local dirPoint = { keypoints[index][1], keypoints[index][2] }
    outercanthusPointArray:PushBack(beautyutils._ArrayToVec4(point, dirPoint));
  end
  
  --LOG(self.coef.." "..rate);
  local outercanthusRefArray = mathfunction.vector2array();
  for i = 1, #beautydefined.OutercanthusEyeRef do
    local refPoint1 = keypoints[beautydefined.OutercanthusEyeRef[1][i] + 1]
    local refPoint2 = keypoints[beautydefined.OutercanthusEyeRef[2][i] + 1]
    local refPoint = { (refPoint1[1] + refPoint2[1]) / 2,  (refPoint1[2] + refPoint2[2]) / 2}
    outercanthusRefArray:PushBack(beautyutils._ArrayToVec2(refPoint))
  end
  
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_COEF,mathfunction.vector1(self.coef * rate));
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_PTS,outercanthusPointArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_REF,outercanthusRefArray);
end

function OuterCanthus:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = self.coef * rate
  local tmpDis = {}
  for i = 1, #self.lens do
    local srcIdx = self.lens[i]:GetSrcIdx()
    local dstIdx = self.lens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.lens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)  
    local tmp = self.lens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx] = tmp
  end
  
  for k, v in pairs(tmpDis) do
    keypointsDisplacement[k][1] = keypointsDisplacement[k][1] + v[1]
    keypointsDisplacement[k][2] = keypointsDisplacement[k][2] + v[2]
  end
  
  if #keypoints >= 240 then
    for i = 1, #beautydefined.AdvOutercanthusWarpMap do
      local src = beautydefined.AdvOutercanthusWarpMap[i][1] + 107
      local dst = beautydefined.AdvOutercanthusWarpMap[i][2] + 1
      keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.AdvOutercanthusWarpCoef[i]
      keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.AdvOutercanthusWarpCoef[i]
    end
  end
end

function OuterCanthus:UpdateParaValue(quadnode, config, key_index, attribute, value)
  if (attribute == "X") then
    config.OutercanthusParam[key_index - beautydefined.OuterCanthus_start + 1][1] = value;
  elseif (attribute == "Y") then
    config.OutercanthusParam[key_index - beautydefined.OuterCanthus_start + 1][2] = value;
  elseif (attribute == "R") then
    config.OutercanthusParamRadius[key_index - beautydefined.OuterCanthus_start + 1][1] = value;
  elseif (attribute == "DistAlpha") then
    config.OutercanthusDistAlphaParam = value;
  end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.OUTERCANTHUS_PARAM, shaderParams.outercanthusParamArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.OUTERCANTHUS_PARAMRADIUS, shaderParams.outercanthusParamRadiusArray);
end

function OuterCanthus:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0;
  else
    self.coef = 0.0;
  end
  LOG("Nose Length coef set to: " .. self.coef);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_OUTERCANTHUS_COEF,mathfunction.vector1(self.coef * rate));
end

function OuterCanthus:GetParams(config)
  local outercanthusParam = beautyutils.Clone(config.OutercanthusParam);
  local outercanthusParamRadius = beautyutils.Clone(config.OutercanthusParamRadius);
  local outercanthusDistAlpha = config.OutercanthusDistAlphaParam;
  LOG("outercanthusParam size: " .. #outercanthusParam);
  return outercanthusParam, outercanthusParamRadius, outercanthusDistAlpha
end

function OuterCanthus:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --LOG("Outercanthus coef: " .. self.coef)
end

function OuterCanthus:Release()
  self.lens = {}
end
  
return OuterCanthus