local AE = require "apolloengine"
local libvenuscore = require "libvenuscore"
local Types = require "venuscore.rtti.types"
local BlueRtti = require "bluecore.core.blue_rtti"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local ArrayItemWrap = require "bluecore.core.array_item_wrap"
local BlueVariableInfo = libvenuscore.Object:extend("BlueVariableInfo");


--[[
UE4 创建变量 支持
1. 变量名字
2. 变量类型
3. 容器类型 (单个变量/数组/集/映射(字典))
4. 访问修饰符(私有/公开/保护)
5. 是否允许在类蓝图中修改改变量(const)
6. 在关卡蓝图上该实例上是否可编辑(const getProperty())
7. 提示文本信息

UE4
1. 需要修改变量的信息都需要重新编译; 修改RTTI类型(含数组)会导致连线断开(不会自动断开但是会提示,编译会出错)
2. 对于复杂类型(引用) 在变量属性面板上 是None, 在蓝图类实例的反射面板上是引用,可选场景中的任何对象

对于变量类型是数组  SET变量节点的输入引脚 是没有立即数 可以设置的

立即数的类型:int float string bool vec1~4 quaternion self(固定); 复杂类型都是引用 数组类型不连接时候为空数组

本模板节点不能直接在蓝图类中直接创建 需要特化成节点(类)后才能在上下文菜单中显示
]]--

function BlueVariableInfo:_GetCombo()
  --LOG("value = "..tostring(self.base_rtti))
  return self.base_rtti_name;
end

function BlueVariableInfo:_SetCombo(value)
  LOG("[BlueVariableInfo] SetCombo "..tostring(self.base_rtti_name).."->"..tostring(value))
  local old = self.base_rtti_name;
  self.base_rtti_name = value -- maybe is clsID
  self:_RedefineDefault();
  self:_NotifyChange(BD.VarProp.BaseType, old, self.base_rtti_name)
end

function BlueVariableInfo:_GetIsArray()
  return self.is_array
end

function BlueVariableInfo:_SetIsArray(value)
  local old = self.is_array
  self.is_array = value ;
  self:_RedefineDefault();
  self:_NotifyChange(BD.VarProp.IsArray, old, self.is_array)
end

function BlueVariableInfo:_GetName()
  return self.name
end

function BlueVariableInfo:_SetName(name)
  if self._CreatedDone then
    error("BlueVariableInfo:_SetName refuse to change name after setup");
  end
  self.name = name ;
end

function BlueVariableInfo:_SetDispName(dispName)

  local expected_prefix = self.graph:GetTypeName() == "BlueFunction" and BD.BPFUNCVAR_PREFIX or BD.BPCLASSVAR_PREFIX;
  if string.sub(dispName, 1, 2) ~= expected_prefix then
    dispName = expected_prefix..tostring(dispName);
  end
  for _,var in pairs(self.graph.bluePrintVariableList) do
    if var:_GetDispName() == dispName then
      WARNING("[BlueVariableInfo] _SetDispName Fail! Name Repeated!!!");
      return;
    end
  end
  local oldName = self.var_name
  self.var_name = dispName ;
  self:_NotifyChange(BD.VarProp.Name, oldName, self.var_name)
end

function BlueVariableInfo:_GetDispName()
  return self.var_name ;
end

function BlueVariableInfo:_GetDefault()
  return self.default_value
end

function BlueVariableInfo:_SetDefault(newDefault)
  self.default_value = newDefault;
end

function BlueVariableInfo:_CreateEmptyItem()

  -- varObj.default_value[1].__typeinfos["value"]:GetTypeName() 获取数组元素的typeInfo 默认是BaseType
  -- 如果是 float/vector 等类型 即使没有了typeInfo会落到 BaseTypeReflector 反射, 但这样缺少了正确的typeInfo
  -- 改变每个实例的TypeInfo但是不改变类的TypInfo

  local item = ArrayItemWrap()
  local tinfo = self:_GetTypeInfo(self.base_rtti,false, true);
  if tinfo then -- 非Lua Object需要typeinfo支持
    item:SetTypeInfo(tinfo)
  else
    WARNING("array elemenet self.base_rtti  is lua object "..tostring(self.base_rtti))
  end
  return item
end

function BlueVariableInfo:_CreateDefaultItem()
  local item = self:_CreateEmptyItem();
  item.value = BU:GetDefaultByRtti(self.base_rtti);
  return item
end

function BlueVariableInfo:_GetDefaultArray()
  --LOG("_GetDefaultArray: " .. TableToStr(self.default_value));
  return { unpack(self.default_value) }
end

function BlueVariableInfo:_SetDefaultArray(value)
  --LOG("_SetDefaultArray: " .. TableToStr(value));
  self.default_value = { unpack(value) }
end

function BlueVariableInfo:_RedefineDefault()

  local rtti, rttiOfArray = BlueRtti.GetRtti(self.base_rtti_name, self.is_array);

    -- For BlueprintClass base_rtti_name is ClassID
  self.base_rtti = rtti

  -- 修改类型 需要重新分配default值
  if not self.is_array then
    -- 避免反序列过程中 由于rtti_name和default_value等设置的顺序 导致默认值被修改的问题--反序列化使用序列化的默认值
    if self._CreatedDone or self.default_value == nil then
      self.default_value = BU:GetDefaultByRtti(self.base_rtti)
    end
    self.rtti = self.base_rtti
  else
    if self._CreatedDone or self.default_value == nil then
      self.default_value = {} -- 如果是数组 默认值是个空数组
    end
    self.rtti = rttiOfArray
  end

  -- 更新实际的Typeinfo
  -- 1. lua ojbect 不使用typeinfo
  -- 2. 非lua object 返回新的Basetype实例 替换原来的 __typeinfos['default_value']
  local typeinfo = self:_GetTypeInfo(self.base_rtti, self.is_array, false,
          self.is_array and BlueVariableInfo._GetDefaultArray or BlueVariableInfo._GetDefault,
          self.is_array and BlueVariableInfo._SetDefaultArray or BlueVariableInfo._SetDefault)

  if typeinfo ~= nil then
    typeinfo:SetKeyname("default_value"); -- basetype 会生成get和set
    self.__typeinfos["default_value"] = typeinfo;
  else
    self.__typeinfos["default_value"] = nil;
    ERROR(string.format("_RedefineDefault %s is lua object", tostring(self.base_rtti:GetTypeName())))
  end

  local typeinfo2 = self:_GetTypeInfo(self.base_rtti, self.is_array, false,
    self.is_array and BlueVariableInfo._GetDefaultArrayOverride or BlueVariableInfo._GetDefaultOverride,
    self.is_array and BlueVariableInfo._SetDefaultArrayOverride or BlueVariableInfo._SetDefaultOverride)

  if typeinfo2 ~= nil then
    typeinfo2:SetKeyname("default_value_override"); -- basetype 会生成get和set
    self.__typeinfos["default_value_override"] = typeinfo2;
    typeinfo2.tag = self:_HideInPrefabEditor() and "none" or nil
  else
    self.typeinfo2["default_value_override"] = nil;
    error(string.format("_RedefineDefaultOverride %s is lua object", tostring(self.base_rtti:GetTypeName())))
  end

end


function BlueVariableInfo:_GetTypeInfo(baseRtti, isArray, isArrayItem, getFtn , setFtn)

  if isArray then
    assert(getFtn and setFtn , "function must be given if array");
    assert(not isArrayItem, "not support 2d-array")
    return Types.ArrayType(BlueVariableInfo._CreateDefaultItem, getFtn, setFtn);
  else
    local typeInfo = BlueRtti.GetTypeInfo(baseRtti)
    if typeInfo == Types.ReferenceType then
      return typeInfo(baseRtti, getFtn, setFtn); -- new Types.ReferenceType()
    elseif baseRtti.GetClassId then -- base_rtti是BlueprintType 所以这个变量类型是蓝图类(引用)

      local targetClassId = baseRtti:GetClassId()

      local filterCallback = function(objRef)
        local blueprintComp = objRef:GetComponent(AE.Node.CT_BLUEPRINT);
        if blueprintComp ~= nil then
          local blueprint = BU.GetIns(blueprintComp)
          if blueprint ~= nil then
            local classId = blueprint:GetClassId();
            if classId == targetClassId then
              return true ;
            end
          end
        end
        return false
      end -- function  filterCallback

      local blueprintTypeRef = Types.ReferenceType(AE.Node:RTTI(), getFtn, setFtn);
      blueprintTypeRef:SetFilterCallback(filterCallback)

     return blueprintTypeRef
    end

    if typeInfo ~= nil then
      return typeInfo(getFtn, setFtn);
    else -- 目前不支持脚本对象
      error("[BlueVariableInfo] Not support "..tostring(baseRtti:GetTypeName()));
    end

    --if baseRtti:isTypeOrDriverType(libvenuscore.Object) then
    --  if isArrayItem then -- 如果数组元素是Lua Object,这里不用返回
    --    error("[BlueVariableInfo] lua object no __typeinfo ");
    --  else -- 如果是default_value是 lua object,这里需要返回
    --    return Types.BaseType(getFtn, setFtn);
    --  end
    --else
    --  error( "[BlueVariableInfo] Not support "..tostring(baseRtti:GetTypeName()));
    --end
  end
end

-- UE4 类型修改后 需要立刻重新编译 并且提示错误连接
-- UE4 名字修改后 变量列表-变量节点-右键上下文菜单(节点类) 都会同步修改
function BlueVariableInfo:_NotifyChange(property, old_value, new_value)
  if _KRATOSEDITOR then
    if not self.observer then
      return -- 反序列化过程中 不会有Observer 最后在变量 _OnDeserializePost之后才会 设置对应的变量节点类
    end
    for _, o in ipairs(self.observer) do
      LOG("[_NotifyChange] observer "..tostring(o.onPropertyChange))
      if o.onPropertyChange then
        o:onPropertyChange(self, property, old_value, new_value);
      else
        ERROR("[_NotifyChange] "..tostring(o).." lost change:("..tostring(property)..")"
                ..tostring(old_value)..">"..tostring(new_value))
      end
    end
  else
    LOG("[_NotifyChange] running mode, skip")
  end
end

function BlueVariableInfo:Subscribe(observer)
  self.observer = self.observer or {}
  table.insert(self.observer, observer);
end


-- 如果是复杂类型 default_value的初始值 应该是 NoneType()
-- UE目前不支持数组的数组,二维数组
-- basertti如果是数组的话 应该是数组元素的类型
function BlueVariableInfo:new(graph, base_rtti_name, is_array, name, is_private, is_const, is_final, tips, displayName)

  self.graph = graph
  self.base_rtti_name = base_rtti_name
  self.is_array = is_array or false
  self.rtti = nil ;
  self.base_rtti = nil;
  self.name = name or "n/a"
  self.var_name = displayName or self.name
  self.is_private = is_private == nil and true or is_private
  --self.is_const = is_const or false
  --self.is_final = is_final or false;
  --self.is_ref = false
  --self.tips = tips or "N/A";


  -- 当前实现相当于在反序列,BlueVariableInfo new实例时 先要设置一些信息(base_rtti_name,is_array)
  -- 来动态的更新某些Member的typeinfo

  self.__typeinfos = {}
  local clazz = getmetatable(self)
  for k, v in pairs(clazz.__typeinfos) do
    self.__typeinfos[k] = v;
  end

  -- 至少要根据base_rtti_name和is_array生成rtti和typeinfo,用来反序列化使用
  -- 反序列化 在new会传入base_rtti_name和is_array
  self:_RedefineDefault();

  -- prefab editor, always is nil
  self.default_value_override = nil;
  self.base_rtti_name_override = nil;
  self.is_array_override = nil;

end

function BlueVariableInfo:EditorCreate()
  self._CreatedDone = true
end

function BlueVariableInfo:PostToPrefabEd()
  if _KRATOSEDITOR then
    WARNING(string.format("PostToPrefabEd %s", self.var_name))
    local BlueFunction = require "bluecore.bluefunction"
    if not self.graph:isType(BlueFunction) then -- BLueFunction is local variable
      self._isPrefabEd = true ;
      local EditorSystem = require "window.editor.system.editsystem"
      if not EditorSystem:GetPrefabMode() then
        self:_ResetOverrideEd();
      end
    end
  end
end

function BlueVariableInfo:PrepareToPrefabEd()
  -- 避免prefab文件包含这些属性
  if _KRATOSEDITOR then
    WARNING(string.format("PrepareToPrefabEd %s", self.var_name))
    self.base_rtti_name_override = nil;
    self.is_array_override = nil;
    self.default_value_override = nil ;
  end
end


function BlueVariableInfo:_OnDeserialize()
  self._CreatedDone = true
  if self.var_name == 'n/a' then self.var_name = self.name end ;
  -- 编辑器 旧素材用新编辑器 就会自动用原来的name作为dName 保存后成为新的素材

  if self.is_array then
    for _, itemWrap in pairs(self.default_value) do
      -- _CreateDefaultItem
      -- reset each item of type_info in ArrayItemWrap
      local tInfo = self:_GetTypeInfo(self.base_rtti,false, true);
      if tInfo then
        itemWrap:SetTypeInfo(tInfo)
      else
        error("_GetTypeInfo not type info")
      end
    end

    if self.default_value_override ~= nil then
      for _, itemWrap in pairs(self.default_value_override) do
        local tInfo = self:_GetTypeInfo(self.base_rtti,false, true);
        if tInfo then
          itemWrap:SetTypeInfo(tInfo)
        else
          error("_GetTypeInfo not type info")
        end
      end
    end
  end
end

function BlueVariableInfo:_OnDeserializePost()
  if _KRATOSEDITOR then
    local BlueFunction = require "bluecore.bluefunction"
    if not self.graph:isType(BlueFunction) then -- BLueFunction is local variable
      local isPrefab = self.graph.Node.HostPrefabPath ~= nil and self.graph.Node.HostPrefabPath ~= "";
      if isPrefab then
        self._isPrefabEd = true ;
        local EditorSystem = require "window.editor.system.editsystem"
        if not EditorSystem:GetPrefabMode() then
          -- prefab实例在反序列化的时候 检查Override是否存在 已经是否匹配; 私有变量每次都同步prefab(若原来是public并修改,这里会恢复为prefab的)

          --self.default_value_override == nil 有两种情况
          --1. prefab拖到场景形成prefab实例
          --2. 把场景中node(变量为引用类型且不是数组)拖到asset形成prefab实例进入prefab编辑器再保存返回
          if self.default_value_override == nil or self.is_private then
            self:_ResetOverrideEd();
          else
            -- 只有当类型修改的时候 做一次同步default; 因为rtti/base_rtti_name/is_array始终同步prefab文件 所以prefab实例的接口依旧用原来的
            if self.base_rtti_name_override ~= self.base_rtti_name or self.is_array_override ~= self.is_array then
              self:_ResetOverrideEd();
            end

            if self.is_array then
              for _,arrayItem in ipairs(self.default_value_override) do
                if libvenuscore.isNil(arrayItem.value) then
                  self:SyncPrefabDefaultToOverrideEd();
                  break;
                end
              end
            end
          end
        else -- just make sure
          self.default_value_override = nil;
          self.base_rtti_name_override = nil;
          self.is_array_override = nil;
        end
      end
    end
  end
end

function BlueVariableInfo:SyncPrefabDefaultToOverrideEd()
  local BlueFunction = require "bluecore.bluefunction"
  if not self.graph:isType(BlueFunction) then -- BLueFunction is local variable
    local isPrefab = self.graph.Node.HostPrefabPath ~= nil and self.graph.Node.HostPrefabPath ~= "";
    if isPrefab then
      local EditorSystem = require "window.editor.system.editsystem"
      if not EditorSystem:GetPrefabMode() then
        self:_ResetOverrideEd(); -- Force Sync Default Value
      end
    end
  end
end

function BlueVariableInfo:_GetDefaultArrayOverride()
  if self.default_value_override == nil then
    return nil
  end
  return { unpack(self.default_value_override) }
end

function BlueVariableInfo:_SetDefaultArrayOverride(value)
  if value == nil then
    return
  end
  self.default_value_override = { unpack(value) }
end

function BlueVariableInfo:_GetDefaultOverride()
  return self.default_value_override
end

function BlueVariableInfo:_SetDefaultOverride(newDefault)
  self.default_value_override = newDefault;
end

function BlueVariableInfo:_ResetOverrideEd()

  WARNING(string.format("Reset Override Info %s", self.var_name))

  self.base_rtti_name_override = self.base_rtti_name;
  self.is_array_override = self.is_array;

  if self.is_array then -- blueprint.CompileCode
    -- 数组元素 '解封装'
    self.default_value_override = {}
    local baseRtti = self.rtti:GetElementType()
    for _, wrap in ipairs(self.default_value) do
      local value = BlueRtti.Dup(baseRtti, wrap.value) -- 复制
      local item = self:_CreateEmptyItem(); -- 重新封装
      item.value = value
      table.insert(self.default_value_override, item)
    end
  else
    self.default_value_override = BlueRtti.Dup(self.rtti, self.default_value);
  end

end

function BlueVariableInfo:IsPrivate()
  return self.is_private
end

function BlueVariableInfo:GetRtti()
  return self.rtti
end

function BlueVariableInfo:GetBaseRtti()
  return self.base_rtti
end

function BlueVariableInfo:GetBaseRttiName()
  return self.base_rtti_name
end

function BlueVariableInfo:IsArray()
  return self.is_array
end

function BlueVariableInfo:GetName()
  return self.name
end

function BlueVariableInfo:GetDisplayName()
  return self.var_name
end

function BlueVariableInfo:GetDefault()
  if self.default_value_override ~= nil then
    return self.default_value_override
  else
    return self.default_value
  end
end

function BlueVariableInfo:GetSpecData()
  local cls_spec_data = {} ;
  cls_spec_data[BD.RefVarName] = self.name ;
  return cls_spec_data
end

function BlueVariableInfo:_HideInPrefabEditor()
  if _KRATOSEDITOR then -- 编辑器debug可以去掉 全部return true
    local EditorSystem = require "window.editor.system.editsystem"
    if EditorSystem:GetPrefabMode() then -- 如果在编辑器 不管什么都去掉 (暂时不考虑嵌套)
      return true
    elseif not self._isPrefabEd then -- 其次在大场景 只有prefab实例 显示, 非prefab实例不显示
      return true
    end
  end
  return false
end



-- name    is ID name, not changeable;
-- var_name is DisplayName, user-changeable
BlueVariableInfo:MemberRegister("name", Types.StringType(BlueVariableInfo._GetName, BlueVariableInfo._SetName), "none"); -- 不会修改的名字 tag="none" 不会在反射面板显示
BlueVariableInfo:MemberRegister("var_name", Types.StringType(BlueVariableInfo._GetDispName, BlueVariableInfo._SetDispName)) -- 显示的名字
BlueVariableInfo:MemberRegister("base_rtti_name", Types.ComboType(BlueRtti.GetComboData, BlueVariableInfo._GetCombo, BlueVariableInfo._SetCombo));
BlueVariableInfo:MemberRegister("is_array", Types.BoolType(BlueVariableInfo._GetIsArray, BlueVariableInfo._SetIsArray) );
BlueVariableInfo:MemberRegister("is_private", Types.BaseType() );
BlueVariableInfo:MemberRegister("default_value");
--BlueVariableInfo:MemberRegister("is_const", Types.BaseType(), "none" );
--BlueVariableInfo:MemberRegister("is_final", Types.BaseType(), "none" );
--BlueVariableInfo:MemberRegister("is_ref", Types.BaseType(), "none" );
--BlueVariableInfo:MemberRegister("tips");

-- write and read only on prefab instance, typeinfo keep same with default_value
BlueVariableInfo:MemberRegister("default_value_override", nil, "none");

-- just readonly on prefab instance
BlueVariableInfo:MemberRegister("base_rtti_name_override", nil, "none");

-- just readonly on prefab instance
BlueVariableInfo:MemberRegister("is_array_override", nil, "none");

return BlueVariableInfo ;