local BluePinInfo = require "bluecore.bluepininfo"
local BlueNode = require "bluecore.bluenode"
local BD = require "bluecore.bluedefined"

local BlueDynamicNode = BlueNode:extend("BlueDynamicNode");

BlueDynamicNode:MemberRegister(BD.INPUTS_INFO);
BlueDynamicNode:MemberRegister(BD.OUTPUTS_INFO);
BlueDynamicNode:MemberRegister(BD.EXEC_INPUTS_INFO);
BlueDynamicNode:MemberRegister(BD.EXEC_OUTPUTS_INFO);

-- 用途:引脚数量不变 但是rtti/tips/name等会在编辑时修改 需要序列化保存引脚信息的节点类
-- 如果有数据引脚是Dynamic 目前全部数据引脚信息都会序列化出去
function BlueDynamicNode:new(...)

  -- BlueNode的基类构造
  -- 根据静态引脚信息,会生成 self.inputs/outputs/execInputs/execOutputs等保存连接信息和字面值
  BlueDynamicNode.super.new(self, ...)

  -- 序列化的pin info信息
  self[BD.INPUTS_INFO] = {}
  self[BD.OUTPUTS_INFO] = {}
  self[BD.EXEC_INPUTS_INFO] = {}
  self[BD.EXEC_OUTPUTS_INFO] = {}

end

-- 编辑器下创建Node后回调,反序列化过程会恢复infoTable所以不用执行
function BlueDynamicNode:_OnCreateWithEditor()

  -- 保留原来有的输出和输入执行引脚信息(可以为nil)
  local originExecInputsInfo = self.infoTable[BD.PIN_EXEC_INPUT]
  local originExecOutputsInfo = self.infoTable[BD.PIN_EXEC_OUTPUT]
  local originDataInputsInfo = self.infoTable[BD.PIN_DATA_INPUT]
  local originDataOutputsInfo = self.infoTable[BD.PIN_DATA_OUTPUT]

  -- override BlueNode's infoTable DynamicRegisterInput和反序列化不会修改基类的infoTable
  self.infoTable = {}
  self.infoTable[BD.PIN_DATA_INPUT] = {}
  self.infoTable[BD.PIN_DATA_OUTPUT] = {}
  self.infoTable[BD.PIN_EXEC_INPUT]  = {}
  self.infoTable[BD.PIN_EXEC_OUTPUT] = {}

  -- 把原来的注册的数据引脚生成为动态的数据引脚
  if originDataInputsInfo ~= nil then
    for _, info in pairs(originDataInputsInfo) do
      self:DynamicRegisterInput(info.argId, info.rtti, info.name, info.tips,
              info.default,  -- 动态和静态注册的default指向同一个实例
              info.delAble, info.groupName);
    end
  end
  if originDataOutputsInfo ~= nil then
    for _, info in pairs(originDataOutputsInfo) do
      self:DynamicRegisterOutput(info.argId, info.rtti, info.name, info.tips,
              nil,
              info.delAble, info.groupName);
    end
  end
  if originExecInputsInfo ~= nil then
    for _, info in pairs(originExecInputsInfo) do
      self:DynamicRegisterExecInput(info.argId, info.rtti, info.name, info.tips,
              nil,
              info.delAble, info.groupName);
    end
  end
  if originExecOutputsInfo ~= nil then
    for _, info in pairs(originExecOutputsInfo) do
      self:DynamicRegisterExecOutput(info.argId, info.rtti, info.name, info.tips,
              nil,
              info.delAble, info.groupName);
    end
  end

  -- 由于infoTable将会重建跟原来静态的一样
  -- 同时inputs/outputs/execInputs/execOutputs是通过pinType和argId索引
  -- 所以这里不用重建inputs/outputs/execInputs/execOutputs
  --self:Init(); -- 导致id浪费

end


function BlueDynamicNode:DynamicRegisterInput(argId, rtti, name, tips, default, delAble, groupName)

  if tips == nil then tips = "["..tostring(argId).."]" end
  if name == nil then name = "["..tostring(argId).."]" end

  local info = BluePinInfo(BD.PIN_DATA_INPUT, argId, rtti, name, tips, default, delAble, groupName);
  self.infoTable[BD.PIN_DATA_INPUT][argId] = info -- 提供给外部访问pin的info && BlueNode底层需要
  self[BD.INPUTS_INFO][argId] = info  -- 序列化

  return info ;  -- 返回info 用来新增pin

end


function BlueDynamicNode:DynamicRegisterOutput(argId, rtti,  name, tips, default, delAble, groupName)

  if tips == nil then tips = "out:"..tostring(argId)  end
  if name == nil then name = tostring(argId)  end

  local info =  BluePinInfo(BD.PIN_DATA_OUTPUT, argId, rtti,  name, tips, default, delAble, groupName);
  self.infoTable[BD.PIN_DATA_OUTPUT][argId] = info
  self[BD.OUTPUTS_INFO][argId] = info

  return info ;
end

function BlueDynamicNode:DynamicRegisterExecInput(argId, rtti,  name, tips, default, delAble, groupName)

  if tips == nil then tips = "execIn:"..tostring(argId)  end
  if name == nil then name = tostring(argId)  end

  local info =  BluePinInfo(BD.PIN_EXEC_INPUT, argId, rtti,  name, tips, default, delAble, groupName);
  self.infoTable[BD.PIN_EXEC_INPUT][argId] = info
  self[BD.EXEC_INPUTS_INFO][argId] = info

  return info ;
end

function BlueDynamicNode:DynamicRegisterExecOutput(argId, rtti,  name, tips, default, delAble, groupName)

  if tips == nil then tips = "execOut:"..tostring(argId)  end
  if name == nil then name = tostring(argId)  end

  local info =  BluePinInfo(BD.PIN_EXEC_OUTPUT, argId, rtti,  name, tips, default, delAble, groupName);
  self.infoTable[BD.PIN_EXEC_OUTPUT][argId] = info
  self[BD.EXEC_OUTPUTS_INFO][argId] = info

  return info ;
end

-- 反序列化从基类到派生类(类似构造函数)
function BlueDynamicNode:_OnDeserialize()

  -- 到这里已经完成了 BlueNode.inputs .outputs .execinputs .execoutputs
  -- 和 BlueDynamicNode.inputsInfo outputsInfo 的反序列化
  -- 根据反序列化信息,重建BlueNode需要的 infoTable 输入和输出引脚信息
  -- 对比编辑器交互式创建 少了 DynamicRegisterInput 的调用(通过反序列信息恢复)

  --local staticExecInputsInfo = self.infoTable[BD.PIN_EXEC_INPUT]
  --local staticExecOutputsInfo = self.infoTable[BD.PIN_EXEC_OUTPUT]

  -- override BlueNode's infoTable DynamicRegisterInput和反序列化不会修改基类的infoTable
  self.infoTable = {}
  self.infoTable[BD.PIN_DATA_INPUT] = {}
  self.infoTable[BD.PIN_DATA_OUTPUT] = {}
  self.infoTable[BD.PIN_EXEC_INPUT]  = {}
  self.infoTable[BD.PIN_EXEC_OUTPUT] = {} 

  for key, info in pairs(self[BD.INPUTS_INFO]) do
    self.infoTable[BD.PIN_DATA_INPUT][info.argId] = info
  end

  for key, info in pairs(self[BD.OUTPUTS_INFO]) do
    self.infoTable[BD.PIN_DATA_OUTPUT][info.argId] = info
  end

  for key, info in pairs(self[BD.EXEC_INPUTS_INFO]) do
    self.infoTable[BD.PIN_EXEC_INPUT][info.argId] = info
  end

  for key, info in pairs(self[BD.EXEC_OUTPUTS_INFO]) do
    self.infoTable[BD.PIN_EXEC_OUTPUT][info.argId] = info
  end

end


return BlueDynamicNode