require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local PhiltrumScale = {}

function PhiltrumScale:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function PhiltrumScale:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function PhiltrumScale:_InitParams(defined)
  beautydefined = defined
  --瘦脸参数
  self.coef = self.coef or 0
  self.coef_temp = 1;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_PHILTRUMCOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_PHILTRUMCOEF");
  apolloengine.ShaderEntity.PHILTRUMPOINTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"PHILTRUMPOINTS");
  apolloengine.ShaderEntity.PHILTRUMPARAMS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"PHILTRUMPARAMS");
end

function PhiltrumScale:SetParams(config, shaderParams)
  shaderParams.philtrumpointArray = mathfunction.vector4array();
  shaderParams.philtrumParaArray = mathfunction.vector2array();
  
  shaderParams.philtrumParaArray:PushBack(beautyutils._ArrayToVec2(config.philtrumparam[1]));
  shaderParams.philtrumParaArray:PushBack(beautyutils._ArrayToVec2(config.philtrumparam[1]));
  shaderParams.philtrumParaArray:PushBack(beautyutils._ArrayToVec2(config.philtrumparam[2]));
  shaderParams.philtrumParaArray:PushBack(beautyutils._ArrayToVec2(config.philtrumparam[2]));
  self:_InitLens(config)
end

function PhiltrumScale:_InitLens(config)
  self.lens = {}
  for i = 1, #beautydefined.PhiltrumKey do
    local index1 = beautydefined.PhiltrumKey[i][1]+1;
    local index2 = beautydefined.PhiltrumDir[i][2]+1;
    self.lens[#self.lens + 1] = Len(index1, index2, 0, config.philtrumparam[i][1], config.philtrumparam[i][2])
    
    index1 = beautydefined.PhiltrumKey[i][2]+1;
    index2 = beautydefined.PhiltrumDir[i][1]+1;
    self.lens[#self.lens + 1] = Len(index1, index2, 0, config.philtrumparam[i][1], config.philtrumparam[i][2])
  end
end

function PhiltrumScale:_InitQuad(quadnode, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_PHILTRUMCOEF, mathfunction.vector1(0.0));
  quadnode:SetParameter(apolloengine.ShaderEntity.PHILTRUMPARAMS, shaderParams.philtrumParaArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.PHILTRUMPOINTS, shaderParams.philtrumpointArray);
end

function PhiltrumScale:_DrawQuad(quadnode, keypoints, rate)
  local philtrumpointArray = mathfunction.vector4array();
  for i = 1, 2, 1 do
    local index1 = beautydefined.PhiltrumKey[i][1]+1;
    local index2 = beautydefined.PhiltrumDir[i][2]+1;
    philtrumpointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
        
    index1 = beautydefined.PhiltrumKey[i][2]+1;
    index2 = beautydefined.PhiltrumDir[i][1]+1;
    philtrumpointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
  end
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_PHILTRUMCOEF, mathfunction.vector1(self.coef * rate));
  quadnode:SetParameter(apolloengine.ShaderEntity.PHILTRUMPOINTS, philtrumpointArray);
end

function PhiltrumScale:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = self.coef * rate
  local tmpDis = {}
  for i = 1, #self.lens do
    local srcIdx = self.lens[i]:GetSrcIdx()
    local dstIdx = self.lens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.lens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)  
    local tmp = self.lens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx] = tmp
  end
  
  local left = beautydefined.PhiltrumKey[2][1]
  local right = beautydefined.PhiltrumKey[2][2]
  local mid = (left + right) / 2
  tmpDis[mid + 1] = {0, 0}
  tmpDis[mid + 1][1] = 0.5 * tmpDis[left + 1][1] + 0.5 * tmpDis[right + 1][1]
  tmpDis[mid + 1][2] = 0.5 * tmpDis[left + 1][2] + 0.5 * tmpDis[right + 1][2]
  
  for k, v in pairs(tmpDis) do
    keypointsDisplacement[k][1] = keypointsDisplacement[k][1] + v[1]
    keypointsDisplacement[k][2] = keypointsDisplacement[k][2] + v[2]
  end
  
  if #keypoints >= 240 then
    for i = 1, #beautydefined.AdvPhiltrumWarpMap do
      local src = beautydefined.AdvPhiltrumWarpMap[i][1] + 107
      local dst = beautydefined.AdvPhiltrumWarpMap[i][2] + 1
      keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.AdvPhiltrumWarpCoef[i]
      keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.AdvPhiltrumWarpCoef[i]
    end
  end
end

function PhiltrumScale:UpdateParaValue(quadnode, config, key_index, attribute, value)
  if (attribute == "Y") then
    config.philtrumparam[key_index - beautydefined.Philtrum_start + 1][1] = value;
    LOG("nose param " .. key_index - beautydefined.Philtrum_start + 1 .. " y set to: " .. value);
  end
  if (attribute == "R") then
    config.philtrumparam[key_index - beautydefined.Philtrum_start + 1][2] = value;
    LOG("nose param " .. key_index - beautydefined.Philtrum_start + 1 .. " R set to: " .. value);
  end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.PHILTRUMPARAMS, shaderParams.philtrumParaArray);
end

function PhiltrumScale:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0;
  else
    self.coef = 0.0;
  end
  LOG("Philtrum Scale coef set to: " .. self.coef);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_PHILTRUMCOEF,mathfunction.vector1(self.coef * rate));
end

function PhiltrumScale:GetParams(config)
  return beautyutils.Clone(config.philtrumparam);
end

function PhiltrumScale:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --LOG("Philtrum Scale coef: " .. self.coef)
end

function PhiltrumScale:Release()
  self.lens = {}
end
  
return PhiltrumScale