require "venusdebug"
local math = require "math"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local videodecet = require "videodecet"
local facecute = require "facecute"
local face = require "videodecet.faceinfo"
local apollonode = require "apolloutility.apollonode"
local renderqueue = require "apolloutility.renderqueue"
local utility = require "apolloutility.nodeutility"
local defined = require "apolloutility.defiend"
local venusjson = require "venusjson"
local faceshin = require "beauty.modules.faceshin"
local noseshin = require "beauty.modules.noseshin"
local mouthscale = require "beauty.modules.mouthscale"
local roundeye = require "beauty.modules.roundeye"
local beautyutils = require "beauty.utils"
local beautydefined = require "beauty.defined_likee"
require "utility"

local FaceliftOptimize = {};
function FaceliftOptimize:Initialize(camera)
  self.newframecallback = nil;
  self.inited = true;
  self.maxFaceNum = 3;
  self.rate = 1.0;

  self.isShow = true;
  self.facenode = {};
  self.camera = {};
  self.rt = {};
  self.tex = {};
  self.backnode = {};

  self.near = camera:GetNear();
  self.far = camera:GetFar();
  self.pos = camera:GetWorldPosition();
  self.lookat = self.pos + camera:GetForward();
  self.up = camera:GetUp();
  self.color = mathfunction.Color(1.0,0.0,0.0,0.0); --清屏颜色
  LOG("FaceliftOptimize:Initialize used!");

  self.resolution = apolloengine.Framework:GetResolution();
  self.aspect_ratio = self.resolution:y() / self.resolution:x();
  LOG("self.resolution: " .. self.resolution:x() .. " " .. self.resolution:y());
  
  self.config = venusjson.LaodJsonFile("docs:facelift/params_likee.json");
  
  if(self.config.FallOff == nil) then
    self.config.FallOff = 40;
  end

  local shaderParams = self:_InitParams()
  self.mainquadnode = self:_InitMainQuad();
  self.mainquadnode:SetSequence(camera:GetSequence());


  local gridX = 50
  local vs,is = utility.TesslateGivenPart(gridX, math.floor(gridX * self.aspect_ratio), false, true,{-0.25,0.25}, {-0.25,0.25});

  for i = 1,self.maxFaceNum,1 do
    self:_CreateCamera(camera:GetSequence()-i+defined.pre_facelift_camera_sequence, i, true);

    local quadnode = self:_InitQuad(vs, is, self.aspect_ratio, shaderParams)
    quadnode:SetSequence(camera:GetSequence()-i+defined.pre_facelift_camera_sequence);

    table.insert(self.facenode,quadnode);
  end

  camera:Activate();

  --local pr = apollonode.PointNode();
  --pr:CreateResource(
    --263,
    --mathfunction.vector4(1,0,0,1),
    --"comm:documents/material/pointrender.material");
   --pr:SetShow(false);
  --renderqueue:After(pr);
  --self.pointrender = pr

  LOG("Facelift Optimize Initialized")
end

function FaceliftOptimize:_Switch(lib, on)
  -- venus editor on/off facial effect's switch
  if lib == "faceshin" then
    faceshin:_Switch(on);
  elseif lib == "mouthscale" then
    mouthscale:_Switch(on);
  elseif lib == "noseshin" then
    noseshin:_Switch(on);
  elseif lib == "roundeye" then
    roundeye:_Switch(on);
  end
end

function FaceliftOptimize:_SetTempCoef(lib, coef)
	-- venus editor temporary params value, used when switched on
  if lib == "faceshin" then
    faceshin:_SetTempCoef(coef);
  elseif lib == "mouthscale" then
    mouthscale:_SetTempCoef(coef);
  elseif lib == "noseshin" then
    noseshin:_SetTempCoef(coef);
  elseif lib == "roundeye" then
    roundeye:_SetTempCoef(coef);
  end
end

function FaceliftOptimize:_SetFallOff(fall_off)
  -- ERROR("fall_off set to: " .. fall_off * 100);
  self.config.FallOff = fall_off * 100;
end

function FaceliftOptimize:_SetFaceFallOff1(fall_off)
  self.config.FaceFallOff[1] = fall_off * 100;
end

function FaceliftOptimize:_SetFaceFallOff2(fall_off)
  self.config.FaceFallOff[2] = fall_off * 100;
end

function FaceliftOptimize:_SetEyescaleFallOff(fall_off)
  self.config.EyescaleFallOff = fall_off * 100;
end

function FaceliftOptimize:_InitParams()
  apolloengine.ShaderEntity.UNIFORM_RATIOASPECT = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_RATIOASPECT");
  apolloengine.ShaderEntity.UNIFORM_ZOOMSCALE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_ZOOMSCALE");
  --apolloengine.ShaderEntity.POINT_COLOR = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"POINT_COLOR");
  -- init shader params
  faceshin:_InitParams(beautydefined)
  noseshin:_InitParams(beautydefined)
  mouthscale:_InitParams(beautydefined)
  roundeye:_InitParams(beautydefined)

  local shaderParams = {}
  shaderParams.zoomscale = mathfunction.Matrix33(0, 0, 0,
                                           0, 0, 0,
                                           0, 0, 0);

  faceshin:SetParams(self.config, shaderParams);
  noseshin:SetParams(self.config, shaderParams);
  mouthscale:SetParams(self.config, shaderParams);
  roundeye:SetParams(self.config, shaderParams);
  return shaderParams
end

function FaceliftOptimize:_InitMainQuad()
  local quadnode = apollonode.QuadNode();
  quadnode:SetShow(false);
  quadnode:CreateResource("docs:material/imageblit_queue_overlay.material",true,false);
  return quadnode
end

function FaceliftOptimize:_InitQuad(vs, is, aspect_ratio, shaderParams)
  local quadnode = apollonode.QuadNode();
  quadnode:CreateResourceSpecific("docs:material/facelift_likee.material", vs, is);
  quadnode:SetShow(false);

  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_RATIOASPECT, mathfunction.vector1(aspect_ratio));
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ZOOMSCALE, shaderParams.zoomscale);

  -- set shader params
  faceshin:_InitQuad(quadnode, shaderParams);
  noseshin:_InitQuad(quadnode, shaderParams);
  mouthscale:_InitQuad(quadnode, shaderParams);
  roundeye:_InitQuad(quadnode, shaderParams);
  return quadnode
end

function FaceliftOptimize:_InitBackQuad(flip)
  local quadnode = apollonode.QuadNode();
  quadnode:SetShow(false);
  quadnode:CreateResource(defined.blit_material_path,flip,false);
  return quadnode
end

function FaceliftOptimize:_CreateCamera(sequence, index, flip)
  local camera = apollonode.CameraNode();
  camera:SetName("FaceliftCamera"..index)
  camera:Activate();
  camera:CreateRealCameraProjection(self.near, self.far);
  camera:LookAt(self.pos, self.lookat, self.up);
  camera:SetSequence(sequence);
  camera:SetClearColor(self.color);
  local rt = apolloengine.RenderTargetEntity();
  rt:PushMetadata(--设置FBO格式
    apolloengine.RenderTargetMetadata(
      apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
      apolloengine.Framework:GetViewport(),
      apolloengine.Framework:GetResolution()));--分辨率
  local tex = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
  tex:PushMetadata(--创建纹理
    apolloengine.TextureRenderMetadata(
      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
      apolloengine.Framework:GetResolution()));

  rt:CreateResource();
  camera:AttachRenderTarget(rt);

  local quadnode = self:_InitBackQuad(flip)
  quadnode:SetSequence(sequence);

  table.insert(self.camera,camera);
  table.insert(self.rt,rt);
  table.insert(self.tex,tex);
  table.insert(self.backnode,quadnode);
end

function FaceliftOptimize:NewframeCallback()
  if self.newframecallback ~= nil then
    return self.newframecallback
  end
  self.newframecallback = function()

    --LOG("NewframeCallback used");
    local camera = renderqueue:GetCamera("FaceLift")
    local sequence = camera:GetSequence()
    self.mainquadnode:SetSequence(sequence)
    for i = 1, self.maxFaceNum do
      local itemSeq = sequence - i + defined.pre_facelift_camera_sequence
      self.facenode[i]:SetSequence(itemSeq)
      self.backnode[i]:SetSequence(itemSeq)
      self.camera[i]:SetSequence(itemSeq)
      self.facenode[i]:SetShow(false);
      self.backnode[i]:SetShow(false);
      self.camera[i]:Deactivate();
    end
    
    if self.isShow == false or self:_AllModuleOff() then
      self.mainquadnode:SetShow(false);
      return;
    end
    local faces = videodecet:GetFaces();
    if not faces or #faces<1 then
      --WARNING("[facelift]: No Render")
      self.mainquadnode:SetShow(false);
      return;
    end

    local faceNum = math.min(self.maxFaceNum, #faces);
    if faceNum < 1 then
       return;
    end
    for k= 1, faceNum, 1 do
      local faceinfo = faces[k];
      self.keypoints = beautyutils.Clone(faceinfo:GetKeypointArray());
      local extrapoints = beautyutils.Clone(faceinfo:GetExtraKeypointArray());
      --LOG("Extrapoints Num: " .. #extrapoints)
      local forehead = faceinfo:GetForeheadLandmark();
      self.rotate = faceinfo:GetRotation();
      self.extrapoints = {}
      if #extrapoints == 134 then
        for i = 1, #self.keypoints do
          self.extrapoints[i] = self.keypoints[i]
        end
        for i = 1, #extrapoints do
          self.extrapoints[i + 106] = extrapoints[i]
        end
      else
        self.extrapoints = extrapoints
      end
      
      local shift = 106
      if #self.extrapoints >= 240 then
        --LOG("[facelift]: using " .. #self.extrapoints)
        self.keypoints[85] = self.extrapoints[(shift + 71)];
        self.keypoints[86] = self.extrapoints[(shift + 74)];
        self.keypoints[87] = self.extrapoints[(shift + 77)];
        self.keypoints[88] = self.extrapoints[(shift + 79)];
        self.keypoints[89] = self.extrapoints[(shift + 81)];
        self.keypoints[90] = self.extrapoints[(shift + 84)];
        self.keypoints[91] = self.extrapoints[(shift + 87)];
        self.keypoints[92] = self.extrapoints[(shift + 131)];
        self.keypoints[93] = self.extrapoints[(shift + 129)];
        self.keypoints[94] = self.extrapoints[(shift + 127)];
        self.keypoints[95] = self.extrapoints[(shift + 125)];
        self.keypoints[96] = self.extrapoints[(shift + 123)];
        self.keypoints[97] = self.extrapoints[(shift + 88)];
        self.keypoints[98] = self.extrapoints[(shift + 93)];
        self.keypoints[99] = self.extrapoints[(shift + 96)];
        self.keypoints[100] = self.extrapoints[(shift + 99)];
        self.keypoints[101] = self.extrapoints[(shift + 104)];
        self.keypoints[102] = self.extrapoints[(shift + 115)];
        self.keypoints[103] = self.extrapoints[(shift + 112)];
        self.keypoints[104] = self.extrapoints[(shift + 109)];
        --LOG("USE ADV FACE LANDMARKS")
      end
      -- LOG("self.keypoints size: " .. #self.keypoints);
      -- LOG("extrapoints size: " .. #extrapoints);
      -- LOG("forehead size: " .. #forehead);

      for i = 1, #forehead do
        self.keypoints[i + 106] = forehead[i]
        self.extrapoints[i + 240] = forehead[i]
      end
      
      local warpedPoints = self:_DrawQuad(self.facenode[k], self.keypoints, self.rotate, true)
      --self.pointrender:SetShow(true);
      --self.pointrender:Update(warpedPoints);
      local warpedKeypoints = {}
      local warpedExtraKeypoints = {}
      local warpedForehead = {}
      if #warpedPoints >= 240 then
        if #warpedPoints >= 263 then
          for i = 241, 263 do
            warpedForehead[i - 240] = warpedPoints[i]
          end
        end
        for i = 107, 240 do
          warpedExtraKeypoints[i - 106] = warpedPoints[i]
        end
      elseif #warpedPoints >= 129 then
        for i = 107, 129 do
          warpedForehead[i - 106] = warpedPoints[i]
        end
      end
      for i = 1, 106 do
        warpedKeypoints[i] = warpedPoints[i]
      end
      
      if #warpedKeypoints >= 106 then
        faceinfo:UpdateKeypoints(warpedKeypoints);
      end
      if #warpedExtraKeypoints >= 134 then
        faceinfo:UpdateExtraKeypoints(warpedExtraKeypoints);
      end
      if #warpedForehead >= 23 then
        faceinfo:UpdateForeheadKeypoints(warpedForehead);
      end
    end
    videodecet:PushNewFaceLandMark()
    -- videodecet:PushNewFaceAdavanceLandmark()

    local prevTex = renderqueue:GetLinkedTexture("FaceLift")
    self.facenode[faceNum]:SetParameter(
      apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
      prevTex
    );
    self.backnode[faceNum]:SetParameter(
      apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
      prevTex
    );

    self.backnode[faceNum]:SetShow(true);
    self.camera[faceNum]:Activate();

    for i = faceNum-1,1,-1 do
      self.facenode[i]:SetParameter(apolloengine.ShaderEntity.TEXTURE_DIFFUSE,self.tex[i+1]);
      self.backnode[i]:SetParameter(apolloengine.ShaderEntity.TEXTURE_DIFFUSE,self.tex[i+1]);
      self.backnode[i]:SetShow(true);
      self.camera[i]:Activate();
    end

    self.mainquadnode:SetParameter(apolloengine.ShaderEntity.TEXTURE_DIFFUSE,self.tex[1]);
    self.mainquadnode:SetShow(true);
  end
  return self.newframecallback;
end

function FaceliftOptimize:_AllModuleOff()
  local res = true
  res = res and faceshin.coef == 0
  res = res and noseshin.coef == 0
  res = res and mouthscale.coef == 0
  res = res and roundeye.coef == 0
  return res
end

function FaceliftOptimize:_DrawQuad(quadnode, keypoints, picthrollyaw)
--  LOG("face lift DrawQuad used");

  --修正侧脸问题
  self.rate = (1.0 - math.abs(picthrollyaw[3]/self.config.FallOff));
  if(self.rate < 0) then
    self.rate = 0.0;
  end

  local bboxData = {min_x = 1, max_x = -1, min_y = 1, max_y = -1}

  faceshin:_Bbox(self.config, keypoints, bboxData)

  local paddingRatio = 0.1;
  local width = bboxData.max_x - bboxData.min_x;
  local height = bboxData.max_y - bboxData.min_y;
  local xymin = {bboxData.min_x - paddingRatio*width, bboxData.min_y - paddingRatio*height};
  local xymax = {bboxData.max_x + paddingRatio*width, bboxData.max_y + paddingRatio*height};

  local centerx = (xymin[1]+xymax[1])*0.5;
  local centery = (xymin[2]+xymax[2])*0.5;
  local zoomscalex = (xymax[1]-xymin[1]) * 2;
  local zoomscaley = (xymax[2]-xymin[2]) * 2;
  local zoomescale = mathfunction.Matrix33(zoomscalex, 0, centerx,
                                           0, zoomscaley, centery,
                                           0, 0, 0);
  local shiftscale = mathfunction.vector2(centerx,centery);

  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ZOOMSCALE,zoomescale);

  local findprint=false
  -- set shader params
  if findprint then LOG("faceshin") end
  if self.config.FaceFallOff then
    self.faceRate = {}
    self.faceRate[1] = (1.0 - math.abs(picthrollyaw[3]/self.config.FaceFallOff[1]));
    self.faceRate[2] = (1.0 - math.abs(picthrollyaw[3]/self.config.FaceFallOff[2]));
    if self.faceRate[1] < 0 then 
      self.faceRate[1] = 0 
    end
    if self.faceRate[2] < 0 then 
      self.faceRate[2] = 0 
    end
  else
    self.faceRate = {self.rate, self.rate}
  end
  faceshin:_DrawQuad(quadnode, keypoints, self.faceRate);
  if self.config.EyescaleFallOff then
    self.eyescaleRate = (1.0 - math.abs(picthrollyaw[3]/self.config.EyescaleFallOff));
    if self.eyescaleRate < 0 then 
      self.eyescaleRate = 0 
    end
  else
    self.eyescaleRate = self.rate
  end
  if findprint then LOG("noseshin") end
  noseshin:_DrawQuad(quadnode, keypoints, self.rate);
  if findprint then LOG("mouthscale") end
  mouthscale:_DrawQuad(quadnode, keypoints, self.rate);
  if findprint then LOG("roundeye") end
  roundeye:_DrawQuad(quadnode, keypoints, self.eyescaleRate, self.config);

  quadnode:SetShow(true);
  local warpedkeypoints = {}
  if self.extrapoints ~= nil and #self.extrapoints >= 240 then
    warpedkeypoints = self:_getWarpedKeyPoints(self.extrapoints)
  else
    warpedkeypoints = self:_getWarpedKeyPoints(keypoints)
  end
  return warpedkeypoints
end

function FaceliftOptimize:GetNewframeCallback()
    return self.newframecallback
end

function FaceliftOptimize:_getWarpedKeyPoints(keypoints)
  local keypointsDisplacement = {}
  for i = 1, #keypoints do
    keypointsDisplacement[i] = {0, 0}
  end

  faceshin:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, self.aspect_ratio, self.config, self.faceRate)
  noseshin:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, self.aspect_ratio, self.config, self.rate)
  mouthscale:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, self.aspect_ratio, self.config, self.rate)
  roundeye:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, self.aspect_ratio, self.config, self.eyescaleRate)

  local warpedkeypoints = {}
  for i = 1, #keypoints do
    --LOG("id "..i-1 ..": "..keypointsDisplacement[i][1]..", "..keypointsDisplacement[i][2])
    warpedkeypoints[i] = {keypoints[i][1] - keypointsDisplacement[i][1], keypoints[i][2] - keypointsDisplacement[i][2]}
    --LOG("id "..i-1 ..": "..keypoints[i][1]..", "..keypoints[i][2])
  end
  return warpedkeypoints
end

function FaceliftOptimize:setWarpLevelforType(typeid, coef, versionnum)
  if typeid == beautydefined.typeid.eyescale then
    roundeye:SetCoef(coef / 220)
  elseif typeid == beautydefined.typeid.faceshin then
    faceshin:SetCoef(coef / 140)
  elseif typeid == beautydefined.typeid.noseshin then
    noseshin:SetCoef(coef / 100)
  elseif typeid == beautydefined.typeid.mouthscale then
    mouthscale:SetCoef(coef / 50)
  end
end

function FaceliftOptimize:Disable()
  self.isShow = false;
end

function FaceliftOptimize:Enable()
 self.isShow = true;
end

function FaceliftOptimize:OnResizeView(w,h)
  self.aspect_ratio = h/w;
--  self.aspect_ratio = 1.0;
  if not self.facenode or #self.facenode < 1 or not self.inited then
    return;
  end
  for i = 1,self.maxFaceNum,1 do
    self.facenode[i]:SetParameter(apolloengine.ShaderEntity.UNIFORM_RATIOASPECT, mathfunction.vector1(self.aspect_ratio));
  end
end

function FaceliftOptimize:Release()
  for k, v in pairs(self.facenode) do
    v:Destroy();
  end

  for k, v in pairs(self.camera) do
    v:Destroy();
  end

  for k, v in pairs(self.backnode) do
    v:Destroy();
  end
  self.mainquadnode:Destroy();

  self.inited = nil;
  self.facenode = {};
  self.camera = {};
  self.rt = {};
  self.tex = {};
  self.backnode = {};
  self.mainquadnode = nil;
  self.newframecallback = nil;
  self.config = nil;
  faceshin:Release()
  noseshin:Release()
  mouthscale:Release()
  roundeye:Release()
end

function FaceliftOptimize:EditorInitialize(camera, aspect_ratio)
  self.inited = true;
  local shaderParams = self:_InitParams();
  self.aspect_ratio = aspect_ratio;

  local gridX = 50
  local vs,is = utility.TesslateGivenPart(gridX, math.floor(gridX * self.aspect_ratio), false, true,{-0.25,0.25}, {-0.25,0.25});
  
  self.face = self:_InitQuad(vs, is, aspect_ratio, shaderParams, apolloengine.RenderComponent.RM_TRIANGLES);
  self.face:SetSequence(camera:GetSequence())
  
  faceshin:SetCoef(0)
  mouthscale:SetCoef(0)
  noseshin:SetCoef(0)
  roundeye:SetCoef(0)
  return self.face
end

function FaceliftOptimize:EditorUpdate(keypoints, picthrollyaw, tex, resolution)
  local half_width = resolution:x() / 2;
  local half_height = resolution:y() / 2;
  self.aspect_ratio = half_height / half_width;
  self.face:SetParameter(apolloengine.ShaderEntity.UNIFORM_RATIOASPECT, mathfunction.vector1(self.aspect_ratio));

  if keypoints ~= nil then
    for i = 1,#keypoints,1 do
      keypoints[i][1] = (keypoints[i][1] - half_width) / half_width;
      keypoints[i][2] = (half_height - keypoints[i][2]) / half_height;
    end
  end

  if keypoints == nil then
    self.face:SetShow(false)
    return
  end
  
  self.keypoints = keypoints;
  self.keypoints_backup = beautyutils.Clone(keypoints);
  self.picthrollyaw = picthrollyaw;

  if self.face then
    self:_DrawQuad(self.face, keypoints, picthrollyaw, true)
    self.face:SetParameter(
      apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
      tex
    )
    self.face:SetShow(true)
  end
end

function FaceliftOptimize:EditorImageUpdate()
  self.keypoints = beautyutils.Clone(self.keypoints_backup);

  if self.face then
    self:_DrawQuad(self.face, self.keypoints, self.picthrollyaw, true)
  end
end

function FaceliftOptimize:UpdateParaValue(key_index, attribute, value)
  if(self.inited == nil) then
--    LOG("facelift not ready, skip update param");
    return;
  end

  if key_index == 0 then
    mouthscale:UpdateParaValue(self.face, self.config, key_index, attribute, value)
  elseif key_index < beautydefined.nose_start then
    faceshin:UpdateParaValue(self.face, self.config, key_index, attribute, value)
  elseif key_index < beautydefined.chin_start then
    noseshin:UpdateParaValue(self.face, self.config, key_index, attribute, value)
  elseif key_index < beautydefined.HairLine_start then
    roundeye:UpdateParaValue(self.face, self.config, key_index, attribute, value)
  end
----------------------------------------------------
end

function FaceliftOptimize:UpdateCoeff(value)
  faceshin:UpdateCoeff(self.face, value, self.rate)
  noseshin:UpdateCoeff(self.face, value, self.rate)
  mouthscale:UpdateCoeff(self.face, value, self.rate)
  roundeye:UpdateCoeff(self.face, value, self.rate, self.config)

  local res = venusjson.SaveJsonObject(self.config, "temp_params.json")
  LOG("Finished saving params as json file.")
end

function FaceliftOptimize:GetParams()
  self.config = venusjson.LaodJsonFile("docs:facelift/params_likee.json");
  if type(self.config.LipParam) ~= "table" then
    self.config.LipParam = {0, self.config.LipParam, self.config.LipParam, 1, 1};
  end
  if(self.config.FallOff == nil) then
    self.config.FallOff = 40;
  end
  local AllParam = {}
  AllParam.FallOff = self.config.FallOff / 100
  AllParam.FaceFallOff = {self.config.FaceFallOff[1] / 100, self.config.FaceFallOff[2] / 100}
  AllParam.EyescaleFallOff = self.config.EyescaleFallOff / 100
  
  AllParam.mouthparams = mouthscale:GetParams(self.config);
  AllParam.noseparams = noseshin:GetParams(self.config)
  AllParam.FaceParam, AllParam.FaceParamRadius = faceshin:GetParams(self.config)
  AllParam.roundeyeParams = roundeye:GetParams(self.config)

  return AllParam
end

return FaceliftOptimize;
