require "utility"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local venuscore = require "venuscore"

local shinelight = {}

function shinelight:Initialize(host, size)
  self.Queue = 100;
  self.BlurCount = 3;
  self.threshold = mathfunction.vector1(0.9);--高光亮度阈值
  self.rayColor = mathfunction.vector3(255.0, 255.0, 255.0);--光束颜色
  self.rayLength = 10;--光束长度
  self.rayLight = 0.5;--光束强度
  self.center = mathfunction.vector2(0.61, -0.55);--光束发射中心
  self.rayColorRatio = mathfunction.vector1(0.6);--光束颜色比例，0光束的颜色是选择的颜色，1就是高光区域原本颜色
  self.BlendMode  = mathfunction.vector1(2);

  self.POST_EFFECT_ALPHA = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "POST_EFFECT_ALPHA");

  self:_InitGreyPass(host, size);
  self:_InitBlurPass(host, size);
  self:_InitVolumnPass(host, size);
  self:_InitBlendPass(host, size);

  return self.Queue;
end

function shinelight:_InitGreyPass(host, size)
  self.RGB_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "RGB_TEXTURE");
  self.THRESHOLD = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "THRESHOLD");

  self.GreyMat = host:CreateMaterial("comm:documents/shaders/posteffect/shinelight.material");
  self.GreyRO = host:CreateRenderObject();
  self.GreyRT = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_A, size);

  self.GreyMat:SetParameter(self.POST_EFFECT_ALPHA, mathfunction.vector1(1.0));
  
  host:RegisterScriptParameter(self, "threshold");
end

function shinelight:_InitBlurPass(host, size)	
  self.SHINE_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "SHINE_TEXTURE");   
  self.BLUR_RADIUS = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "BLUR_RADIUS");  
  self.UNIFORM_OFFSET = apolloengine.IMaterialSystem:NewParameterSlot(
  	apolloengine.ShaderEntity.UNIFORM,
    "UNIFORM_OFFSET");

  self.blurXMat = host:CreateMaterial("comm:documents/shaders/posteffect/shinelightblur.material");
  self.blurXRO = host:CreateRenderObject();   
  self.blurYMat = host:CreateMaterial("comm:documents/shaders/posteffect/shinelightblur.material");
  self.blurYRO = host:CreateRenderObject();  
  self.blurXRT = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_B, size);
  self.blurYRT = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_C, size);

  self.blurXMat:SetParameter(self.BLUR_RADIUS, mathfunction.vector1(3.0));
  self.blurXMat:SetParameter(self.UNIFORM_OFFSET, mathfunction.vector2(1.0 / size:x(), 0.0));
  
  self.blurYMat:SetParameter(self.BLUR_RADIUS, mathfunction.vector1(3.0));
  self.blurYMat:SetParameter(self.UNIFORM_OFFSET, mathfunction.vector2(0.0, 1.0 / size:y()));
end

function shinelight:_InitVolumnPass(host, size)     
  self.BRIGHT_TEXTURE = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "BRIGHT_TEXTURE");
  self.UNIFORM_LIGHT_COLOR = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "UNIFORM_LIGHT_COLOR");
  self.CENTER = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "CENTER");
  self.SAMPLES = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "SAMPLES"); 
  self.STEP = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "STEP"); 
  self.RATIO = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "RATIO");

  self.VolumnMat = host:CreateMaterial( "comm:documents/shaders/posteffect/shinevolumnlight.material");
  self.VolumnRO = host:CreateRenderObject();
  
  self.VolumnRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_D, size );
  self.VolumnMat:SetParameter(self.SAMPLES, mathfunction.vector1(100));
  self.VolumnMat:SetParameter(self.POST_EFFECT_ALPHA, mathfunction.vector1(1.0));

  host:RegisterScriptParameter(self, "rayColor");
  host:RegisterScriptParameter(self, "rayLength");
  host:RegisterScriptParameter(self, "center");
  host:RegisterScriptParameter(self, "rayLight");
  host:RegisterScriptParameter(self, "rayColorRatio");
end

function shinelight:_InitBlendPass(host, size)
  self._BlendMode = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_BlendMode");
  self.BlendOpacity = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_BlendOpacity");
  self._MainTex = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_MainTex");   
  self._BlendTex = apolloengine.IMaterialSystem:NewParameterSlot(
    apolloengine.ShaderEntity.UNIFORM,
    "_BlendTex");
  self.BlendMat = host:CreateMaterial("comm:documents/shaders/posteffect/blendmode.material");
  self.BlendRO = host:CreateRenderObject();
  
  self.BlendRT = host:CreateRenderTarget( apolloengine.RenderTargetEntity.ST_SWAP_E, size );
  self.BlendMat:SetParameter(self.BlendOpacity, mathfunction.vector1(2.0));
  self.BlendMat:SetParameter(self.POST_EFFECT_ALPHA, mathfunction.vector1(1.0));

  host:RegisterScriptParameter(self, "BlendMode");
end

function shinelight:_DrawGreyPass(context, Original, Scene, Output)
  context:BeginRenderPass(self.GreyRT, apolloengine.RenderTargetEntity.CF_COLOR);
  self.GreyMat:SetParameter(self.RGB_TEXTURE, Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  self.GreyMat:SetParameter(self.THRESHOLD, self.threshold);
  context:Draw(self.GreyRO, self.GreyMat);
  context:EndRenderPass();
end

function shinelight:_DrawBlurPass(context, Original, Scene, Output)
  local last = self.GreyRT;
  for i = 1, self.BlurCount do
	context:BeginRenderPass(self.blurXRT, apolloengine.RenderTargetEntity.CF_COLOR);
	self.blurXMat:SetParameter(self.SHINE_TEXTURE, last:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0)); 
	context:Draw(self.blurXRO, self.blurXMat);
	context:EndRenderPass();

	context:BeginRenderPass(self.blurYRT, apolloengine.RenderTargetEntity.CF_COLOR);
	self.blurYMat:SetParameter(self.SHINE_TEXTURE, self.blurXRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
	context:Draw(self.blurYRO, self.blurYMat);
	context:EndRenderPass();

    last = self.blurYRT;
  end
end

function shinelight:_DrawVolumnPass(context, Original, Scene, Output)
  context:BeginRenderPass(self.VolumnRT, apolloengine.RenderTargetEntity.CF_COLOR);

  self.VolumnMat:SetParameter(self.BRIGHT_TEXTURE, self.blurYRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  local color = (self.rayColor / 255) * self.rayLight;
  self.VolumnMat:SetParameter(self.UNIFORM_LIGHT_COLOR, color);
  self.VolumnMat:SetParameter(self.CENTER, self.center);
  local step = mathfunction.vector1(0.005) * self.rayLength;
  self.VolumnMat:SetParameter(self.STEP, step);
  local ratio = self.rayColorRatio;
  self.VolumnMat:SetParameter(self.RATIO, ratio);
  context:Draw(self.VolumnRO, self.VolumnMat);
  context:EndRenderPass();
end

function shinelight:_DrawBlendPass(context, Original, Scene, Output)
  context:BeginRenderPass(Output, apolloengine.RenderTargetEntity.CF_COLOR);
  self.BlendMat:SetParameter(self._MainTex, Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  self.BlendMat:SetParameter(self._BlendTex, self.VolumnRT:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0));
  self.BlendMat:SetParameter(self._BlendMode, self.BlendMode);
  context:Draw(self.BlendRO, self.BlendMat);
  context:EndRenderPass();
end

function shinelight:Resizeview(size) 
  self.blurXMat:SetParameter( self.UNIFORM_OFFSET, mathfunction.vector2( 1.0 / size:x(), 0 ) );
  self.blurYMat:SetParameter( self.UNIFORM_OFFSET, mathfunction.vector2( 0, 1.0 / size:y() ) );
end

function shinelight:Process(context, Original, Scene, Output)
  self:_DrawGreyPass(context, Original, Scene, Output);
  self:_DrawBlurPass(context, Original, Scene, Output);
  self:_DrawVolumnPass(context, Original, Scene, Output);
  self:_DrawBlendPass(context, Original, Scene, Output);
end

return shinelight;  

