
#pragma once

SamplerState TEXTURE_DIFFUSE_Sampler;
float FXAA_RELATIVE_LUMA_THRESHOLD;
float FXAA_ABSOLUTE_LUMA_THRESHOLD;
#define FXAA_MAX_EAGE_SEARCH_SAMPLE_COUNT 12
static float edgeSearchSteps[FXAA_MAX_EAGE_SEARCH_SAMPLE_COUNT] = {1,1,1,1,1,1.5,2,2,2,2,4,8};

struct FXAACrossData{
    float4 M;
    float4 N;
    float4 S;
    float4 W;
    float4 E;
};

struct FXAACornerData{
    float4 NW;
    float4 NE;
    float4 SW;
    float4 SE;
};

struct FXAAEdge{
    float2 dir;
    float2 normal;
    bool isHorz;
    float lumaEdge; //往normal方向偏移0.5个像素的亮度
    float4 oppRGBL;
};

inline float rgb2luma(float3 color){
    return dot(color,float3(0.299,0.587,0.114));
}

inline float4 SampleLinear(Texture2D tex,float2 uv){
    return tex.Sample(TEXTURE_DIFFUSE_Sampler,uv);
}


inline float4 SampleRGBLumaLinear(Texture2D tex,float2 uv){
    float3 color = SampleLinear(tex,uv).rgb;
    return float4(color,rgb2luma(color));
}


///采集上下左右4个像素 + 中心像素
inline FXAACrossData SampleCross(Texture2D tex,float2 uv,float4 offset){
    FXAACrossData crossData;
    crossData.M = SampleRGBLumaLinear(tex,uv);
    crossData.S = SampleRGBLumaLinear(tex,uv + float2(0,-offset.y));
    crossData.N = SampleRGBLumaLinear(tex,uv + float2(0,offset.y));
    crossData.W = SampleRGBLumaLinear(tex,uv + float2(-offset.x,0));
    crossData.E = SampleRGBLumaLinear(tex,uv + float2(offset.x,0));
    return crossData;
}

inline float4 CalculateContrast(in FXAACrossData cross){
    float lumaMin = min(min(min(cross.N.a,cross.S.a),min(cross.W.a,cross.E.a)),cross.M.a);
    float lumaMax = max(max(max(cross.N.a,cross.S.a),max(cross.W.a,cross.E.a)),cross.M.a);
    float lumaContrast = lumaMax - lumaMin;
    return float4(lumaContrast,lumaMin,lumaMax,0);
}


//offset由(x,y,-x,-y)组成
inline FXAACornerData SampleCorners(Texture2D tex,float2 uv,float4 offset){
    FXAACornerData cornerData;
    float3 rgbNW = SampleLinear(tex,uv + offset.zy).rgb;
    float3 rgbSW = SampleLinear(tex,uv + offset.zw).rgb;
    float3 rgbNE = SampleLinear(tex,uv + offset.xy).rgb;
    float3 rgbSE = SampleLinear(tex,uv + offset.xw).rgb;

    cornerData.NW = float4(rgbNW,rgb2luma(rgbNW));
    cornerData.NE = float4(rgbNE,rgb2luma(rgbNE));
    cornerData.SW = float4(rgbSW,rgb2luma(rgbSW));
    cornerData.SE = float4(rgbSE,rgb2luma(rgbSE));
    return cornerData;
}

inline FXAAEdge GetEdge(in FXAACrossData cross, in FXAACornerData corner)
{

    FXAAEdge edge;

    float lumaM = cross.M.a;
    float lumaN = cross.N.a;
    float lumaS = cross.S.a;
    float lumaW = cross.W.a;
    float lumaE = cross.E.a;

    float lumaGradS = lumaS - lumaM;
    float lumaGradN = lumaN - lumaM;
    float lumaGradW = lumaW - lumaM;
    float lumaGradE = lumaE - lumaM;

    float lumaGradH = abs(lumaGradW + lumaGradE);
    float lumaGradV = abs(lumaGradS + lumaGradN);


    float lumaNW = corner.NW.a;
    float lumaNE = corner.NE.a;
    float lumaSW = corner.SW.a;
    float lumaSE = corner.SE.a;

    lumaGradH = abs(lumaNW + lumaNE - 2 * lumaN)
    + 2 * lumaGradH
    + abs(lumaSW + lumaSE - 2 * lumaS);

    lumaGradV = abs(lumaNW + lumaSW - 2 * lumaW)
    + 2 * lumaGradV
    + abs(lumaNE + lumaSE - 2 * lumaE);


    bool isHorz = lumaGradV >= lumaGradH;
    edge.isHorz = isHorz;
    if(isHorz){
        float s = sign(abs(lumaGradN) - abs(lumaGradS));
        edge.dir = float2(1,0);
        edge.normal = float2(0,s);
        edge.lumaEdge = s > 0? (lumaN + lumaM) * 0.5:(lumaS + lumaM) * 0.5;
        edge.oppRGBL = s > 0 ? cross.N:cross.S;
    }else{
        float s = sign(abs(lumaGradE) - abs(lumaGradW));
        edge.dir = float2(0,1);
        edge.normal = float2(s,0);
        edge.lumaEdge = s > 0 ? (lumaE + lumaM) * 0.5:(lumaW + lumaM) * 0.5;
        edge.oppRGBL = s > 0 ? cross.E:cross.W;
    }
    return edge;
}

inline float GetLumaGradient(FXAAEdge edge,FXAACrossData crossData){
    float luma1,luma2;
    float lumaM = crossData.M.a;
    if(edge.isHorz){
        luma1 = crossData.S.a;
        luma2 = crossData.N.a;
    }else{
        luma1 = crossData.W.a;
        luma2 = crossData.E.a;
    }
    return max(abs(lumaM - luma1),abs(lumaM-luma2));
}


inline float GetEdgeBlend(Texture2D tex,float2 uv,FXAAEdge edge,FXAACrossData crossData){
    float2 invScreenSize = CAMERA_RESOLUTION_INV;

    float lumaM = crossData.M.a;
    float lumaGrad = GetLumaGradient(edge,crossData);
    float lumaGradScaled = lumaGrad * 0.25;
    uv += edge.normal * 0.5 * invScreenSize;

    float2 dir = edge.dir;

    float lumaStart = edge.lumaEdge;

    float4 rgblP,rgblN;

    float2 posP = float2(0,0) ;
    float2 posN = float2(0,0) ;
    bool endP = false;
    bool endN = false;

    for(uint i = 0; i < FXAA_MAX_EAGE_SEARCH_SAMPLE_COUNT; i ++){
        float step = edgeSearchSteps[i];
        if(!endP){
            posP += step * dir;
            rgblP = SampleRGBLumaLinear(tex,uv + posP * invScreenSize);
            endP = abs(rgblP.a - lumaStart) > lumaGradScaled;
        }
        if(!endN){
            posN -= step * dir;
            rgblN = SampleRGBLumaLinear(tex,uv + posN * invScreenSize);
            endN = abs(rgblN.a - lumaStart) > lumaGradScaled;
        }
        if(endP && endN){
            break;
        }
    }
    posP = abs(posP);
    posN = abs(posN);
    float dstP = max(posP.x,posP.y);
    float dstN = max(posN.x,posN.y);
    float dst,lumaEnd;
    if(dstP > dstN){
        dst = dstN;
        lumaEnd = rgblN.a;
    }else{
        dst = dstP;
        lumaEnd = rgblP.a;
    }
    if((lumaM - lumaStart) * (lumaEnd - lumaStart) > 0){
        return 0;
    }
    //blend的范围为0~0.5
    return 0.5 - dst/(dstP + dstN);
}


float4 FXAA(Texture2D tex,float2 uv)
{
    float2 invTextureSize = CAMERA_RESOLUTION_INV; // x = 1/screenWidth, y = 1/screenHeight
    float4 offset = float4(1,1,-1,-1) * invTextureSize.xyxy;
    FXAACrossData cross = SampleCross(tex,uv,offset);
    //计算对比度
    float lumaMinNS = min(cross.N.a,cross.S.a);
    float lumaMinWE = min(cross.W.a,cross.E.a);
    float lumaMin = min(cross.M.a,min(lumaMinNS,lumaMinWE));
    float lumaMaxNS = max(cross.N.a,cross.S.a);
    float lumaMaxWE = max(cross.W.a,cross.E.a);
    float lumaMax = max(cross.M.a,max(lumaMaxNS,lumaMaxWE));
    float lumaContrast = lumaMax - lumaMin;

    if(lumaContrast > max(lumaMax * FXAA_RELATIVE_LUMA_THRESHOLD,FXAA_ABSOLUTE_LUMA_THRESHOLD))
    {
        FXAACornerData cornerData = SampleCorners(tex,uv,offset);
        FXAAEdge edge = GetEdge(cross,cornerData);

        float blend = GetEdgeBlend(tex,uv,edge,cross);
        return lerp(cross.M,edge.oppRGBL,blend);
    }
    else
    {
        return float4(cross.M.rgb,1);
    }
}


#define QUALITY(q) ((q) < 5 ? 1.0 : ((q) > 5 ? ((q) < 10 ? 2.0 : ((q) < 11 ? 4.0 : 8.0)) : 1.5))
#define ITERATIONS 12
#define SUBPIXEL_QUALITY 0.75

float4 FXAA_PRO(Texture2D tex, float2 uv)
{
    float4 colorCenter = tex.Sample(TEXTURE_DIFFUSE_Sampler,uv);

    float2 inverseScreenSize = CAMERA_RESOLUTION_INV;
    // Luma at the current fragment
    float lumaCenter = rgb2luma(colorCenter.rgb);

    // Luma at the four direct neighbours of the current fragment.
    float lumaDown 	= rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(0,-1) * inverseScreenSize).rgb);
    float lumaUp 	= rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(0,1) * inverseScreenSize).rgb);
    float lumaLeft 	= rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(-1,0) * inverseScreenSize).rgb);
    float lumaRight = rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(1,0) * inverseScreenSize).rgb);

    // Find the maximum and minimum luma around the current fragment.
    float lumaMin = min(lumaCenter, min(min(lumaDown, lumaUp), min(lumaLeft, lumaRight)));
    float lumaMax = max(lumaCenter, max(max(lumaDown, lumaUp), max(lumaLeft, lumaRight)));

    // Compute the delta.
    float lumaRange = lumaMax - lumaMin;

    // If the luma variation is lower that a threshold (or if we are in a really dark area), we are not on an edge, don't perform any AA.
    if(lumaRange < max(FXAA_ABSOLUTE_LUMA_THRESHOLD, lumaMax * FXAA_RELATIVE_LUMA_THRESHOLD)){
        return colorCenter;
    }

    // Query the 4 remaining corners lumas.
    float lumaDownLeft 	= rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(-1,-1) * inverseScreenSize).rgb);
    float lumaUpRight 	= rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(1,1) * inverseScreenSize).rgb);
    float lumaUpLeft 	= rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(-1,1) * inverseScreenSize).rgb);
    float lumaDownRight = rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler, uv + float2(1,-1) * inverseScreenSize).rgb);

    // Combine the four edges lumas (using intermediary variables for future computations with the same values).
    float lumaDownUp = lumaDown + lumaUp;
    float lumaLeftRight = lumaLeft + lumaRight;

    // Same for corners
    float lumaLeftCorners = lumaDownLeft + lumaUpLeft;
    float lumaDownCorners = lumaDownLeft + lumaDownRight;
    float lumaRightCorners = lumaDownRight + lumaUpRight;
    float lumaUpCorners = lumaUpRight + lumaUpLeft;

    // Compute an estimation of the gradient along the horizontal and vertical axis.
    float edgeHorizontal =	abs(-2.0 * lumaLeft + lumaLeftCorners)	+ abs(-2.0 * lumaCenter + lumaDownUp ) * 2.0	+ abs(-2.0 * lumaRight + lumaRightCorners);
    float edgeVertical =	abs(-2.0 * lumaUp + lumaUpCorners)		+ abs(-2.0 * lumaCenter + lumaLeftRight) * 2.0	+ abs(-2.0 * lumaDown + lumaDownCorners);

    // Is the local edge horizontal or vertical ?
    bool isHorizontal = (edgeHorizontal >= edgeVertical);

    // Choose the step size (one pixel) accordingly.

    float stepLength = isHorizontal ? inverseScreenSize.y : inverseScreenSize.x;

    // Select the two neighboring texels lumas in the opposite direction to the local edge.
    float luma1 = isHorizontal ? lumaDown : lumaLeft;
    float luma2 = isHorizontal ? lumaUp : lumaRight;
    // Compute gradients in this direction.
    float gradient1 = luma1 - lumaCenter;
    float gradient2 = luma2 - lumaCenter;

    // Which direction is the steepest ?
    bool is1Steepest = abs(gradient1) >= abs(gradient2);

    // Gradient in the corresponding direction, normalized.
    float gradientScaled = 0.25*max(abs(gradient1),abs(gradient2));

    // Average luma in the correct direction.
    float lumaLocalAverage = 0.0;
    if(is1Steepest){
        // Switch the direction
        stepLength = - stepLength;
        lumaLocalAverage = 0.5*(luma1 + lumaCenter);
    } else {
        lumaLocalAverage = 0.5*(luma2 + lumaCenter);
    }

    // Shift UV in the correct direction by half a pixel.
    float2 currentUv = uv;
    if(isHorizontal){
        currentUv.y += stepLength * 0.5;
    } else {
        currentUv.x += stepLength * 0.5;
    }

    // Compute offset (for each iteration step) in the right direction.
    float2 offset = isHorizontal ? float2(inverseScreenSize.x,0.0) : float2(0.0,inverseScreenSize.y);
    // Compute UVs to explore on each side of the edge, orthogonally. The QUALITY allows us to step faster.
    float2 uv1 = currentUv - offset * QUALITY(0);
    float2 uv2 = currentUv + offset * QUALITY(0);

    // Read the lumas at both current extremities of the exploration segment, and compute the delta wrt to the local average luma.
    float lumaEnd1 = rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler,uv1).rgb);
    float lumaEnd2 = rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler,uv2).rgb);
    lumaEnd1 -= lumaLocalAverage;
    lumaEnd2 -= lumaLocalAverage;

    // If the luma deltas at the current extremities is larger than the local gradient, we have reached the side of the edge.
    bool reached1 = abs(lumaEnd1) >= gradientScaled;
    bool reached2 = abs(lumaEnd2) >= gradientScaled;
    bool reachedBoth = reached1 && reached2;

    // If the side is not reached, we continue to explore in this direction.
    if(!reached1){
        uv1 -= offset * QUALITY(1);
    }
    if(!reached2){
        uv2 += offset * QUALITY(1);
    }

    // If both sides have not been reached, continue to explore.
    if(!reachedBoth){

        for(int i = 2; i < ITERATIONS; i++){
            // If needed, read luma in 1st direction, compute delta.
            if(!reached1){
                lumaEnd1 = rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler,uv1).rgb);
                lumaEnd1 = lumaEnd1 - lumaLocalAverage;
            }
            // If needed, read luma in opposite direction, compute delta.
            if(!reached2){
                lumaEnd2 = rgb2luma(tex.Sample(TEXTURE_DIFFUSE_Sampler,uv2).rgb);
                lumaEnd2 = lumaEnd2 - lumaLocalAverage;
            }
            // If the luma deltas at the current extremities is larger than the local gradient, we have reached the side of the edge.
            reached1 = abs(lumaEnd1) >= gradientScaled;
            reached2 = abs(lumaEnd2) >= gradientScaled;
            reachedBoth = reached1 && reached2;

            // If the side is not reached, we continue to explore in this direction, with a variable quality.
            if(!reached1){
                uv1 -= offset * QUALITY(i);
            }
            if(!reached2){
                uv2 += offset * QUALITY(i);
            }

            // If both sides have been reached, stop the exploration.
            if(reachedBoth){ break;}
        }

    }

    // Compute the distances to each side edge of the edge (!).
    float distance1 = isHorizontal ? (uv.x - uv1.x) : (uv.y - uv1.y);
    float distance2 = isHorizontal ? (uv2.x - uv.x) : (uv2.y - uv.y);

    // In which direction is the side of the edge closer ?
    bool isDirection1 = distance1 < distance2;
    float distanceFinal = min(distance1, distance2);

    // Thickness of the edge.
    float edgeThickness = (distance1 + distance2);

    // Is the luma at center smaller than the local average ?
    bool isLumaCenterSmaller = lumaCenter < lumaLocalAverage;

    // If the luma at center is smaller than at its neighbour, the delta luma at each end should be positive (same variation).
    bool correctVariation1 = (lumaEnd1 < 0.0) != isLumaCenterSmaller;
    bool correctVariation2 = (lumaEnd2 < 0.0) != isLumaCenterSmaller;

    // Only keep the result in the direction of the closer side of the edge.
    bool correctVariation = isDirection1 ? correctVariation1 : correctVariation2;

    // UV offset: read in the direction of the closest side of the edge.
    float pixelOffset = - distanceFinal / edgeThickness + 0.5;

    // If the luma variation is incorrect, do not offset.
    float finalOffset = correctVariation ? pixelOffset : 0.0;

    // Sub-pixel shifting
    // Full weighted average of the luma over the 3x3 neighborhood.
    float lumaAverage = (1.0/12.0) * (2.0 * (lumaDownUp + lumaLeftRight) + lumaLeftCorners + lumaRightCorners);
    // Ratio of the delta between the global average and the center luma, over the luma range in the 3x3 neighborhood.
    float subPixelOffset1 = clamp(abs(lumaAverage - lumaCenter)/lumaRange,0.0,1.0);
    float subPixelOffset2 = (-2.0 * subPixelOffset1 + 3.0) * subPixelOffset1 * subPixelOffset1;
    // Compute a sub-pixel offset based on this delta.
    float subPixelOffsetFinal = subPixelOffset2 * subPixelOffset2 * SUBPIXEL_QUALITY;

    // Pick the biggest of the two offsets.
    finalOffset = max(finalOffset,subPixelOffsetFinal);

    // Compute the final UV coordinates.
    float2 finalUv = uv;
    if(isHorizontal){
        finalUv.y += finalOffset * stepLength;
    } else {
        finalUv.x += finalOffset * stepLength;
    }

    // Read the color at the new UV coordinates, and use it.
    float4 finalColor = tex.Sample(TEXTURE_DIFFUSE_Sampler,finalUv);
    return finalColor;
}
