CGPROGRAM

#include "hair_common.inc"
#include "common.inc"

//--------------------------------------------------------------------------------------
//
//  Uses Verlet integration to calculate the new position for the current time step
//
//--------------------------------------------------------------------------------------
float3 Integrate(float3 curPosition, float3 oldPosition, float3 initialPos, float dampingCoeff = 1.0f)
{
	float3 force = g_GravityMagnitude * float3(0, -1.0, 0);
	// float decay = exp(-g_TimeStep/decayTime)
	float decay = exp(-dampingCoeff * g_TimeStep * 60.0f); 
	//return curPosition  + force * g_TimeStep * g_TimeStep + decay * (curPosition - oldPosition); 
	return curPosition  + force * g_TimeStep * g_TimeStep + (1-dampingCoeff) * (curPosition - oldPosition); 
}



//--------------------------------------------------------------------------------------
//
//  Updates the  hair vertex positions based on the physics simulation
//
//--------------------------------------------------------------------------------------
void UpdateFinalVertexPositions(float4 oldPosition, float4 newPosition, int globalVertexIndex)
{
	g_HairVertexPositionsPrevPrev[globalVertexIndex] = g_HairVertexPositionsPrev[globalVertexIndex];
    g_HairVertexPositionsPrev[globalVertexIndex] = oldPosition;
	g_HairVertexPositions[globalVertexIndex] = newPosition;
}



#pragma compute Simulate

[numthreads(THREAD_GROUP_SIZE, 1, 1)]
void Simulate(uint GIndex : SV_GroupIndex,
	uint3 GId : SV_GroupID,
	uint3 DTid : SV_DispatchThreadID)
{

	uint globalStrandIndex, localStrandIndex, globalVertexIndex, localVertexIndex, numVerticesInTheStrand, indexForSharedMem;
	CalcIndicesInVertexLevelMaster(GIndex, GId.x, globalStrandIndex, localStrandIndex, globalVertexIndex, localVertexIndex, numVerticesInTheStrand, indexForSharedMem);

	float4 initPos = ApplyWorldTransformToVertex(g_InitialHairPositions[globalVertexIndex],_Transform);


	float4 currentPos = g_HairVertexPositions[globalVertexIndex];
	float4 oldPos = g_HairVertexPositionsPrev[globalVertexIndex];

	sharedPos[indexForSharedMem] = g_HairVertexPositions[globalVertexIndex];
	GroupMemoryBarrierWithGroupSync();

	if (g_ResetPositions != 0.0f)
	{
		if(IsMovable(currentPos))
		{
			float dampingCoeff = GetDamping();
			sharedPos[indexForSharedMem].xyz =  Integrate(currentPos.xyz,oldPos.xyz,initPos.xyz,dampingCoeff);
		}
		else
		{
			sharedPos[indexForSharedMem] = initPos;
		}
	}
	else
	{
		sharedPos[indexForSharedMem] = initPos;
		currentPos = initPos;
		oldPos = initPos;
	}

	 // Global Shape Constraints
    float stiffnessForGlobalShapeMatching = GetGlobalStiffness();
    float globalShapeMatchingEffectiveRange = GetGlobalRange();

    if (stiffnessForGlobalShapeMatching > 0 && globalShapeMatchingEffectiveRange)
    {
        if (IsMovable(sharedPos[indexForSharedMem]))
        {
            if ((float)localVertexIndex < globalShapeMatchingEffectiveRange * (float)numVerticesInTheStrand)
            {
                float factor = stiffnessForGlobalShapeMatching;
                float3 del = factor * (initPos - sharedPos[indexForSharedMem]).xyz;
                sharedPos[indexForSharedMem].xyz += del;
            }
        }
    }

	UpdateFinalVertexPositions(currentPos,sharedPos[indexForSharedMem],globalVertexIndex);
}

ENDCG

