#ifndef THREAD_GROUP_SIZE
#define THREAD_GROUP_SIZE 64
#endif


float4x4 _Transform;

float4 g_Wind;
float4 g_Wind1;
float4 g_Wind2;
float4 g_Wind3;

float4 g_Shape;                 // damping, local stiffness, global stiffness, global range.
float4 g_GravTimeTip;           // gravity maginitude (assumed to be in negative y direction.)
float4 g_SimInts;               // Length iterations, local iterations, collision flag.
float4 g_Counts;                // num strands per thread group, num follow hairs per guid hair, num verts per strand.
float4 g_VSP;                   // VSP parmeters

float g_ResetPositions;
float g_ClampPositionDelta;

#define g_GravityMagnitude                g_GravTimeTip.x
#define g_TimeStep                        g_GravTimeTip.y
#define g_TipSeparationFactor             g_GravTimeTip.z



#define g_NumOfStrandsPerThreadGroup (uint)g_Counts.x
#define g_NumFollowHairsPerGuideHair (uint)g_Counts.y
#define g_NumVerticesPerStrand       (uint)g_Counts.z


struct StrandLevelData
{
	float4 skinningQuat;
	float4 vspQuat;
	float4 vspTranslation;
};

//UAVs
RWStructuredBuffer<float4> g_HairVertexPositions;
RWStructuredBuffer<float4> g_HairVertexPositionsPrev;
RWStructuredBuffer<float4> g_HairVertexPositionsPrevPrev;
RWStructuredBuffer<float4> g_HairVertexTangents;
RWStructuredBuffer<StrandLevelData> g_StrandLevelData;


//SRVs
StructuredBuffer<float4> g_InitialHairPositions;
StructuredBuffer<float> g_HairRestLength;



groupshared float4 sharedPos[THREAD_GROUP_SIZE];
groupshared float4 sharedTangent[THREAD_GROUP_SIZE];
groupshared float  sharedLength[THREAD_GROUP_SIZE];


//--------------------------------------------------------------------------------------
//
//  Helper Functions for the main simulation shaders
//
//--------------------------------------------------------------------------------------
bool IsMovable(float4 particle)
{
	if (particle.w > 0)
		return true;
	return false;
}

//--------------------------------------------------------------------------------------
//
//  Uses Verlet integration to calculate the new position for the current time step
//
//--------------------------------------------------------------------------------------
void CalcIndicesInVertexLevelMaster(uint local_id, uint group_id, inout uint globalStrandIndex, inout uint localStrandIndex, inout uint globalVertexIndex, inout uint localVertexIndex, inout uint numVerticesInTheStrand, inout uint indexForSharedMem)
{
	indexForSharedMem = local_id;
	numVerticesInTheStrand = (THREAD_GROUP_SIZE / g_NumOfStrandsPerThreadGroup);

	localStrandIndex = local_id % g_NumOfStrandsPerThreadGroup;
	globalStrandIndex = group_id * g_NumOfStrandsPerThreadGroup + localStrandIndex;
	globalStrandIndex *= (g_NumFollowHairsPerGuideHair + 1);
	localVertexIndex = (local_id - localStrandIndex) / g_NumOfStrandsPerThreadGroup;

	globalVertexIndex = globalStrandIndex * numVerticesInTheStrand + localVertexIndex;

}


void CalcIndicesInStrandLevelMaster(uint local_id, uint group_id, inout uint globalStrandIndex, inout uint numVerticesInTheStrand, inout uint globalRootVertexIndex)
{
    globalStrandIndex = THREAD_GROUP_SIZE*group_id + local_id;
    globalStrandIndex *= (g_NumFollowHairsPerGuideHair+1);
    numVerticesInTheStrand = (THREAD_GROUP_SIZE / g_NumOfStrandsPerThreadGroup);
    globalRootVertexIndex = globalStrandIndex * numVerticesInTheStrand;
}


float GetDamping()
{
    return g_Shape.x;
}

float GetLocalStiffness()
{
    return g_Shape.y;
}

float GetGlobalStiffness()
{
    return g_Shape.z;
}

float GetGlobalRange()
{
    return g_Shape.w;
}

float GetVelocityShockPropogation()
{
    return g_VSP.x;
}

float GetVSPAccelThreshold()
{
    return g_VSP.y;
}

float3 MultQuaternionAndVector(float4 q, float3 v)
{
    float3 uv, uuv;
    float3 qvec = float3(q.x, q.y, q.z);
    uv = cross(qvec, v);
    uuv = cross(qvec, uv);
    uv *= (2.0f * q.w);
    uuv *= 2.0f;

    return v + uv + uuv;
}

float4 InverseQuaternion(float4 q)
{
    float lengthSqr = q.x*q.x + q.y*q.y + q.z*q.z + q.w*q.w;

    if ( lengthSqr < 0.001 )
        return float4(0, 0, 0, 1.0f);

    q.x = -q.x / lengthSqr;
    q.y = -q.y / lengthSqr;
    q.z = -q.z / lengthSqr;
    q.w = q.w / lengthSqr;

    return q;
}


float4 NormalizeQuaternion(float4 q)
{
    float4 qq = q;
    float n = q.x*q.x + q.y*q.y + q.z*q.z + q.w*q.w;

    if (n < 1e-10f)
    {
        qq.w = 1;
        return qq;
    }

    qq *= 1.0f / sqrt(n);
    return qq;
}

// Compute a quaternion which rotates u to v. u and v must be unit vector. 
float4 QuatFromTwoUnitVectors(float3 u, float3 v)
{
    float r = 1.f + dot(u, v);
    float3 n;

    // if u and v are parallel
    if (r < 1e-7)
    {
        r = 0.0f;
        n = abs(u.x) > abs(u.z) ? float3(-u.y, u.x, 0.f) : float3(0.f, -u.z, u.y);
    }
    else
    {
        n = cross(u, v);  
    }

    float4 q = float4(n.x, n.y, n.z, r);
    return NormalizeQuaternion(q);
}

float4 ApplyWorldTransformToVertex(float4 vertex,float4x4 transform)
{
	float4 mulVert = float4(vertex.xyz,1.0);
	float4 mulTrans =  mul(mulVert,transform);
    return float4(mulTrans.xyz,vertex.w);
}
