local TransactionCenter = {}

local _transactionIds = setmetatable({}, { __mode = 'v' })
local _transactionCbs = setmetatable({} , { __mode = "k"})

--[[
脚本层 可以设置这个回调
onRequest = function(requestType, requestId/transactionId, timeout, jsonData)
end
]]--
-- 注册请求回调 testshell/venuseditor/likee main.lua
function TransactionCenter:SetOnRequest(onRequest)
  self.onRequest = onRequest ;
end

-- 判断id是否被占用
function TransactionCenter:HasTransaction(tId)
  return _transactionIds[tId] ~= nil
end

-- 一个Callback对象在一个请求完成之前不能再发送请求
function TransactionCenter:HasPendingWork(tCb)
  return _transactionCbs[tCb] ~= nil
end

function TransactionCenter:IsValid(tId, tCb)
  return _transactionIds[tId] == nil -- and _transactionCbs[tCb] == nil
end

--[[
蓝图 使用transaction_node.lua
脚本层 也可以直接实现这个
transactionCallback = {
  function _IsTimeOut(deltaMs) : boolean
  function _TimeOut()
  function _Ready(jsonData): boolean
            -- 返回true, 表示在Ready应答之后,是否继续响应同一transitionID请求的应答
            -- 返回false或者不返回, 表示之后不再响应该请求的应答
}
]]--
-- tCb - transactionCallback
-- tId - transactionId
function TransactionCenter:Request(tCb, tId, requestType, timeout, jsonRequest)
  if not self:IsValid(tId, tCb) then
    ERROR("it's not allowed to request twice until response");
    return
  end
  _transactionIds[tId] = tCb; -- 被 cg 这一项会被回收(释放素材之后会执行一次cg)
  _transactionCbs[tCb] = tId;  -- 建议主动调用clear及时清除

  if self.onRequest then
    self.onRequest(requestType, tId, timeout, jsonRequest);
  else
    ERROR("lost request")
  end
end

-- requestId == transactionId
function TransactionCenter:Response(requestType, requestId, jsonResult)
  local tCb = _transactionIds[requestId];
  if tCb then
    local persist = tCb:_Ready(jsonResult); -- response is ready,but it should check jsonResult like TransactionNode@_OnCheckReady
    if persist then
      WARNING("persist transaction ".. tostring(requestType)..","..tostring(requestId));
    else
      self:Clear(requestId);
    end
  else
    WARNING("drop transaction "
            .. tostring(requestType) .. "," .. tostring(requestId) .. "," .. tostring(jsonResult));
  end

end

function TransactionCenter:Update(delta) -- delta unit is second
  -- 超时的Transaction~
  delta = delta * 1000 -- second to milli-second
  for id, node in pairs(_transactionIds) do
    if node:_IsTimeOut(delta) then
      local cb = self:Clear(id); -- in pairs 可以便利 说明callback没有被cg
      cb:_TimeOut();
    end
  end

end

function TransactionCenter:Clear(tId)
  WARNING("TransactionCenter Clear ... "..tostring(tId))
  if tId == nil then return nil end
  local tCb = _transactionIds[tId];
  _transactionIds[tId] = nil
  if tCb ~= nil then _transactionCbs[tCb] = nil end
  return tCb
end

--[[
main -->
  SetOnRequest()
  Update()
  Response()

request -->
  HasTransaction()
  HasPendingWork()
  IsValid()
  Request()
  Clear()
]]--
return TransactionCenter;
