local BlueAnyTypeNode = require "bluecore.base.blue_anytype_node"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"

local BlueDynamicAnyTypePinsNode = BlueAnyTypeNode:extend("BlueDynamicAnyTypePinsNode");

-- 用途:
-- 1.Dynamic: 引脚数量不变 但是rtti/tips/name等会在编辑时修改 需要序列化保存引脚信息的节点类
-- 2.AnyType: 存在引脚rtti是any类型 任何引脚的连接 将会触发其他引脚的特化
-- 3.Pins:    引脚数目会动态增删(交互增加或者自动增加)
function BlueDynamicAnyTypePinsNode:new(...)
  BlueDynamicAnyTypePinsNode.super.new(self,...)
  -- 标记哪些引脚需要删除的,放在BluePrintPin中
end

function BlueDynamicAnyTypePinsNode:AppendInput(rtti, name, tips, default, pinUid, nodeUid, delAble, groupName)
  -- Step 1 增加 BluePinInfo
  local newArgId = 1
  for _ , __ in pairs(self[BD.INPUTS_INFO]) do
    newArgId = newArgId + 1
  end
  local pinInfo =  self:DynamicRegisterInput(newArgId, rtti, name, tips, default);

  -- Step 2 增加 BluePrintPin :连接和字面值信息
  local inputPin = {
    uid     = pinUid,
    nodeUid = nodeUid,
    pinType = BD.PIN_DATA_INPUT,
    argId   = pinInfo.argId,
    links   = {},
    useLiteral = true ,
    literal = BU:GenerateLiteral(pinInfo.rtti, pinInfo.default),
    delAble = delAble or false  -- 可以被删除
  }
  table.insert(self.inputs, inputPin); -- 会序列化

  -- Step 3 加入组
  if groupName ~= nil then
    self:JoinGroup(groupName, inputPin);
  end

  return inputPin;
end

function BlueDynamicAnyTypePinsNode:AppendOutput(rtti,  name, tips, default, pinUid, nodeUid, delAble, groupName)
  -- Step 1 增加 BluePinInfo
  local newArgId = 1
  for _ , __ in pairs(self[BD.OUTPUTS_INFO]) do
    newArgId = newArgId + 1
  end
  local pinInfo = self:DynamicRegisterOutput(newArgId, rtti, name, tips, default);

  -- Step 2 增加 BluePrintPin :连接和字面值信息
  local outputPin = {
    uid     = pinUid,
    nodeUid = nodeUid,
    pinType = BD.PIN_DATA_OUTPUT,
    argId   = pinInfo.argId,
    links   = {},
    --useLiteral = true ,
    --literal = BU:GenerateLiteral(pinInfo.rtti, pinInfo.default),
    delAble = delAble or false  -- 可以被删除
  }
  table.insert(self.outputs, outputPin); -- 会序列化

  -- Step 3 加入组
  if groupName ~= nil then
    self:JoinGroup(groupName, outputPin);
  end

  return outputPin
end

function BlueDynamicAnyTypePinsNode:AppendExecInput(rtti,  name, tips, default, pinUid, nodeUid, delAble)
  -- Step 1 增加 BluePinInfo
  local newArgId = 1
  for _ , __ in pairs(self[BD.EXEC_INPUTS_INFO]) do
    newArgId = newArgId + 1
  end
  local pinInfo = self:DynamicRegisterExecInput(newArgId, rtti, name, tips, default);

  -- Step 2 增加 BluePrintPin :连接和字面值信息
  local inputPin = {
    uid     = pinUid,
    nodeUid = nodeUid,
    pinType = BD.PIN_EXEC_INPUT,
    argId   = pinInfo.argId,
    links   = {},
    --useLiteral = true ,
    --literal = BU:GenerateLiteral(pinInfo.rtti, pinInfo.default),
    delAble = delAble or false  -- 可以被删除
  }
  table.insert(self.execInputs, inputPin); -- 会序列化
  -- 执行引脚不涉及到特化 不需要加入组遍历节点特化相关引脚

  return inputPin
end

function BlueDynamicAnyTypePinsNode:AppendExecOutput(rtti,  name, tips, default, pinUid, nodeUid, delAble)
  -- Step 1 增加 BluePinInfo
  local newArgId = 1
  for _ , __ in pairs(self[BD.EXEC_OUTPUTS_INFO]) do
    newArgId = newArgId + 1
  end
  local pinInfo = self:DynamicRegisterExecOutput(newArgId, rtti, name, tips, default);

  --local execoutput = { rtti = Types.ExecPinType, argId = argId, name = name, tips = tips, abort = false,
  --                    pinType = BD.PIN_EXEC_OUTPUT}

  -- Step 2 增加 BluePrintPin :连接和字面值信息
  local outputPin = {
    uid     = pinUid,
    nodeUid = nodeUid,
    pinType = BD.PIN_EXEC_OUTPUT,
    argId   = pinInfo.argId,
    links   = {},
    --useLiteral = true ,
    --literal = BU:GenerateLiteral(pinInfo.rtti, pinInfo.default),
    delAble = delAble or false  -- 可以被删除
  }
  table.insert(self.execOutputs, outputPin); -- 会序列化

  -- 执行引脚不涉及到特化 不需要加入组遍历节点特化相关引脚
  --if groupName ~= nil then
  --  self:JoinGroup(groupName, outputPin);
  --end

  return outputPin
end

-- 传入要删除的 pin info, 但没有处理pin删除
function BlueDynamicAnyTypePinsNode:_DynamicDeregisterInput(pinInfo)

  local argId = pinInfo.argId ;
  self[BD.INPUTS_INFO][argId] = nil

  --local remap = {} -- 旧->新
  --{1 2 3} -> {1 3} 此时iremap{1:1, 2:3}
  local iremap= {} -- 新->旧
  local new_index = 1
  for index, _ in pairs(self[BD.INPUTS_INFO]) do -- pairs 过滤nil的
    --remap[index] = new_index
    iremap[new_index] = index
    new_index = new_index + 1
  end

  -- 后面的pin将会往迁移
  for index = 1, new_index-1 do
    self[BD.INPUTS_INFO][index] = self[BD.INPUTS_INFO][iremap[index]] ;
    self.infoTable[BD.PIN_DATA_INPUT][index] = self.infoTable[BD.PIN_DATA_INPUT][iremap[index]];
    local old = self[BD.INPUTS_INFO][index].argId ;
    self[BD.INPUTS_INFO][index].argId = index
    if old ~= index then
      self:_OnOrderChange(self[BD.INPUTS_INFO][index], old, index);
    end

  end

  -- 移走最后一个
  table.remove(self.infoTable[BD.PIN_DATA_INPUT], new_index)
  table.remove(self[BD.INPUTS_INFO], new_index )

  return iremap -- 返回映射表 用来更新pin

end

-- 这里只处理  self[BD.OUTPUTS_INFO] 和 self.infoTable[BD.PIN_DATA_OUTPUT]
-- 并不处理 self.inputs 和 self.outputs 的引脚链接信息, 返回的映射表用于处理 self.inputs和outputs
function BlueDynamicAnyTypePinsNode:_DynamicDeregisterOutput(pinInfo)

    local argId = pinInfo.argId ;
    self[BD.OUTPUTS_INFO][argId] = nil

    local iremap= {} -- 新->旧
    local new_index = 1
    for index, _ in pairs(self[BD.OUTPUTS_INFO]) do
      iremap[new_index] = index
      new_index = new_index + 1
    end

    for index = 1, new_index-1 do
      self[BD.OUTPUTS_INFO][index] = self[BD.OUTPUTS_INFO][iremap[index]]
      self.infoTable[BD.PIN_DATA_OUTPUT][index] = self.infoTable[BD.PIN_DATA_OUTPUT][iremap[index]]
      local old = self[BD.OUTPUTS_INFO][index].argId ;
      self[BD.OUTPUTS_INFO][index].argId = index ;
      if old ~= index then
        self:_OnOrderChange(self[BD.OUTPUTS_INFO][index], old, index);
      end
    end

    -- 移走最后一个
    table.remove(self.infoTable[BD.PIN_DATA_OUTPUT], new_index)
    table.remove(self[BD.OUTPUTS_INFO], new_index )
    return iremap -- 返回映射表 用来更新pin
end

function BlueDynamicAnyTypePinsNode:_DynamicDeregisterExecInput(pinInfo)

    local argId = pinInfo.argId ;
    self[BD.EXEC_INPUTS_INFO][argId] = nil

    local iremap= {} -- 新->旧
    local new_index = 1
    for index, _ in pairs(self[BD.EXEC_INPUTS_INFO]) do
      iremap[new_index] = index
      new_index = new_index + 1
    end
    
    --index  iremap[index]

    for index = 1, new_index-1 do
      self[BD.EXEC_INPUTS_INFO][index] = self[BD.EXEC_INPUTS_INFO][iremap[index]]
      self.infoTable[BD.PIN_EXEC_INPUT][index] = self.infoTable[BD.PIN_EXEC_INPUT][iremap[index]]
      local old = self[BD.EXEC_INPUTS_INFO][index].argId ;
      self[BD.EXEC_INPUTS_INFO][index].argId = index ;
      if old ~= index then
        self:_OnOrderChange(self[BD.EXEC_INPUTS_INFO][index], old, index);
      end
    end

    -- 移走最后一个
    table.remove(self.infoTable[BD.PIN_EXEC_INPUT], new_index)
    table.remove(self[BD.EXEC_INPUTS_INFO], new_index )
    return iremap -- 返回映射表 用来更新pin
end

-- 这里只处理  self[BD.EXEC_OUTPUTS_INFO] 和 self.infoTable[BD.PIN_EXEC_OUTPUT]
-- 并不处理 self.execOutputs 的引脚链接信息, 返回的映射表用于处理 self.execOutputs
function BlueDynamicAnyTypePinsNode:_DynamicDeregisterExecOutput(pinInfo)

    local argId = pinInfo.argId ;
    self[BD.EXEC_OUTPUTS_INFO][argId] = nil

    local iremap= {} -- 新->旧
    local new_index = 1
    for index, _ in pairs(self[BD.EXEC_OUTPUTS_INFO]) do
      iremap[new_index] = index
      new_index = new_index + 1
    end
    
    --index  iremap[index]

    for index = 1, new_index-1 do
      self[BD.EXEC_OUTPUTS_INFO][index] = self[BD.EXEC_OUTPUTS_INFO][iremap[index]]
      self.infoTable[BD.PIN_EXEC_OUTPUT][index] = self.infoTable[BD.PIN_EXEC_OUTPUT][iremap[index]]
      local old = self[BD.EXEC_OUTPUTS_INFO][index].argId ;
      self[BD.EXEC_OUTPUTS_INFO][index].argId = index ;
      if old ~= index then
        self:_OnOrderChange(self[BD.EXEC_OUTPUTS_INFO][index], old, index);
      end
    end

    -- 移走最后一个
    table.remove(self.infoTable[BD.PIN_EXEC_OUTPUT], new_index)
    table.remove(self[BD.EXEC_OUTPUTS_INFO], new_index )
    return iremap -- 返回映射表 用来更新pin
end


function BlueDynamicAnyTypePinsNode:RemoveInput(inputPin)

  -- Step 1 删除引脚信息
  local tobeDeletedInfo = self.infoTable[inputPin.pinType][inputPin.argId]
  local iremap = self:_DynamicDeregisterInput(tobeDeletedInfo);

  -- Step 2 动态增加的引脚从group中移除(由于node可能不在group,所以由派生类)
  self:ExitGroup(BlueAnyTypeNode.DEFAULT_GROUP, inputPin.uid);

  -- Step 3 更新pin 由于 self["inputs"] 是数组 所以这里重建
  local oldInputs = self.inputs
  self.inputs = {}
  for _, info in ipairs(self.infoTable[inputPin.pinType]) do
    local pin = oldInputs[iremap[info.argId]]
    pin.argId = info.argId;
    table.insert(self.inputs, pin)
  end
end

function BlueDynamicAnyTypePinsNode:RemoveOutput(outputPin)

  -- Step 1 删除引脚信息
  local tobeDeletedInfo = self.infoTable[outputPin.pinType][outputPin.argId]
  local iremap = self:_DynamicDeregisterOutput(tobeDeletedInfo);

  -- Step 2 动态增加的引脚从group中移除(由于node可能不在group,所以由派生类)
  self:ExitGroup(BlueAnyTypeNode.DEFAULT_GROUP, outputPin.uid);

  -- Step 3 重建outputs
  local oldOutputs = self.outputs
  self.outputs  = {}
  for _, info in ipairs(self.infoTable[outputPin.pinType]) do
    local pin = oldOutputs[iremap[info.argId]]
    pin.argId = info.argId;
    table.insert( self.outputs, pin)
  end

end

function BlueDynamicAnyTypePinsNode:RemoveExecInput(execinputPin)

  -- Step 1 删除引脚信息
  local tobeDeletedInfo = self.infoTable[execinputPin.pinType][execinputPin.argId]
  local iremap = self:_DynamicDeregisterExecInput(tobeDeletedInfo);

  -- Step 3 重建inputs
  local oldExecInputs = self.execInputs
  self.execInputs  = {}
  for _, info in ipairs(self.infoTable[execinputPin.pinType]) do
    local pin = oldExecInputs[iremap[info.argId]]
    pin.argId = info.argId;
    table.insert( self.execInputs, pin)
  end

end

function BlueDynamicAnyTypePinsNode:RemoveExecOutput(execoutputPin)

  -- Step 1 删除引脚信息
  local tobeDeletedInfo = self.infoTable[execoutputPin.pinType][execoutputPin.argId]
  local iremap = self:_DynamicDeregisterExecOutput(tobeDeletedInfo);

  -- Step 2 动态增加的引脚从group中移除(由于node可能不在group,所以由派生类)
  --self:ExitGroup(BlueAnyTypeNode.DEFAULT_GROUP, execoutputPin.uid);

  -- Step 3 重建outputs
  local oldExecOutputs = self.execOutputs
  self.execOutputs  = {}
  for _, info in ipairs(self.infoTable[execoutputPin.pinType]) do
    local pin = oldExecOutputs[iremap[info.argId]]
    pin.argId = info.argId;
    table.insert( self.execOutputs, pin)
  end

end


function BlueDynamicAnyTypePinsNode:_OnDeserialize()
  BlueDynamicAnyTypePinsNode.super._OnDeserialize(self);
  -- 反序列化必须先调用基类的 _OnDeserialize 再执行派生类的反序列化操作
end

function BlueDynamicAnyTypePinsNode:_OnDeserializePost()
  BlueDynamicAnyTypePinsNode.super._OnDeserializePost(self)
end

function BlueDynamicAnyTypePinsNode:_OnCreateWithEditor()
  BlueDynamicAnyTypePinsNode.super._OnCreateWithEditor(self);
  -- 把 MemberRegister注册时标记可被删除,在生成pin上标记删除
  for _, pinInfo in pairs(self.infoTable[BD.PIN_DATA_INPUT]) do
    if pinInfo.delAble == true then
      local pin = self:GetPinByInfo(pinInfo);
      pin.delAble = true
    end
  end
  for _, pinInfo in pairs(self.infoTable[BD.PIN_DATA_OUTPUT]) do
    if pinInfo.delAble == true then
      local pin = self:GetPinByInfo(pinInfo);
      pin.delAble = true
    end
  end
  for _, pinInfo in pairs(self.infoTable[BD.PIN_EXEC_INPUT]) do
    if pinInfo.delAble == true then
      local pin = self:GetPinByInfo(pinInfo);
      pin.delAble = true
    end
  end
  for _, pinInfo in pairs(self.infoTable[BD.PIN_EXEC_OUTPUT]) do
    if pinInfo.delAble == true then
      local pin = self:GetPinByInfo(pinInfo);
      pin.delAble = true
    end
  end
end

--派生类实现 目前用于改变引脚name和tips
function BlueDynamicAnyTypePinsNode:_OnOrderChange(pinInfo, oldIdx, newIdx)
end


return BlueDynamicAnyTypePinsNode