local ae = require "apolloengine"
local vc = require "venuscore"
local mf = require "mathfunction"

local function _CreateRenderTarget(size)
  local fbo = ae.RenderTargetEntity()
  fbo:PushMetadata(
    ae.RenderTargetMetadata(
      ae.RenderTargetEntity.RT_RENDER_TARGET_2D,
      ae.RenderTargetEntity.ST_SWAP_UNIQUE,
      mf.vector4(
        0,
        0,
        size:x(),
        size:y()),
      mf.vector2(
        size:x(),
        size:y())))
  local tex = fbo:MakeTextureAttachment(
          ae.RenderTargetEntity.TA_COLOR_0)
  tex:PushMetadata(
    ae.TextureRenderMetadata(
      ae.RenderTargetEntity.ST_SWAP_UNIQUE,
      mf.vector2(size:x(), size:y()),
      ae.TextureEntity.TW_CLAMP_TO_BORDER,
      ae.TextureEntity.TW_CLAMP_TO_BORDER,
      ae.TextureEntity.TF_LINEAR,
      ae.TextureEntity.TF_LINEAR,
      1
    ))
  fbo:SetJobType(vc.IJob.JT_SYNCHRONOUS)
  fbo:CreateResource()
  return fbo
end

local function _CreateEmptyTexture()
  local tex = ae.TextureEntity()
  tex:PushMetadata(ae.TextureBufferMetadata(mf.vector2(4, 4)))
  tex:SetJobType(vc.IJob.JT_SYNCHRONOUS)
  tex:CreateResource()
  return tex
end

-------------------------------------------------------------------------------

local RenderMedian = vc.Object:extend()

function RenderMedian:new(host, size)
  self._MainTex = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "_MainTex"
  )
  self._MainSize = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "_MainSize"
  )
  self._Step = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "_Step"
  )
  self.render = host:CreateRenderObject("comm:documents/material/median.material")
  self:SetStep(0.0, 0.0)
  self:Resize(size)
end

function RenderMedian:SetStep(angle, length)
  local rad = angle * math.pi / 180.0
  local step = mf.vector2(
    math.cos(rad),
    math.sin(rad)
  ) * length
  self.render:SetParameter(
    self._Step, 
    step
  )
end

function RenderMedian:SetTexture(tex)
  self.render:SetParameter(
    self._MainTex,
    tex
  )
end

function RenderMedian:Resize(size)
  self.render:SetParameter(
    self._MainSize,
    size
  )
end

function RenderMedian:Draw(pipeline)
  self.render:Draw(pipeline)
end
-------------------------------------------------------------------------------

local RenderBlurr = vc.Object:extend()

function RenderBlurr:new(host, size)
  self.BlUR_LENGTH = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "BlUR_LENGTH")
  self.ANGLE = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "ANGLE")
  self.ratio = self:GetRatio(size)
  self.len = 10
  self.angle = 0
  self.step = 1
  self.render = host:CreateRenderObject("comm:documents/material/directionalblur.material")
  self:SetLength(self.len)
  self:SetAngle(self.angle, self.step)
end

function RenderBlurr:GetRatio(size)
  local ratio = 1;
  if size:y() ~= 0 then
    ratio = size:x()/size:y();
  end
  return ratio;
end

function RenderBlurr:SetLength(len)
  self.len = len
  self.render:SetParameter(
    self.BlUR_LENGTH, 
    mf.vector1(math.floor(math.abs(len))))
end

function RenderBlurr:SetAngle(angle, step)
  self.angle = angle
  self.step = step
  self.render:SetParameter(
    self.ANGLE, 
    mf.vector2(
      -math.cos(angle/180.0 * math.pi),
      math.sin(angle/180.0 * math.pi) * self.ratio
    ) * step * 0.001
  )
end

function RenderBlurr:SetTexture(tex)
  self.render:SetParameter(
		ae.ShaderEntity.TEXTURE_DIFFUSE,
		tex)
end

function RenderBlurr:Resize(size)
  self.ratio = self:GetRatio(size)
  self:SetLength(self.len)
  self:SetAngle(self.angle, self.step)
end

function RenderBlurr:Draw(pipeline)
  self.render:Draw(pipeline)
end

-------------------------------------------------------------------------------

local RenderBlit = vc.Object:extend()

function RenderBlit:new(host, size)
  self.render = host:CreateRenderObject("comm:documents/material/imageblit.material")
end

function RenderBlit:SetTexture(tex)
  self.render:SetParameter(
    ae.ShaderEntity.TEXTURE_DIFFUSE,
    tex)
end

function RenderBlit:Draw(pipeline)
  self.render:Draw(pipeline)
end

-------------------------------------------------------------------------------

local RenderStuff = vc.Object:extend()

function RenderStuff:new(host, size)
  local uniforms = 
  {
    "_MaskTex",
    "_BgTex",
    "_FgTex",
    "_BgSize",
    "_MaskMatR1",
    "_MaskMatR2",
    "_MaskMatR3",
    "_RefRect1",
    "_RefRect2",
    "_RefRect3",
    "_RefRect4",
  }
  for i = 1, #uniforms do
    self[uniforms[i]] = ae.IMaterialSystem:NewParameterSlot(ae.ShaderEntity.UNIFORM, uniforms[i])
  end
  self.render = host:CreateRenderObject("comm:documents/material/skin_stuff.material")
end

function RenderStuff:SetMaskTexture(tex)
  self.render:SetParameter(
    self._MaskTex,
    tex)
end

function RenderStuff:SetBgTexture(tex)
  self.render:SetParameter(
    self._BgTex,
    tex)
end

function RenderStuff:SetFgTexture(tex)
  self.render:SetParameter(
    self._FgTex,
    tex)
end

function RenderStuff:SetBgSize(bgSize)
  self.render:SetParameter(self._BgSize, bgSize)
end

function RenderStuff:SetMaskMat(mat)
  self.render:SetParameter(self._MaskMatR1, mat:Row(0))
  self.render:SetParameter(self._MaskMatR2, mat:Row(1))
  self.render:SetParameter(self._MaskMatR3, mat:Row(2))
end

function RenderStuff:SetRefRect(r1, r2, r3, r4)
  self.render:SetParameter(self._RefRect1, r1)
  self.render:SetParameter(self._RefRect2, r2)
  self.render:SetParameter(self._RefRect3, r3)
  self.render:SetParameter(self._RefRect4, r4)
end

function RenderStuff:Draw(pipeline)
  self.render:Draw(pipeline)
end

-------------------------------------------------------------------------------

local RenderSmooth = vc.Object:extend()

function RenderSmooth:new(host, size)
   
  self.TARGETSIZE = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "iResolution")
  self.SIGMA_R = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "BSIGMA")
  self.render = host:CreateRenderObject("comm:documents/material/mopi.material")
  self.sigma_r = mf.vector1(0.3)
  self.render:SetParameter(self.TARGETSIZE, size)
  self.render:SetParameter(self.SIGMA_R, self.sigma_r)
end

function RenderSmooth:Resize(size)
  self.render:SetParameter(self.TARGETSIZE, size)
end

function RenderSmooth:SetSigmaR(sigmaR)
  self.sigma_r = mf.vector1(sigmaR)
  self.render:SetParameter(self.SIGMA_R, self.sigma_r)
end

function RenderSmooth:SetTexture(tex)
  self.render:SetParameter(
    ae.ShaderEntity.TEXTURE_DIFFUSE,
    tex)
end

function RenderSmooth:Draw(pipeline)
  self.render:Draw(pipeline)
end

-------------------------------------------------------------------------------

local RenderHaldLUT = vc.Object:extend()

function RenderHaldLUT:new(host, size)
  self.LOOKUPTABLE = ae.IMaterialSystem:NewParameterSlot(
    ae.ShaderEntity.UNIFORM,
    "LOOKUPTABLE")
  self.render = host:CreateRenderObject("comm:documents/material/hald_lut.material")
  self.lutPath = "comm:documents/texture/posteffect/lutpv.png"
  self.tex = ae.TextureEntity()
  self.tex:PushMetadata(ae.TextureFileMetadata(
    ae.TextureEntity.TU_STATIC,
    ae.TextureEntity.PF_AUTO,
    1,
    false,
    ae.TextureEntity.TW_CLAMP_TO_EDGE,
    ae.TextureEntity.TW_CLAMP_TO_EDGE,
    ae.TextureEntity.TF_NEAREST,
    ae.TextureEntity.TF_NEAREST,
    self.lutPath
  ))
  self.tex:SetJobType(vc.IJob.JT_SYNCHRONOUS)
  self.tex:CreateResource()
  self.render:SetParameter(self.LOOKUPTABLE, self.tex)
end

function RenderHaldLUT:SetTexture(tex)
  self.render:SetParameter(
    ae.ShaderEntity.TEXTURE_DIFFUSE,
    tex)
end

function RenderHaldLUT:Draw(pipeline)
  self.render:Draw(pipeline)
end

-------------------------------------------------------------------------------

local PostEffect = {}

function PostEffect:_InitParameters(host)
  self.stuffLE = false;
  self.stuffRE = false;
  self.stuffLB = false;
  self.stuffRB = false;
  self.stuffNS = false;
  self.stuffMH = false;
  self.stuffFC = false;

  self.maskLE = _CreateEmptyTexture();
  self.maskRE = _CreateEmptyTexture();
  self.maskLB = _CreateEmptyTexture();
  self.maskRB = _CreateEmptyTexture();
  self.maskNS = _CreateEmptyTexture();
  self.maskMH = _CreateEmptyTexture();
  self.maskFC = _CreateEmptyTexture();

  self.maskMatLE = mf.Matrix33()
  self.maskMatRE = mf.Matrix33()
  self.maskMatLB = mf.Matrix33()
  self.maskMatRB = mf.Matrix33()
  self.maskMatNS = mf.Matrix33()
  self.maskMatMH = mf.Matrix33()
  self.maskMatFC = mf.Matrix33()
  
  self.refRect1LE = mf.vector4(0,0,1,1)
  self.refRect2LE = mf.vector4(0,0,1,1)
  self.refRect3LE = mf.vector4(0,0,1,1)
  self.refRect4LE = mf.vector4(0,0,1,1)
  self.refRect1RE = mf.vector4(0,0,1,1)
  self.refRect2RE = mf.vector4(0,0,1,1)
  self.refRect3RE = mf.vector4(0,0,1,1)
  self.refRect4RE = mf.vector4(0,0,1,1)
  self.refRect1LB = mf.vector4(0,0,1,1)
  self.refRect2LB = mf.vector4(0,0,1,1)
  self.refRect3LB = mf.vector4(0,0,1,1)
  self.refRect4LB = mf.vector4(0,0,1,1)
  self.refRect1RB = mf.vector4(0,0,1,1)
  self.refRect2RB = mf.vector4(0,0,1,1)
  self.refRect3RB = mf.vector4(0,0,1,1)
  self.refRect4RB = mf.vector4(0,0,1,1)
  self.refRect1NS = mf.vector4(0,0,1,1)
  self.refRect2NS = mf.vector4(0,0,1,1)
  self.refRect3NS = mf.vector4(0,0,1,1)
  self.refRect4NS = mf.vector4(0,0,1,1)
  self.refRect1MH = mf.vector4(0,0,1,1)
  self.refRect2MH = mf.vector4(0,0,1,1)
  self.refRect3MH = mf.vector4(0,0,1,1)
  self.refRect4MH = mf.vector4(0,0,1,1)
  self.refRect1FC = mf.vector4(0,0,1,1)
  self.refRect2FC = mf.vector4(0,0,1,1)
  self.refRect3FC = mf.vector4(0,0,1,1)
  self.refRect4FC = mf.vector4(0,0,1,1)

  self.bgSize = mf.vector2(4,4)
end

function PostEffect:Initialize(host, size)
  self:_InitParameters(host)
  self.rtPreProc1 = _CreateRenderTarget(size)
  self.rtPreProc2 = _CreateRenderTarget(size)
  self.rtProc1 = _CreateRenderTarget(size)
  self.rtProc2 = _CreateRenderTarget(size)
  self.roBlit = RenderBlit(host, size)
  self.roStuff = RenderStuff(host, size)
  self.roSmooth = RenderSmooth(host, size)
  self.roLUT = RenderHaldLUT(host, size)
  self.roMedian = RenderMedian(host, size)
  self.roBlur = RenderBlurr(host, size)
  return 310
end

function PostEffect:Resizeview(size)
  if self.rtPreProc1 then
    self.rtPreProc1:Discard()
  end
  if self.rtPreProc2 then
    self.rtPreProc2:Discard()
  end
  if self.rtProc1 then
    self.rtProc1:Discard()
  end
  if self.rtProc2 then
    self.rtProc2:Discard()
  end
  self.rtPreProc1 = _CreateRenderTarget(size)
  self.rtPreProc2 = _CreateRenderTarget(size)
  self.rtProc1 = _CreateRenderTarget(size)
  self.rtProc2 = _CreateRenderTarget(size)
  self.roSmooth:Resize(size)
  self.roMedian:Resize(size)
  self.roBlur:Resize(size)
end

function PostEffect:_SwapProcRT()
  local t = self.rtProc1
  self.rtProc1 = self.rtProc2
  self.rtProc2 = t
end

function PostEffect:_SwapPreProcRT()
  local t = self.rtPreProc1
  self.rtPreProc1 = self.rtPreProc2
  self.rtPreProc2 = t
end
--[[
function PostEffect:Process(pipeline, Original, Scene, Output)
  local CF_COLOR = ae.RenderTargetEntity.CF_COLOR
  local TA_COLOR_0 = ae.RenderTargetEntity.TA_COLOR_0
  --Smooth 1
  self.rtPreProc1:PushRenderTarget()
  self.rtPreProc1:ClearBuffer(CF_COLOR)
  self.roSmooth:SetSigmaR(0.3)
  self.roSmooth:SetTexture(Scene:GetAttachment(TA_COLOR_0))
  self.roSmooth:Draw(pipeline)
  --Smooth 2
  self.rtPreProc2:PushRenderTarget()
  self.rtPreProc2:ClearBuffer(CF_COLOR)
  self.roSmooth:SetSigmaR(0.05)
  self.roSmooth:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roSmooth:Draw(pipeline)
  self:_SwapPreProcRT()
  --LUT (Lighten)
  self.rtPreProc2:PushRenderTarget()
  self.rtPreProc2:ClearBuffer(CF_COLOR)
  self.roLUT:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roLUT:Draw(pipeline)
  self:_SwapPreProcRT()
  --Blit Original
  self.rtProc2:PushRenderTarget()
  self.rtProc2:ClearBuffer(CF_COLOR)
  self.roBlit:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roBlit:Draw(pipeline)
  --WARNING(string.format("SUNTYLOG: BgSize %d %d", self.bgSize:x(), self.bgSize:y()))
  --Left Eye
  if self.stuffLE then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatLE)
    self.roStuff:SetMaskTexture(self.maskLE)
    self.roStuff:SetRefRect(
      self.refRect1LE,
      self.refRect2LE,
      self.refRect3LE,
      self.refRect4LE
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Right Eye
  if self.stuffRE then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatRE)
    self.roStuff:SetMaskTexture(self.maskRE)
    self.roStuff:SetRefRect(
      self.refRect1RE,
      self.refRect2RE,
      self.refRect3RE,
      self.refRect4RE
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Left Brow
  if self.stuffLB then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatLB)
    self.roStuff:SetMaskTexture(self.maskLB)
    self.roStuff:SetRefRect(
      self.refRect1LB,
      self.refRect2LB,
      self.refRect3LB,
      self.refRect4LB
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Right Brow
  if self.stuffRB then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatRB)
    self.roStuff:SetMaskTexture(self.maskRB)
    self.roStuff:SetRefRect(
      self.refRect1RB,
      self.refRect2RB,
      self.refRect3RB,
      self.refRect4RB
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Nose
  if self.stuffNS then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatNS)
    self.roStuff:SetMaskTexture(self.maskNS)
    self.roStuff:SetRefRect(
      self.refRect1NS,
      self.refRect2NS,
      self.refRect3NS,
      self.refRect4NS
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Mouth
  if self.stuffMH then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatMH)
    self.roStuff:SetMaskTexture(self.maskMH)
    self.roStuff:SetRefRect(
      self.refRect1MH,
      self.refRect2MH,
      self.refRect3MH,
      self.refRect4MH
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Face
  if self.stuffFC then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatFC)
    self.roStuff:SetMaskTexture(self.maskFC)
    self.roStuff:SetRefRect(
      self.refRect1FC,
      self.refRect2FC,
      self.refRect3FC,
      self.refRect4FC
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Blit Output
  Output:PushRenderTarget()
  Output:ClearBuffer(CF_COLOR)
  self.roBlit:SetTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
  self.roBlit:Draw(pipeline)
end
]]--

function PostEffect:Process(pipeline, Original, Scene, Output)
  local CF_COLOR = ae.RenderTargetEntity.CF_COLOR
  local TA_COLOR_0 = ae.RenderTargetEntity.TA_COLOR_0
  --Median X1
  self.rtPreProc1:PushRenderTarget()
  self.rtPreProc1:ClearBuffer(CF_COLOR)
  self.roMedian:SetStep(0.0, 11.0)
  self.roMedian:SetTexture(Scene:GetAttachment(TA_COLOR_0))
  self.roMedian:Draw(pipeline)
  --Median X2
  self.rtPreProc2:PushRenderTarget()
  self.rtPreProc2:ClearBuffer(CF_COLOR)
  self.roMedian:SetStep(0.0, 4.0)
  self.roMedian:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roMedian:Draw(pipeline)
  self:_SwapPreProcRT()
  --Median Y1
  self.rtPreProc2:PushRenderTarget()
  self.rtPreProc2:ClearBuffer(CF_COLOR)
  self.roMedian:SetStep(90.0, 11.0)
  self.roMedian:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roMedian:Draw(pipeline)
  self:_SwapPreProcRT()
  --Median Y2
  self.rtPreProc2:PushRenderTarget()
  self.rtPreProc2:ClearBuffer(CF_COLOR)
  self.roMedian:SetStep(90.0, 4.0)
  self.roMedian:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roMedian:Draw(pipeline)
  self:_SwapPreProcRT()
  --Smooth Preproc
  self.rtPreProc2:PushRenderTarget()
  self.rtPreProc2:ClearBuffer(CF_COLOR)
  self.roSmooth:SetSigmaR(0.3)
  self.roSmooth:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  self.roSmooth:Draw(pipeline)
  self:_SwapPreProcRT()
  --Lighten Preproc
  -- self.rtPreProc2:PushRenderTarget()
  -- self.rtPreProc2:ClearBuffer(CF_COLOR)
  -- self.roLUT:SetTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
  -- self.roLUT:Draw(pipeline)
  -- self:_SwapPreProcRT()
  --Smooth Original 1
  self.rtProc2:PushRenderTarget()
  self.rtProc2:ClearBuffer(CF_COLOR)
  self.roSmooth:SetSigmaR(0.3)
  self.roSmooth:SetTexture(Scene:GetAttachment(TA_COLOR_0))
  self.roSmooth:Draw(pipeline)
  --Smooth Original 2
  self.rtProc1:PushRenderTarget()
  self.rtProc2:ClearBuffer(CF_COLOR)
  self.roSmooth:SetSigmaR(0.05)
  self.roSmooth:SetTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
  self.roSmooth:Draw(pipeline)
  self:_SwapProcRT()
  --Lighten Original
  -- self.rtProc1:PushRenderTarget()
  -- self.rtProc2:ClearBuffer(CF_COLOR)
  -- self.roLUT:SetTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
  -- self.roLUT:Draw(pipeline)
  -- self:_SwapProcRT()
  --Left Eye
  if self.stuffLE then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatLE)
    self.roStuff:SetMaskTexture(self.maskLE)
    self.roStuff:SetRefRect(
      self.refRect1LE,
      self.refRect2LE,
      self.refRect3LE,
      self.refRect4LE
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Right Eye
  if self.stuffRE then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatRE)
    self.roStuff:SetMaskTexture(self.maskRE)
    self.roStuff:SetRefRect(
      self.refRect1RE,
      self.refRect2RE,
      self.refRect3RE,
      self.refRect4RE
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Left Brow
  if self.stuffLB then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatLB)
    self.roStuff:SetMaskTexture(self.maskLB)
    self.roStuff:SetRefRect(
      self.refRect1LB,
      self.refRect2LB,
      self.refRect3LB,
      self.refRect4LB
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Right Brow
  if self.stuffRB then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatRB)
    self.roStuff:SetMaskTexture(self.maskRB)
    self.roStuff:SetRefRect(
      self.refRect1RB,
      self.refRect2RB,
      self.refRect3RB,
      self.refRect4RB
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Nose
  if self.stuffNS then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatNS)
    self.roStuff:SetMaskTexture(self.maskNS)
    self.roStuff:SetRefRect(
      self.refRect1NS,
      self.refRect2NS,
      self.refRect3NS,
      self.refRect4NS
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Mouth
  if self.stuffMH then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatMH)
    self.roStuff:SetMaskTexture(self.maskMH)
    self.roStuff:SetRefRect(
      self.refRect1MH,
      self.refRect2MH,
      self.refRect3MH,
      self.refRect4MH
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Face
  if self.stuffFC then
    self.rtProc1:PushRenderTarget()
    self.rtProc1:ClearBuffer(CF_COLOR)
    self.roStuff:SetMaskMat(self.maskMatFC)
    self.roStuff:SetMaskTexture(self.maskFC)
    self.roStuff:SetRefRect(
      self.refRect1FC,
      self.refRect2FC,
      self.refRect3FC,
      self.refRect4FC
    )
    self.roStuff:SetBgTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
    self.roStuff:SetFgTexture(self.rtPreProc1:GetAttachment(TA_COLOR_0))
    self.roStuff:SetBgSize(self.bgSize)
    self.roStuff:Draw(pipeline)
    self:_SwapProcRT()
  end
  --Blit Output
  Output:PushRenderTarget()
  Output:ClearBuffer(CF_COLOR)
  self.roBlit:SetTexture(self.rtProc2:GetAttachment(TA_COLOR_0))
  self.roBlit:Draw(pipeline)
  
end

return PostEffect