﻿local BundleSystem = require "venuscore.bundle.bundlesystem"
local mathfunction = require "mathfunction"
local apolloengine = require "apollocore"
local venuscore = require "venuscore"
local apollocore = require "apollocore"

local EditorMixEffectManager = {}

function EditorMixEffectManager:Initialize(sequenceBase)
  self.resourceType = sequenceBase
  self.sequenceBase = sequenceBase
  self.mixeffectRootNode = nil;
  self.mixNodeList = {};
  self.cameraRenderTarget = nil;
  self.nodeBehaviorInstances = {};
end

function EditorMixEffectManager:LoadMixeffect(cutterScene, path, rootconfig)
  ERROR("EditorMixEffectManager:LoadMixeffect " .. path)
  self:_CreateMixEffectRootNodeIfNeed(cutterScene)
  local mixnode = self:_CreateMixNode(cutterScene, path, rootconfig)
  if mixnode ~= nil then
    if self.cameraRenderTarget then
      LOG("self.cameraRenderTarget exist")
    end
    self:_ConfigRenderTarget(mixnode, self.cameraRenderTarget)
    self.mixeffectRootNode:AttachNode(mixnode);
    table.insert(self.mixNodeList, mixnode);
    LOG("EditorMixEffectManager:LoadMixeffect num= " .. #self.mixNodeList)
    return true;
  else
    LOG("EditorMixEffectManager:LoadMixeffect mixnode is nil")
  end
  return false;
end

function EditorMixEffectManager:UnloadLastMixEffect(scene)
  if scene == nil then
    LOG("UnloadLastMixEffect Error: scene is nil!");
    return false;
  end
  local id = #self.mixNodeList;
  LOG("UnloadLastMixEffect id "..tostring(id));
  local mixnode = self.mixNodeList[id];
  if mixnode then
    self.mixeffectRootNode:DetachNode(mixnode);
    scene:DeleteNode(mixnode);
    table.remove(self.mixNodeList);
    table.remove(self.nodeBehaviorInstances)
    return true;
  end
  return false;
end

function EditorMixEffectManager:RemoveLastMixeffect(effectNum, scene)
  local isMixeffect = false;
  for i=1,effectNum do
    isMixeffect = self:UnloadLastMixEffect(scene);
  end
  return isMixeffect;
end

function EditorMixEffectManager:ClearMixeffect(scene)
  for i=1,#self.mixNodeList do
    self:UnloadLastMixEffect(scene);
  end
end

function EditorMixEffectManager:IsExistMixeffect()
  return #self.mixNodeList > 0;
end

function EditorMixEffectManager:SetCurMixeffectRange(startFrameIndex, endFrameIndex)
  LOG("EditorMixEffectManager:SetCurMixeffectRange ".. tostring(startFrameIndex).. ", ".. tostring(endFrameIndex))
  if #self.nodeBehaviorInstances > 0 then
    self.nodeBehaviorInstances[#self.nodeBehaviorInstances]:SetFrameRange(startFrameIndex, endFrameIndex)
  end
end

function EditorMixEffectManager:updateFrameInfo(frameIndex, pts)
  local findActive = 0;
  for index = #self.nodeBehaviorInstances, 1, -1 do
    local instance = self.nodeBehaviorInstances[index];
    if findActive == 0 then
      if instance:checkActiveStatus(frameIndex) then
        findActive = index;
        self.mixNodeList[findActive].Active = true;
        instance:updateFrameInfo(frameIndex, pts)
        LOG("EditorMixEffectManager findActive id:"..tostring(index).."frameIndex"..tostring(frameIndex));
      else
        self.mixNodeList[index].Active = false;
      end
    else
      self.mixNodeList[index].Active = false;
    end
  end
end

function EditorMixEffectManager:SetMainrt(rt)
  LOG("EditorMixEffectManager:SetMainrt")
  for index = 1, #self.mixNodeList do
    local mixnode = self.mixNodeList[index]
    self:_ConfigRenderTarget(mixnode, rt)
  end
  self.cameraRenderTarget = rt;
end


function EditorMixEffectManager:_CreateMixNode(cutterScene, path, rootconfig)
  if rootconfig.scene == nil then   
    return nil;
  end

  -- NOTE: 创建一个新的节点，新的节点挂一个behavior,用bahavior去管理node的active状态
  local mixnode = cutterScene:CreateNode(apollocore.Node.CT_NODE);
  mixnode:SetName("MixEffect_"..tostring(#self.mixNodeList + 1));

  local component = mixnode:CreateComponent(apollocore.Node.CT_TRANSFORM);
  component:SetLocalPosition(mathfunction.vector3(0.0,0.0,0.0));

  local scriptComp = mixnode:CreateComponent(apolloengine.Node.CT_SCRIPT);
  local scriptPath = "scrs:behavior/mix_node_behavior.lua";
  local instance = self:_LoadBehavior(scriptComp, scriptPath)

  local baseobjects = self:_CreateInstanceFromBundle2(
    cutterScene,
    path,
    rootconfig.scene,
    #self.mixNodeList + 1,
    mixnode
  )

  if instance then
    instance:initBehaviorData(mixnode, baseobjects);
    table.insert(self.nodeBehaviorInstances, instance)
  end

  self:_CameraInit(baseobjects);
  mixnode.Active = false;

  return mixnode
end

function EditorMixEffectManager:_CreateInstanceFromBundle2(scene, resPath, resSceneConfig, id, mixnode)
  local pathDir = string.match(resPath, "(.+)/[^/]*%.%w+$");
  local rootdir = pathDir.."/"; 
  venuscore.IFileSystem:SetResourcePath(rootdir);

  local bundlepath = venuscore.IFileSystem:PathAssembly(resSceneConfig);
  --local file = io.open(bundlepath, "rb");
  --local str = file:read("*a");
  local sceneRoot = scene:GetRootNode();
  --local skipRootNodeForScene = false;
  --local objects = BundleSystem:DeSerialize(str,
  --                                        scene,
  --                                        BundleSystem.DeserializeMode.Prefab,
  --                                        skipRootNodeForScene);
  
  local _,objects = BundleSystem:DeserializeFromPath(bundlepath,BundleSystem.DeserializeMode.Prefab,scene);

  
  -- 找到root node
  for i=1,#objects do
    local nativeNode = objects[i];
    if nativeNode.GetNativeNode then
      nativeNode = objects[i]:GetNativeNode();
    end
    local nodeRoot = nativeNode:GetRoot();
    if nodeRoot and nodeRoot:GetObjectID() == sceneRoot:GetObjectID() then
      -- 已经找到root node
      mixnode:AttachNode(nativeNode);
    end
  end

  return objects;
end

function EditorMixEffectManager:_CameraInit(baseobject)
  self.resourceType = self.resourceType + 1
  local sequenceBase = self.sequenceBase
  local sequenceMax = 0
  for i=1,#baseobject do
    local cameracom = baseobject[i]:GetComponent(apolloengine.Node.CT_CAMERA);
    if cameracom then
      cameracom:SetResourceType(self.resourceType);
      local sequence = cameracom:GetSequence() + sequenceBase;
      if sequence > sequenceMax then
        sequenceMax = sequence
      end
      cameracom:SetSequence(sequence);
      cameracom:SetSequenceCulling(false);
    end
    
    local renderCom = baseobject[i]:GetComponent(apolloengine.Node.CT_RENDER);
    if renderCom then
      renderCom:SetResourceType(self.resourceType);
    end
  end
  self.sequenceBase = sequenceMax + 1
end

function EditorMixEffectManager:_ConfigRenderTarget(node, rt)
  if rt ~= nil then
    local scriptComp = node:GetComponent(apolloengine.Node.CT_SCRIPT);
    if scriptComp ~= nil then
      local paraSet = scriptComp.Instances
      for scrKey,scrValue in pairs(paraSet) do
        scrValue:ConfigRenderTarget(rt);
      end
    end
  else
    LOG("EditorMixEffectManager:_ConfigRenderTarget is nil")
  end
end

function EditorMixEffectManager:_LoadBehavior(scriptComp, scriptPath)
  local instance = venuscore.LoadBehavior(scriptPath);
  if instance then
    scriptComp:InsertInstance(instance, scriptPath);
  else
    LOG("[_LoadBehavior] load failed. "..scriptPath)
  end
  return instance;
end

function EditorMixEffectManager:_CreateMixEffectRootNodeIfNeed(scene)
  if self.mixeffectRootNode == nil then
    self.mixeffectRootNode = scene:CreateNode(apollocore.Node.CT_NODE);
    self.mixeffectRootNode:SetName("MixEffectRootNode");
    local component = self.mixeffectRootNode:CreateComponent(apollocore.Node.CT_TRANSFORM);
    component:SetLocalPosition(mathfunction.vector3(0.0,0.0,0.0));
  end
end

return EditorMixEffectManager;
