
local TransNode = require "apolloutility.apollonode.trasnnode"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local util = require "behavior.vtuber_behavior.util"
local apolloDefine = require "apolloutility.defiend"
local SkeletonNode = TransNode:extend();
local radiusscale = math.pi / 180;


function SkeletonNode:new()
  SkeletonNode.super.new(self);
  
  self.kinectTPoseRotX = mathfunction.vector3(1,0,0);
  self.kinectTPoseRotY = mathfunction.vector3(0,1,0);
  self.kinectTPoseRotZ = mathfunction.vector3(0,0,1);
  self.kinectInitRot = mathfunction.Matrix33(
        1,0,0,
        0,1,0,
        0,0,1
    )
  self.spineTposRot = {
    mathfunction.Matrix33(
      1,0,0,
      0,1,0,
      0,0,1
    ),
    mathfunction.Matrix33(
      1,0,0,
      0,1,0,
      0,0,1
    ),
    mathfunction.Matrix33(
       1,0,0,
      0,1,0,
      0,0,1
    ),
    mathfunction.Matrix33(
       1,0,0,
      0,1,0,
      0,0,1
    ), --neck
    
  }
    
    
  self.modelTposRot = {
       mathfunction.Matrix33(
        0,1,0,
        -1,0,0,
        0,0,1
      ),
      mathfunction.Matrix33(
        0,-1,0,
        -1,0,0,
        0,0,-1
      ),
       mathfunction.Matrix33(
        0,1,0,
        -1,0,0,
        0,0,1
      ),
      mathfunction.Matrix33(
        0,1,0,
        -1,0,0,
        0,0,1
      ),
      nil,
      nil,
      mathfunction.Matrix33(
        0,-1,0,
        1,0,0,
        0,0,1
      ),
       mathfunction.Matrix33(
        0,1,0,
        1,0,0,
        0,0,-1
      ),
       mathfunction.Matrix33(
         0,-1,0,
        1,0,0,
        0,0,1
      ),
       mathfunction.Matrix33(
         0,-1,0,
        1,0,0,
        0,0,1
      ),
      nil,
      nil,
  }
end

function SkeletonNode:SetBonesRef(table_bones)
  self.BoneNodes = {};
  for name,node in pairs(table_bones)  do
    self.BoneNodes[name] = node;
  end
end

function SkeletonNode:SetPoseEstimateBoneNames(rootname,spinebone,bonenames,children,bodypair)
  self.rootbone = rootname;
  self:_SetRootJoint(rootname);
  
  self.spinebone = spinebone;
  self.bonenamemap = bonenames;
  self.bodybones = {
      self.bonenamemap[1],
      self.bonenamemap[7],
      self.bonenamemap[13],
      self.bonenamemap[17]
    }
    
  self.children = children;
  self.bodypair = bodypair;
  self.bonecount = #self.bonenamemap;
end

function SkeletonNode:SetLaplacianExtras(laplacianextras)
  self.laplacianextras = laplacianextras;
 
end

function SkeletonNode:InitCollision( colliders,names )
  if colliders==nil then
    return;
  end
  self.rigidId2Name = {};
  self.collisionshapenames = {};
  for i=1 ,#colliders  do
    local collidernode = colliders[i]:GetHostNode();
    self.rigidId2Name[colliders[i]:GetObjectID()] = names[i];
    table.insert(self.collisionshapenames,names[i]);
  end
end

function SkeletonNode:GetCurrentPose()
  local currentpose = {}
  for i=1,self.bonecount,1 do
    currentpose[i] = self:GetTPos(i);
  end
  local mid = {}
  mid[1] = (currentpose[13][1] + currentpose[17][1])/2
  mid[2] = (currentpose[13][2] + currentpose[17][2])/2
  mid[3] = (currentpose[13][3] + currentpose[17][3])/2
  for i=1,22,1 do
    currentpose[i][1] = currentpose[i][1] -  mid[1]
    currentpose[i][2] = currentpose[i][2] -  mid[2];
    currentpose[i][3] = currentpose[i][3] -  mid[3];
  end
  return currentpose;
end

function SkeletonNode:GetLaplacianExtraPose()
  local currentpose = {}
  for i=1,self.bonecount,1 do
    currentpose[i] = self:GetTPos(i);
  end
  local mid = {}
  mid[1] = (currentpose[13][1] + currentpose[17][1])/2
  mid[2] = (currentpose[13][2] + currentpose[17][2])/2
  mid[3] = (currentpose[13][3] + currentpose[17][3])/2
  local extrapose = {};
  
  for i=1,#self.laplacianextras do
    local midvec = self.laplacianextras[i]:GetComponent(apolloengine.Node.CT_TRANSFORM):GetWorldPosition();
    extrapose[i] = {};
    extrapose[i][1] = midvec:x() - mid[1];
    extrapose[i][2] = midvec:y() - mid[2];
    extrapose[i][3] = midvec:z() - mid[3];
  end
  return extrapose;
end

function SkeletonNode:GetSpineMidPose()
  local currentpose = {}
  for i=1,self.bonecount,1 do
    currentpose[i] = self:GetTPos(i);
  end
  local mid = {}
  mid[1] = (currentpose[13][1] + currentpose[17][1])/2
  mid[2] = (currentpose[13][2] + currentpose[17][2])/2
  mid[3] = (currentpose[13][3] + currentpose[17][3])/2
  local extrapose = {};
  

  local midvec =  self:GetJoint(self.spinebone[1]).trans:GetWorldPosition();
  local spinemidpose = {};
  spinemidpose[1] = midvec:x() - mid[1];
  spinemidpose[2] = midvec:y() - mid[2];
  spinemidpose[3] = midvec:z() - mid[3];

  return spinemidpose;
end


function SkeletonNode:BindPose()
  self.Tdir = {}
  self.Trot = {}
  self.Tpos = {}
  self.TBody = {}
  
  for i=1,22,1 do
    self.Trot[i] = self:GetTRot(i);
    self.Tdir[i] = self:GetTDir(i);
    self.Tpos[i] = self:GetTPos(i)
  end
  
  self.TBody = self:GetBodyPose();
  self.TBodyRot = self:GetBodyRot();
  
  self.RootInitRot = self.trans:GetWorldRotation();
  
  self.BodyStartRotMat = util:ConstractRotationFromVector(self.TBody[2],self.TBody[3]);
  self.BodyStartRotMat:Transpose();
  LOG(self.TBody[2]);
  LOG(self.TBody[3]);
  LOG("AT INIT GO");
  
  local rootjoint = self:GetJoint(self.rootbone);
  local comp = rootjoint.trans;
  self.rootpos = comp:GetWorldPosition();
end


  
function SkeletonNode:GetNameByID(id)
  if self.rigidId2Name then
    for key, value in pairs(self.rigidId2Name) do
      if key == id then
        return value;
      end
    end
  else
    LOG("self.rigidId2Name is nil")
  end
end

function SkeletonNode:GetBodyRot()
  local bonenames = {self.rootbone,self.bonenamemap[21]}
  for i=1,#self.spinebone do
    LOG("SPINE BONE NAME IS "..self.spinebone[i])
    if self.spinebone[i]~=nil then
      bonenames[2 + i] = self.spinebone[i]
    end
  end
  local ret = {}
  for i=1,#bonenames do
    LOG("BONE NAME IS "..bonenames[i])
    local transcom = self:GetJoint(bonenames[i]).trans;
    ret[bonenames[i]] = transcom:GetWorldRotation();
  end
  return ret;
end

function SkeletonNode:GetBodyPose()
  local bonepose = {}
  for i =1, #self.bodypair do
    local first =  self.bodybones[self.bodypair[i][1]]
    local second = self.bodybones[self.bodypair[i][2]]
    
    local fristjoint = self:GetJoint(first);
    local firstworldpose = fristjoint.trans:GetWorldPosition();
   
    local secondjoint = self:GetJoint(second);
    local secondworldpose = secondjoint.trans:GetWorldPosition();
    bonepose[i] = secondworldpose - firstworldpose;
    bonepose[i]:NormalizeSelf();
  end

  bonepose[3] = (bonepose[3]+bonepose[4]/2);
  bonepose[3]:NormalizeSelf();

  bonepose[4] = nil;
  return bonepose;
end

function SkeletonNode:GetTPos(boneidx)

  local name = self.bonenamemap[boneidx];
  if name==nil then
    return nil
  end
  local joint = self:GetJoint(name);
  local comp = joint.trans;
  local pos = comp:GetWorldPosition();
  return {pos:x(),pos:y(),pos:z()};
end


function SkeletonNode:GetTRot(boneidx)

  local name = self.bonenamemap[boneidx];

  if name==nil then
    return nil
  end
  LOG("GetTRot:"..name)
  local joint = self:GetJoint(name);
  local comp = joint.trans;

  return comp:GetWorldRotation();
end

function SkeletonNode:GetTDir(boneidx)
  if self.children[boneidx] == 0 then
    return nil
  end
  local name1 = self.bonenamemap[boneidx];
  local name2 = self.bonenamemap[self.children[boneidx]];
  if name1==nil or name2==nil then
    return nil
  end
  local joint1 = self:GetJoint(name1);
  local comp1 = joint1.trans;
  
  local joint2 = self:GetJoint(name2);
  local comp2 = joint2.trans;
  
  local pos1 = comp1:GetWorldPosition();
  local pos2 = comp2:GetWorldPosition();
  
  local dir =  pos2 - pos1;
  dir:NormalizeSelf();

  return dir;
end

function SkeletonNode:GetJoint(name)
  if self.BoneNodes[name] == nil then
    LOG("GetJoint:no bone named "..name)
    return nil
  end
  local transnode = {}
  transnode = TransNode:cast(transnode);
  transnode.node = self.BoneNodes[name];
  transnode.trans = self.BoneNodes[name]:GetComponent(apolloengine.Node.CT_TRANSFORM);
  return transnode;      
end

function SkeletonNode:GetTargets()
  local target = {}
  for i =1 , #self.children do
    local joint = self:GetJoint(self.bonenamemap[i]);
    local trans = joint.trans;
    --target[self.bonenamemap[i] ] = trans:GetWorldPosition();
    target[self.bonenamemap[i] ] = trans;
  end

  return target,self.RootRot;
end

function SkeletonNode:UpdateSkeleton(pos3d, cameraT,ignorehand,spinerot)
  
  local baserot = nil;
  
  if spinerot==nil then
    local tempppair = {{1,7},{13,17},{1,13},{7,17}}
    local tempdir = {}
    for i=1,#tempppair do
        local fpos = mathfunction.vector3(pos3d[tempppair[i][1] ][1],pos3d[tempppair[i][1] ][2],pos3d[tempppair[i][1] ][3]);
        local spos = mathfunction.vector3(pos3d[tempppair[i][2] ][1],pos3d[tempppair[i][2] ][2],pos3d[tempppair[i][2] ][3]);
        local dir = (spos - fpos);
        dir:NormalizeSelf();
        table.insert(tempdir,dir);
    end
    tempdir[3] = (tempdir[3]+tempdir[4])/2
    tempdir[3]:NormalizeSelf();
    
    
    local currentBodyRotMat = util:ConstractRotationFromVector(tempdir[2],tempdir[3]);
    
    local rotationmat =  self.BodyStartRotMat * currentBodyRotMat ;
    
    local rootrot = rotationmat:ToQuaternion();
    
    self.RootRot = rootrot;

    local rotation1 = mathfunction.Quaternion();
    rotation1:AxisToAxis(self.TBody[1]*rotationmat,tempdir[1]);
    
    local trans2 = self:GetJoint(self.rootbone).trans;
    trans2:SetWorldRotation(self.TBodyRot[self.rootbone]*rootrot);  
    
    if cameraT ~= nil then
      trans2:SetWorldPosition( ( cameraT )/1000 );
    end  


    for i = 1, #self.spinebone do
      local spine_rot = mathfunction.Mathutility:Slerp(mathfunction.Quaternion(),rotation1,i/#self.spinebone);
      local transS1 = self:GetJoint(self.spinebone[i]).trans;
      local rotS1 = self.TBodyRot[self.spinebone[i] ];
      transS1:SetWorldRotation(rotS1*rootrot*spine_rot);
    end
    baserot = rootrot;
  else
    self:UpdateSpineByRot(spinerot);
    baserot = mathfunction.Quaternion(spinerot[1][1],spinerot[1][2],spinerot[1][3],spinerot[1][4]);
  end

  local target = {}
  for i =1 , #self.children-2 do
        
    local joint = self:GetJoint(self.bonenamemap[i]);
    local trans = joint.trans;
    local chidx = self.children[i]
   
    --ignorehand = true;
    
    if  chidx ~= 0 and pos3d[i]~=nil and  pos3d[chidx]~=nil and #pos3d[i]>0 and #pos3d[chidx]>0 then
      
      if (ignorehand==true and (i==3 or i==4 or i==5 or i==9 or i==10 or i==11)) then
        
      else
        local mypos = mathfunction.vector3(pos3d[i][1],pos3d[i][2],pos3d[i][3]);
        
        local chpos = mathfunction.vector3(pos3d[chidx][1],pos3d[chidx][2],pos3d[chidx][3]);
        
        local dir =  chpos - mypos;
        dir:NormalizeSelf();
        
        local rotation = mathfunction.Quaternion();
        rotation:AxisToAxis(self.Tdir[i]*baserot,dir);
        if joint ~= nil then
          local rotated = self.Trot[i]*baserot*rotation
          trans:SetWorldRotation( rotated );
        end
      end
    end
    target[self.bonenamemap[i] ] = trans:GetWorldPosition();
  end
  
  return target,rootrot;
end



function SkeletonNode:LimitTwist(rotation, limit, axis)  
  local fixedRotation = util:RemoveTwist(rotation, axis)
  if limit <= 0 then
    return fixedRotation;
  end
  
  local angle = util:Angle(fixedRotation,rotation);
  --LOG("Twist "..angle)
  if angle < 0.000001 then
    return rotation;
  end
  return mathfunction.Mathutility:Slerp(fixedRotation, rotation, math.min(1.0, limit / angle));
end

function SkeletonNode:LimitHinge(rotation, range, axis)
  -- Get 1 degree of freedom rotation along axis
  local addR = util:Limit1DOF(rotation, axis);
  addR:NormalizeSelf();
  
  local dot = addR.mw;
	local addA = 0.0;
  if dot < 1.0 - 0.000001 then
    addA = math.acos(math.min(math.abs(dot), 1.0)) * 2.0 * 180.0 / 3.1415926;
  end
  
  -- !(x==y==z)
  local secondaryAxis = mathfunction.vector3(axis.mz, axis.mx, axis.my);
  local cross = secondaryAxis:Cross(axis);
  if cross:Dot(secondaryAxis * addR) > 0.0 then
    addA = - addA;
  end
  --LOG("hinge angle ".. addA) 
  --Clamp to limits
  local angle = math.min(math.max(addA, range.mx), range.my);
  local rot = mathfunction.Mathutility:RotateAxis(axis, angle / (180.0 / 3.1415926));
  
  return rot;
end

function SkeletonNode:UpdateArmRot_R(rot3d,pos3d,bLimit)
  local rotElbow   = mathfunction.Quaternion(rot3d[2][1],rot3d[2][2],rot3d[2][3],rot3d[2][4]);
  local rotWrist   = mathfunction.Quaternion(rot3d[3][1],rot3d[3][2],rot3d[3][3],rot3d[3][4]);
  local rotHand    = mathfunction.Quaternion(rot3d[4][1],rot3d[4][2],rot3d[4][3],rot3d[4][4]);
  rotElbow   :  NormalizeSelf();
  rotWrist   :  NormalizeSelf();
  rotHand    :  NormalizeSelf();
  
  local curRotElbow   = self:GetJoint(self.bonenamemap[2]).trans:GetWorldRotation();
  --local curRotWrist   = self:GetJoint(self.bonenamemap[3]).trans:GetWorldRotation();
  --local curRotHand    = self:GetJoint(self.bonenamemap[4]).trans:GetWorldRotation();
  --local curRotHandTip = self:GetJoint(self.bonenamemap[5]).trans:GetWorldRotation();
  --local curRotThumb   = self:GetJoint(self.bonenamemap[6]).trans:GetWorldRotation();
      
  --手肘的世界旋转
  local ydir = mathfunction.vector3(0,1,0)* rotWrist;
  local zdir = mathfunction.vector3(0,0,1)* rotElbow;
  local rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
  local worldRotElbow = self.Trot[2] * (self.modelTposRot[2]:ToQuaternion():Inverse()) * rotmat:ToQuaternion();
  worldRotElbow:NormalizeSelf();
  --手腕的世界旋转    
  ydir = mathfunction.vector3(0,1,0)* rotHand;
  zdir = mathfunction.vector3(0,0,1)* rotWrist;
  rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
  local worldRotWrist = self.Trot[3] * (self.modelTposRot[3]:ToQuaternion():Inverse())*rotmat:ToQuaternion();
  worldRotWrist:NormalizeSelf();
  --手的世界旋转
  ydir = mathfunction.vector3(pos3d[5][1] - pos3d[4][1], pos3d[5][2] - pos3d[4][2],pos3d[5][3] - pos3d[4][3]);
  zdir = mathfunction.vector3(0,0,1)* rotHand;
  rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
  local worldRotHand = self.Trot[4] * (self.modelTposRot[4]:ToQuaternion():Inverse())*rotmat:ToQuaternion();
  worldRotHand:NormalizeSelf();
  
  local localRotWrist = worldRotWrist * curRotElbow:Inverse();
  local localRotHand = worldRotHand * worldRotWrist:Inverse();
  
  if bLimit then
    --手腕旋转矫正和限制
    local localRotWrist_T = self.Trot[3] * self.Trot[2]:Inverse();  
    local fromTo = localRotWrist_T:Inverse() * localRotWrist;
    fromTo:NormalizeSelf();
    --手腕只保留twist,即消除swing
    local axis = self.Tdir[3] * self.Trot[2]:Inverse();
    fromTo = fromTo * util:FromToRotation(axis * fromTo, axis);
    --限制自旋
    local newLocalRotWrist = localRotWrist_T * self:LimitTwist(fromTo, 60, axis);
  
    --手掌旋转矫正和限制
    local localRotHand_T = self.Trot[4] * self.Trot[3]:Inverse();  
    fromTo = localRotHand_T:Inverse() * localRotHand;
    fromTo:NormalizeSelf();
    --[[
    if self.rightHandLastRot == nil then
      self.rightHandLastRot =  mathfunction.Quaternion();
      self.rightHandLastAngle = 0;
    end
    self.rightHandLastRot, self.rightHandLastAngle = self:LimitHinge(fromTo,mathfunction.vector2(-30, 60),
                mathfunction.vector3(0,0,1) * self.Trot[3]:Inverse(),
                self.rightHandLastRot, self.rightHandLastAngle);
    local newLocalRotHand = localRotHand_T * self.rightHandLastRot;
    ]]
    local newLocalRotHand = localRotHand_T * self:LimitHinge(fromTo,mathfunction.vector2(-30, 60),
                mathfunction.vector3(0,0,1) * self.Trot[3]:Inverse());

    if self.lastLocalRotRightHand ~= nil then
      newLocalRotHand = mathfunction.Mathutility:Slerp(self.lastLocalRotRightHand, newLocalRotHand, 0.25 );
    end
    self.lastLocalRotRightHand = newLocalRotHand;
    
    self:GetJoint(self.bonenamemap[3]).trans:SetLocalRotation(newLocalRotWrist);
    self:GetJoint(self.bonenamemap[4]).trans:SetLocalRotation(newLocalRotHand);
  else
    self:GetJoint(self.bonenamemap[3]).trans:SetLocalRotation(localRotWrist);
    self:GetJoint(self.bonenamemap[4]).trans:SetLocalRotation(localRotHand);  
  end
  
  
end

function SkeletonNode:UpdateArmRot_L(rot3d,pos3d,bLimit)
  local rotElbow   = mathfunction.Quaternion(rot3d[8][1],rot3d[8][2],rot3d[8][3],rot3d[8][4]);
  local rotWrist   = mathfunction.Quaternion(rot3d[9][1],rot3d[9][2],rot3d[9][3],rot3d[9][4]);
  local rotHand    = mathfunction.Quaternion(rot3d[10][1],rot3d[10][2],rot3d[10][3],rot3d[10][4]);
  rotElbow   :  NormalizeSelf();
  rotWrist   :  NormalizeSelf();
  rotHand    :  NormalizeSelf();
  
  local curRotElbow   = self:GetJoint(self.bonenamemap[8]).trans:GetWorldRotation();
  --local curRotWrist   = self:GetJoint(self.bonenamemap[9]).trans:GetWorldRotation();
  --local curRotHand    = self:GetJoint(self.bonenamemap[10]).trans:GetWorldRotation();

  
  --手肘的世界旋转
  local ydir = mathfunction.vector3(0,1,0)* rotWrist;
  local zdir = mathfunction.vector3(0,0,1)* rotElbow;
  local rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
  local worldRotElbow = self.Trot[8] * (self.modelTposRot[8]:ToQuaternion():Inverse()) * rotmat:ToQuaternion();
  worldRotElbow:NormalizeSelf();
  --手腕的世界旋转    
  ydir = mathfunction.vector3(0,1,0)* rotHand;
  zdir = mathfunction.vector3(0,0,1)* rotWrist;  
  rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
  local worldRotWrist = self.Trot[9] * (self.modelTposRot[9]:ToQuaternion():Inverse())*rotmat:ToQuaternion();
  worldRotWrist:NormalizeSelf();
  --手的世界旋转
  ydir = mathfunction.vector3(pos3d[11][1] - pos3d[10][1], pos3d[11][2] - pos3d[10][2],pos3d[11][3] - pos3d[10][3]);
  zdir = mathfunction.vector3(0,0,1)* rotHand;
  rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
  local worldRotHand = self.Trot[10] * (self.modelTposRot[10]:ToQuaternion():Inverse())*rotmat:ToQuaternion();
  worldRotHand:NormalizeSelf();
  
  local localRotWrist = worldRotWrist * curRotElbow:Inverse();
  local localRotHand = worldRotHand * worldRotWrist:Inverse();
  
  --滤波
  --[[
  if self.lastLocalRotWrist ~= nil then
    localRotWrist = mathfunction.Mathutility:Slerp(self.lastLocalRotWrist, localRotWrist, 0.25 );
  end
  self.lastLocalRotWrist = localRotWrist;
  if self.lastLocalRotHand ~= nil then
    localRotHand = mathfunction.Mathutility:Slerp(self.lastLocalRotHand, localRotHand, 0.25 );
  end
  self.lastLocalRotHand = localRotHand;
  ]]

  if bLimit then
    --手腕旋转矫正和限制
    --local localRotWrist = curRotWrist * rotation * curRotElbow:Inverse();
    local localRotWrist_T = self.Trot[9] * self.Trot[8]:Inverse();
    local fromTo = localRotWrist_T:Inverse() * localRotWrist;
    fromTo:NormalizeSelf();
    --手腕只保留twist,即消除swing
    local axis = self.Tdir[9] * self.Trot[8]:Inverse();
    fromTo = fromTo * util:FromToRotation(axis * fromTo, axis);
    --限制自旋
    local newLocalRotWrist = localRotWrist_T * self:LimitTwist(fromTo, 60, axis);
  
    --手掌旋转矫正和限制
    local localRotHand_T = self.Trot[10] * self.Trot[9]:Inverse();  
    fromTo = localRotHand_T:Inverse() * localRotHand;
    fromTo:NormalizeSelf();
    --[[
    if self.leftHandLastRot == nil then
      self.leftHandLastRot =  mathfunction.Quaternion();
      self.leftHandLastAngle = 0;
    end
    self.leftHandLastRot, self.leftHandLastAngle = self:LimitHinge(fromTo,mathfunction.vector2(-30, 60),
                mathfunction.vector3(0,0,-1) * self.Trot[9]:Inverse(),
                self.leftHandLastRot, self.leftHandLastAngle);
    local newLocalRotHand = localRotHand_T * self.leftHandLastRot;
    ]]
    local newLocalRotHand = localRotHand_T * self:LimitHinge(fromTo,mathfunction.vector2(-30, 60),
                mathfunction.vector3(0,0,-1) * self.Trot[9]:Inverse());
  
    if self.lastLocalRotLeftHand ~= nil then
      newLocalRotHand = mathfunction.Mathutility:Slerp(self.lastLocalRotLeftHand, newLocalRotHand, 0.25 );
    end
    self.lastLocalRotLeftHand = newLocalRotHand;
  
    self:GetJoint(self.bonenamemap[9]).trans:SetLocalRotation(newLocalRotWrist);
    self:GetJoint(self.bonenamemap[10]).trans:SetLocalRotation(newLocalRotHand);
  else
    self:GetJoint(self.bonenamemap[9]).trans:SetLocalRotation(localRotWrist);
    self:GetJoint(self.bonenamemap[10]).trans:SetLocalRotation(localRotHand);
  end
  
end

function SkeletonNode:FilterArmsTwist()
  if self.lastRotationTable == nil then
    self.lastRotationTable = {};
  end
  local joints = 
    {
      self:GetJoint("R_Shoulder").trans,
      self:GetJoint("R_Elbow").trans,
      self:GetJoint("R_Wrist").trans,
      self:GetJoint("L_Shoulder").trans,
      self:GetJoint("L_Elbow").trans,
      self:GetJoint("L_Wrist").trans,
    }
  local indice = {1,2,3,7,8,9}
  local parentTRot = 
    {
      self.TBodyRot[self.spinebone[3]], 
      self.Trot[1], 
      self.Trot[2], 
      self.TBodyRot[self.spinebone[3]], 
      self.Trot[7], 
      self.Trot[8]
    }
  local alpha = 0.25;
  for i = 1, #indice do    
    local trans = joints[i];
    local curLocalRot = trans:GetLocalRotation();
    local localRot_T = self.Trot[indice[i]] * parentTRot[i]:Inverse();  
    local fromTo = localRot_T:Inverse() * curLocalRot;
    fromTo:NormalizeSelf();
    local axis = self.Tdir[indice[i]] * parentTRot[i]:Inverse();
    local rotSwing = util:FromToRotation(axis, axis*fromTo);
    local rotTwist = curLocalRot * rotSwing:Inverse();
    if self.lastRotationTable[i] ~= nil then
      self.lastRotationTable[i] = mathfunction.Mathutility:Slerp(self.lastRotationTable[i], rotTwist,alpha) ;
      local newRot = self.lastRotationTable[i] * rotSwing;
      trans:SetLocalRotation(newRot);
    else
      self.lastRotationTable[i] = rotTwist;
    end
  end
end

--肘关节会产生相对于肩关节的自旋
function SkeletonNode:UpdateSkeletonFromCurPosition(pos3d)
  local wrist = {1,2,7,8};
  
  for ii =1 , #wrist do
    local i = wrist[ii]
    local joint = self:GetJoint(self.bonenamemap[i]);
    local trans = joint.trans;
    local chidx = self.children[i]
     
    if chidx ~= 0 and pos3d[i]~=nil and  pos3d[chidx]~=nil and #pos3d[i]>0 and #pos3d[chidx]>0 then
      local childjoint = self:GetJoint(self.bonenamemap[chidx]);
      local childtrans = childjoint.trans;
      
      local mypos = mathfunction.vector3(pos3d[i][1],pos3d[i][2],pos3d[i][3]);
      local chpos = mathfunction.vector3(pos3d[chidx][1],pos3d[chidx][2],pos3d[chidx][3]);
      
      local currentmypos = trans:GetWorldPosition();
      local currentchildpos = childtrans:GetWorldPosition();
      
      local dir =  chpos - mypos;
      dir:NormalizeSelf();
      
      local currentdir =  currentchildpos - currentmypos;
      currentdir:NormalizeSelf();
      
      local rotation = mathfunction.Quaternion();
      rotation:AxisToAxis(currentdir,dir);
      if joint ~= nil then
        local rotated = trans:GetWorldRotation()*rotation
        trans:SetWorldRotation( rotated );
      end
    end
  end

end


function SkeletonNode:UpdateSpineByRot(rot3d)
  local spinenames = {"Spinebase","Spinemid","Spineshoulder","Neck","Head"}
  for i=1,5 do
    local idx = i;
    if  rot3d[idx]~=nil then
     
      local kinectrot = mathfunction.Quaternion(rot3d[idx][1],rot3d[idx][2],rot3d[idx][3],rot3d[idx][4]);
      --[[local Zdir = mathfunction.vector3(0,1,0)*kinectrot;
      local Ydir = mathfunction.vector3(0,0,1)*kinectrot;
      
      LOG(spinenames[idx].."dir is")
      LOG(Zdir);
      LOG(Ydir);]]
      
      local worldrot = kinectrot;
      
      local joint = self:GetJoint(spinenames[i]);
      local trans = joint.trans;
      trans:SetWorldRotation(worldrot);
    end
  end
end


function SkeletonNode:UpdateSkeletonByRot(rot3d,pos3d)
  local wrist = {3,9}
  for j=1, #wrist do
    local i=wrist[j];
    local childidx = self.children[i];
    local idx = i;
    if childidx~=0 and self.modelTposRot[childidx]~=nil and self.modelTposRot[idx]~=nil then
      
      local Trotmat = self.modelTposRot[idx];
      local Trotrot = Trotmat:ToQuaternion();
      
      
      local kinectparrot = mathfunction.Quaternion(rot3d[idx][1],rot3d[idx][2],rot3d[idx][3],rot3d[idx][4]);
      local kinectrot = mathfunction.Quaternion(rot3d[childidx][1],rot3d[childidx][2],rot3d[childidx][3],rot3d[childidx][4]);
      kinectparrot:NormalizeSelf();
      kinectrot:NormalizeSelf();
      
      local ydir = mathfunction.vector3(0,1,0)* kinectrot;
      local zdir = mathfunction.vector3(0,0,1)* kinectparrot;
     
      
      local rotmat =  util:ConstractKinectRotationFromVector(zdir,ydir);
      
      local worldrot = Trotrot:Inverse()*rotmat:ToQuaternion();
      
      local joint = self:GetJoint(self.bonenamemap[i]);
      local trans = joint.trans;
      
      
      if i==3 or i==9 then
        local rodrot = trans:GetWorldRotation();
        local rody = mathfunction.vector3(-0.988,0.008,0.151)*(self.Trot[idx]:Inverse()*rodrot);
        local rodz = mathfunction.vector3(0,-1,0)*rodrot;
        
        --LOG("DIR CHECK 1 "..rody:Dot(ydir));
        
        local rodrotmat =  util:ConstractKinectRotationFromVector(rodz,rody);
        rodrotmat:Transpose();
        local twistrot = rodrotmat*rotmat;
        local twistquat = twistrot:ToQuaternion();
        local costheta2 = twistquat:w();
        local theta2 = math.acos(costheta2);
        local sintheta2 = math.sin(theta2);
        local dirvec
        if sintheta2~=0 then
          dirvec = mathfunction.vector3(twistquat:x()/sintheta2,twistquat:y()/sintheta2,twistquat:z()/sintheta2);
        else
          --LOG("ZERO BUG IS COMING")
          dirvec = mathfunction.vector3(0,0,0);
        end
        local dir = dirvec:Dot(ydir);
        --LOG("theta is  "..theta2*2);
        if dir<0 then
          dir = -1
        else
          dir = 1
        end
        
        if theta2*2 >math.pi then
          dir = dir*-1;
          theta2 = math.pi-theta2;
        end
        
        local shoulderjoint = self:GetJoint(self.bonenamemap[i-2]);
        local shouldertrans = shoulderjoint.trans;
        local rot = mathfunction.Quaternion(rot3d[idx-1][1],rot3d[idx-1][2],rot3d[idx-1][3],rot3d[idx-1][4]);
        rot:NormalizeSelf()
        local shoulderrotdir = mathfunction.vector3(0,1,0)*rot;
        local shouldersintheta = math.sin(theta2*(0.33));
        local shouldercostheta = math.cos(theta2*(0.33));
        
        shoulderrotdir = shoulderrotdir*dir;

        local shoulderrot = mathfunction.Quaternion(shoulderrotdir:x()*shouldersintheta,shoulderrotdir:y()*shouldersintheta,shoulderrotdir:z()*shouldersintheta,shouldercostheta);
        
        local eblowjoint = self:GetJoint(self.bonenamemap[i-1]);
        local eblowtrans = eblowjoint.trans;
        local eblowprerotate = eblowtrans:GetWorldRotation();
        
        shouldertrans:SetWorldRotation(shouldertrans:GetWorldRotation()*shoulderrot);
 
        local eblowrotdir = mathfunction.vector3(0,1,0)* mathfunction.Quaternion(rot3d[idx][1],rot3d[idx][2],rot3d[idx][3],rot3d[idx][4])
        eblowrotdir = eblowrotdir*dir;
     
        local eblowsintheta = math.sin(theta2*(0.66));
        local eblowcostheta = math.cos(theta2*(0.66));
       
        local eblowrot = mathfunction.Quaternion(eblowrotdir:x()*eblowsintheta,eblowrotdir:y()*eblowsintheta,eblowrotdir:z()*eblowsintheta,eblowcostheta);
        
        if eblowrot:x()*eblowrot:x()+
            eblowrot:y()*eblowrot:y()+
            eblowrot:z()*eblowrot:z()+
            eblowrot:w()*eblowrot:w()<1.0 then
              
          LOG("QUATERNION SUPER ERROR!");
        end
        
        eblowtrans:SetWorldRotation(eblowprerotate*eblowrot);
  
      end
      
      trans:SetWorldRotation(self.Trot[idx]*worldrot);
      
    end
  end
  
end

--[[
参数说明:
rotUpper:大臂世界旋转
rotFore：小臂世界旋转
upperarm：大臂方向向量
forearm：小臂方向向量
bendAxisUpper：初始大臂外侧朝向 bendAxisUpper * rotUpper = upperarm X forearm
bendAxisFore：初始小臂外侧朝向  bendAxisFore * rotFore = upperarm X forearm
isLeft: (bool)是否为左臂
]]--
function SkeletonNode:EstimateArmRot_2(rotUpper,rotFore,upperarm,forearm,bendAxisUpper,bendAxisFore,isLeft)
  
  local resRotUpper = rotUpper;
  local resRotFore = rotFore;
  
  upperarm : NormalizeSelf();
  forearm : NormalizeSelf();
  local cosine = math.abs(upperarm : Dot(forearm));

  if cosine < 0.996 then
    local elbowRotAxisW0 = upperarm:Cross(forearm); 
    if isLeft then
      elbowRotAxisW0 = mathfunction.vector3(0.0,0.0,0.0) - elbowRotAxisW0;    
    end
    
    local initRotAxis1 = elbowRotAxisW0 * rotUpper:Inverse();
    local rotation0 = mathfunction.Quaternion();
    rotation0:AxisToAxis(bendAxisUpper,initRotAxis1);
    resRotUpper = rotation0 * resRotUpper;

    local fromTo = rotUpper:Inverse() * resRotUpper;
    fromTo = fromTo * util:FromToRotation(upperarm * fromTo, upperarm);
    resRotUpper = rotUpper * fromTo;

    local initRotAxis2 = elbowRotAxisW0 * rotFore : Inverse();
    local rotation1 = mathfunction.Quaternion();
    rotation1:AxisToAxis(bendAxisFore,initRotAxis2);
    resRotFore = rotation1 * resRotFore;

    fromTo = rotFore:Inverse() * resRotFore;
    fromTo = fromTo * util:FromToRotation(forearm * fromTo, forearm);
    resRotFore = rotFore * fromTo;

  else
    return nil,nil;
  end
  
  return resRotUpper,resRotFore;
end
--与前方法系不同实现，都对
function SkeletonNode:EstimateArmRot(rotUpper,rotFore,upperarm,forearm,bendAxisUpper,bendAxisFore,isLeft)

  local resRotUpper = rotUpper;
  local resRotFore = rotFore;
  
  upperarm : NormalizeSelf();
  forearm : NormalizeSelf();
  local cosine = math.abs(upperarm : Dot(forearm));

  if cosine < 0.996 then
    
    local elbowRotAxisW0 = upperarm:Cross(forearm); 
    elbowRotAxisW0:NormalizeSelf();
    if isLeft then
      elbowRotAxisW0 = elbowRotAxisW0 * (-1);    
    end
    
    local curAxis = bendAxisUpper*rotUpper;
    local angle0 = math.acos(curAxis:Dot(elbowRotAxisW0) );
    if upperarm:Dot(curAxis:Cross(elbowRotAxisW0) ) < 0 then
      angle0 = -angle0;
    end
    local rot = mathfunction.Mathutility:RotateAxis(upperarm, angle0);
    resRotUpper = rotUpper * rot;


    local curAxis2 = bendAxisFore * rotFore;
    local angle2 = math.acos(curAxis2:Dot(elbowRotAxisW0));
    if forearm:Dot(curAxis2:Cross(elbowRotAxisW0) ) < 0 then
      angle2 = -angle2;
    end
    
    local rot2 = mathfunction.Mathutility:RotateAxis(forearm, angle2);
    resRotFore = rotFore * rot2;

  else
    return nil,nil;
  end
  
  return resRotUpper,resRotFore;
end

function SkeletonNode:CorrectArmsTwist(localRotLastFrame)
  
  if self.leftElbowAxis0 == nil then
    self.leftElbowAxis = mathfunction.vector3(0.0,1.0,0.0);
    self.leftElbowAxis1 = self.leftElbowAxis * self.Trot[8] : Inverse();
    self.leftElbowAxis0 = self.leftElbowAxis * self.Trot[7] : Inverse();
    self.leftElbowAxis0 : NormalizeSelf()
    self.leftElbowAxis1 : NormalizeSelf()

    self.rightElbowAxis = mathfunction.vector3(0.0,1.0,0.0);
    self.rightElbowAxis1 = self.rightElbowAxis * self.Trot[2] : Inverse();
    self.rightElbowAxis0 = self.rightElbowAxis * self.Trot[1] : Inverse();
    self.rightElbowAxis0 : NormalizeSelf()
    self.rightElbowAxis1 : NormalizeSelf()
    --LOG("rightElbowAxis0 :"..self.rightElbowAxis0:x()..","..self.rightElbowAxis0:y()..","..self.rightElbowAxis0:z());
    --LOG("rightElbowAxis1 :"..self.rightElbowAxis1:x()..","..self.rightElbowAxis1:y()..","..self.rightElbowAxis1:z());
  end

  --left UpperArm/ForeArm/Wrist
  local transUpperL = self:GetJoint(self.bonenamemap[7]).trans;
  local rotUpperL = transUpperL : GetWorldRotation();
  local targetUpperL = transUpperL : GetWorldPosition();
  
  local transForeL = self:GetJoint(self.bonenamemap[8]).trans;
  local rotForeL = transForeL:GetWorldRotation();
  local targetForeL = transForeL : GetWorldPosition();

  local transHandL = self:GetJoint(self.bonenamemap[9]).trans;
  local targetHandL = transHandL : GetWorldPosition();
  
  local upperarmL = targetForeL - targetUpperL;
  local forearmL = targetHandL - targetForeL;

  local resrotUpperL,resrotForeL = self:EstimateArmRot(rotUpperL,rotForeL,upperarmL,forearmL,
    self.leftElbowAxis0,self.leftElbowAxis1,true);
  if resrotUpperL ~= nil and resrotForeL ~= nil then
    --肩部twist限制
    transUpperL : SetWorldRotation( resrotUpperL  );
    local localRotUpper = transUpperL:GetLocalRotation();
    local localRotUpper_T = self.Trot[7] * self.TBodyRot[self.spinebone[3]]:Inverse();  
    local fromTo = localRotUpper_T:Inverse() * localRotUpper;
    fromTo:NormalizeSelf();
    local axis = self.Tdir[7] * self.TBodyRot[self.spinebone[3]]:Inverse();
    local newLocalRotUpper = localRotUpper_T * self:LimitTwist(fromTo, 90, axis);
    transUpperL : SetLocalRotation( newLocalRotUpper  );
  
    --肘关节twist限制
    transForeL : SetWorldRotation( resrotForeL );
    local localRotFore = transForeL:GetLocalRotation();
    local localRotFore_T = self.Trot[8] * self.Trot[7]:Inverse();  
    fromTo = localRotFore_T:Inverse() * localRotFore;
    fromTo:NormalizeSelf();
    axis = self.Tdir[8] * self.Trot[7]:Inverse(); 
    local newLocalRotFore = localRotFore_T * self:LimitTwist(fromTo, 90, axis);
    transForeL : SetLocalRotation( newLocalRotFore );
    
    localRotLastFrame[3] = newLocalRotUpper;
    localRotLastFrame[4] = newLocalRotFore;
  else
    if localRotLastFrame[3]~= nil and localRotLastFrame[4] ~= nil then
      transUpperL : SetLocalRotation( localRotLastFrame[3]  );
      transForeL : SetLocalRotation( localRotLastFrame[4] );    
    end
  end

  --right UpperArm/ForeArm/Wrist
  local transUpperR = self:GetJoint(self.bonenamemap[1]).trans;
  local rotUpperR = transUpperR : GetWorldRotation();
  local targetUpperR = transUpperR : GetWorldPosition();
  
  local transForeR = self:GetJoint(self.bonenamemap[2]).trans;
  local rotForeR = transForeR:GetWorldRotation();
  local targetForeR = transForeR : GetWorldPosition();

  local transHandR = self:GetJoint(self.bonenamemap[3]).trans;
  local targetHandR = transHandR : GetWorldPosition();
  
  local upperarmR = targetForeR - targetUpperR;
  local forearmR = targetHandR - targetForeR;
  
  local resrotUpperR,resrotForeR = self:EstimateArmRot(rotUpperR,rotForeR,upperarmR,forearmR,
    self.rightElbowAxis0,self.rightElbowAxis1,false);

  if resrotUpperR ~= nil and resrotForeR ~= nil then
    --肩关节twist限制
    transUpperR : SetWorldRotation( resrotUpperR  );
    local localRotUpper = transUpperR:GetLocalRotation();
    local localRotUpper_T = self.Trot[1] * self.TBodyRot[self.spinebone[3]]:Inverse();  
    local fromTo = localRotUpper_T:Inverse() * localRotUpper;
    fromTo:NormalizeSelf();
    local axis = self.Tdir[1] * self.TBodyRot[self.spinebone[3]]:Inverse();
    local newLocalRotUpper = localRotUpper_T * self:LimitTwist(fromTo, 90, axis);
    transUpperR : SetLocalRotation( newLocalRotUpper  );
  
    --肘关节twist限制
    transForeR : SetWorldRotation( resrotForeR );
    local localRotFore = transForeR:GetLocalRotation();
    local localRotFore_T = self.Trot[2] * self.Trot[1]:Inverse();  
    fromTo = localRotFore_T:Inverse() * localRotFore;
    fromTo:NormalizeSelf();
    axis = self.Tdir[2] * self.Trot[1]:Inverse(); 
    local newLocalRotFore = localRotFore_T * self:LimitTwist(fromTo, 90, axis);
    transForeR : SetLocalRotation( newLocalRotFore );
    
    localRotLastFrame[1] = newLocalRotUpper;
    localRotLastFrame[2] = newLocalRotFore;
  else
    if localRotLastFrame[1] ~= nil and localRotLastFrame[2] ~= nil then
      transUpperR : SetLocalRotation( localRotLastFrame[1] );
      transForeR : SetLocalRotation( localRotLastFrame[2] );
    end
  end

end

function SkeletonNode:_GetIK()
  if self.ik == nil then
    self.ik = self.BoneNodes[self.rootbone]:GetComponent(apolloengine.Node.CT_KINEMATICS);
    if self.ik == nil then
    LOG("IK is nil")
    end
  end
  return self.ik;
end

function SkeletonNode:_UpdateRootRot(rootrot,cameraT)
  local trans = self:GetJoint(self.rootbone).trans;
  trans:SetWorldRotation(self.TBodyRot[self.rootbone]*rootrot);
  if cameraT ~= nil then
    trans:SetWorldPosition( ( cameraT )/1000 );
  end
end

function SkeletonNode:UpdateIK(rootRot,cameraT,scores)
  if self:_GetIK() == nil then
    return
  end
  self:_UpdateRootRot(rootRot,cameraT)

  if scores ~= nil then
    for i=5,#self.bonenamemap do
      local bone = self.BoneNodes[self.bonenamemap[i]]:GetComponent(apolloengine.Node.CT_TRANSFORM);
      local CONFIDENCE = 1.0;
      if self:_GetIK() : GetJoint(bone:GetStaticID()) ~= nil then
        self:_GetIK():SetConfidence(bone:GetStaticID(), scores[i - 4] > CONFIDENCE);    
      end
    end
  end
  self:_GetIK():UpdateKinematics(0.2,20);  
end

function SkeletonNode:CreateJoint(joint, lx, ux, ly, uy, lz, uz)
  self:CreateJointV3(
    joint,
    mathfunction.vector3(lx, ly, lz),
    mathfunction.vector3(ux, uy, uz));
end

function SkeletonNode:CreateJointV3(name, rotlimlow, rotlimup)
  local node = self:GetJoint(name);
  local joint = self:_GetIK() : GetJoint(node.trans:GetStaticID())
  if joint == nil then
    self:_GetIK():CreateJoint(node.trans);
    joint = self:_GetIK() : GetJoint(node.trans:GetStaticID());
  end
  joint.Constraint.UpperLimit = rotlimup;
  joint.Constraint.LowerLimit = rotlimlow;
end

function SkeletonNode:LinkJoints(j1, j2)
  local joint1 = self:GetJoint(j1).trans;
  local joint2 = self:GetJoint(j2).trans;
  local joint2Key = joint2 : GetStaticID();
  self:_GetIK():SetParent(joint2Key, joint1);
end

function SkeletonNode:SetTarget(targetNodes)
  if self:_GetIK() == nil then
    return
  end
  for i=1, #self.bonenamemap do
    local bone   = self.BoneNodes[self.bonenamemap[i]]:GetComponent(apolloengine.Node.CT_TRANSFORM);
    local target =    targetNodes[self.bonenamemap[i]]:GetComponent(apolloengine.Node.CT_TRANSFORM);
    self:_GetIK():SetTarget(bone:GetStaticID(),target);
  end
end

function SkeletonNode:_SetRootJoint(name)
  local joint = self:GetJoint(name);
  local comp = joint.trans;
  self.rootpos = comp:GetWorldPosition();
end

return SkeletonNode;