local venuscore = require "venuscore"
local venusjson = require "venusjson"
local apolloengine = require "apollocore"
local mathfunction = require "mathfunction"
local libmasquerade = require "libmasque" -- for DeformationComponent init

local Deformer = venuscore.VenusBehavior:extend("DeformerBehavior");

function Deformer:new()
  self.config = ""
  self.inited = false
end

function Deformer:GetConfig()
  return self.config
end

function Deformer:SetConfig(config)
  self.config = config
  self.exact_config = venuscore.IFileSystem:PathAssembly(config)
  self:Init()
end

function Deformer:_OnAwake()
  self:Init()
end

function Deformer:_OnUpdate(delta)
  if self.deform then
    self.deform:Update()
  end
end

function Deformer:BindTargetRender(render)
  self.target = render
  self:Init()
end

function Deformer:Init()
  if self.config == "" or self.target == nil or self.inited then
    return
  end
  
  LOG("exact_config: "..self.exact_config)
  local vtxstream, idxstream = self:load_config(self.exact_config)
  if self.Node:HaveComponent(apolloengine.Node.CT_DEFORM) then
    self.deform = self.Node:GetComponent(apolloengine.Node.CT_DEFORM)
  else
    self.deform = self.Node:CreateComponent(apolloengine.Node.CT_DEFORM)
  end
  self.deform:PushMetadata(
    apolloengine.ReferenceVertexMetadata(apolloengine.VertexBufferEntity.MU_STATIC, vtxstream),
    apolloengine.ReferenceIndicesMetadata(apolloengine.IndicesBufferEntity.MU_STATIC, idxstream))
  self.deform:CreateResource()
  self.deform:SetBindMode(1)
  self.deform:SetBindComponent(self.target)
  self.inited = true
end

function Deformer:load_config(path)
  local rootconfig = venusjson.LaodJsonFile(path)
  if rootconfig.infl == nil then
    return false
  end
  
  local infl = rootconfig.infl
  local vtxstream = apolloengine.VertexStream()
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            3);
  
  local idxStream = apolloengine.IndicesStream()
  idxStream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
  idxStream:ReserveBuffer(#infl * 3);
  
  for i = 1, #infl do
    local data = infl[i]
    local barycentric = mathfunction.vector3(data.barycentric[1], data.barycentric[2], data.barycentric[3])
    vtxstream:PushVertexData(apolloengine.ShaderEntity.ATTRIBUTE_POSITION, barycentric);
    for j = 1, 3 do
      idxStream:PushIndicesData(data.triangle[j]);
    end
  end
  return vtxstream, idxStream
end


Deformer:MemberRegister("config",
  venuscore.ScriptTypes.FilePathType(
    {"json"},    
    Deformer.GetConfig,
    Deformer.SetConfig
))

Deformer:MemberRegister(
  "target",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.RenderComponent:RTTI(),    
    nil,
    function(thiz, value)
        thiz.target = value
        thiz:Init()
    end
  )
)

return Deformer