local apollonode = require "apolloutility.apollonode"
local defined = require "apolloutility.defiend"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"



local renderqueue = {
  CAMERA_LAYER_FIRST = 1 - defined.camera_zero,
  CAMERA_LAYER_LAST = defined.queue_camera_count + 1 - defined.camera_zero,
  CAMERA_LAYER_ZERO = 0
}
local SEQ_INTERVAL = 100

function renderqueue:_CameraFunction(maincamera)
  self._cameras = {};
  self._rendertargets = {};
  self._textures = {};
  local near = maincamera:GetNear();
  local far = maincamera:GetFar();
  local pos = maincamera:GetWorldPosition();
  local lookat = pos + maincamera:GetForward();
  local up = maincamera:GetUp();
  local swaps = 
  {
    apolloengine.RenderTargetEntity.ST_SWAP_E,--标记作用，主要用于底层资源共享
    apolloengine.RenderTargetEntity.ST_SWAP_F
  }
  local index = 1;
  local sequence = maincamera:GetSequence() - defined.queue_camera_count * SEQ_INTERVAL;
  local color = mathfunction.Color(0.5,0.5,0.5,1);
  local reso = apolloengine.Framework:GetResolution()
  LOG("SUNTYLOG: renderqueue:_CameraFunction " .. string.format("reso=%dx%d", reso:x(), reso:y()))

  return function (inputswap)
    local camera = apollonode.CameraNode();--新建摄像机
    self.cameraroot:AttachNode(camera);
    camera:CreateRealCameraProjection(near, far);--设置摄像机
    camera:LookAt(pos, lookat, up);
    camera:SetSequence(sequence);
    camera:SetName("Queue Camera "..tostring(sequence));
    sequence = sequence + SEQ_INTERVAL;
    local swap = inputswap;--apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE;
    if not swap then
      swap = swaps[index];
      index = index + 1 > #swaps and 1 or index + 1;
    end    
    local rt = apolloengine.RenderTargetEntity();--创建一个FBO
    rt:PushMetadata(--设置FBO格式
      apolloengine.RenderTargetMetadata(
        apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
        swap,--标记作用，主要用于底层资源共享
        apolloengine.Framework:GetViewport(),
        apolloengine.Framework:GetResolution()));--分辨率
    local depth = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_DEPTH_STENCIL);
    depth:PushMetadata(
      apolloengine.DepthRenderBufferMetadata(
        swap,
        apolloengine.Framework:GetResolution(),
        apolloengine.TextureEntity.PF_DEPTH24_STENCIL8
      ));
    local tex = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);--增加color0纹理
    tex:PushMetadata(--创建纹理
      apolloengine.TextureRenderMetadata(
        swap,--此处的纹理swap和尺寸必须和rt的相同不然将导致未定义的错误
        apolloengine.Framework:GetResolution()));
    rt:CreateResource();
    camera:AttachRenderTarget(rt);
    table.insert(self._cameras, camera);
    table.insert(self._rendertargets, rt);
    table.insert(self._textures, tex);
    return camera, rt, tex;
  end  
end

function renderqueue:Initialize(maincamera)

  self.rootnode = apollonode.VirtualNode();
  self.rootnode:SetName("Queue Camera Root");
  self.cameraroot = apollonode.VirtualNode();
  self.cameraroot:SetName("Camera Root");
  self.renderroot = apollonode.VirtualNode();
  self.renderroot:SetName("Render Root");
  self.rootnode:AttachNode(self.cameraroot);
  self.rootnode:AttachNode(self.renderroot);

  self.textures = {}
  self.sequences = {}

  self.objs = {};

  self.renders = {}
  self.cameras = {}
  self.activates = {}
  self.linksMap = {}

  self.insertions = {}

  local createCamera = self:_CameraFunction(maincamera);

  --创建queue摄像机
  for i = 1, defined.queue_camera_count do
    local camera, rt, tex = createCamera()
    table.insert(self.textures, tex)
    table.insert(self.cameras, camera)
    table.insert(self.sequences, camera:GetSequence())
    table.insert(self.activates, false)
  end

  table.insert(self.cameras, maincamera)
  table.insert(self.sequences, maincamera:GetSequence())
  table.insert(self.activates, false)


  for i = 1, defined.queue_camera_count do
    local render = apollonode.QuadNode()
    self.renderroot:AttachNode(render);
    -- Since the first camera attaches input render
    -- all the sequence move afterward
    local seq = self:_GetSequence(i + 1 - defined.camera_zero);
    render:CreateResource(defined.blit_material_path, true)
    render:SetSequence(seq)
    render:SetCull(false)
    render:SetName("Queue Quad "..tostring(seq));
    table.insert(self.renders, render)
  end
  
  self.facemaxz = -2000;

  self:Update()
end

function renderqueue:Prev(node)
  node:SetSequence(self:_GetSequence(self.CAMERA_LAYER_FIRST))
end

function renderqueue:Before(node)
  node:SetSequence(self:_GetSequence(self.CAMERA_LAYER_ZERO))
end

function renderqueue:After(node)
  node:SetSequence(self:_GetSequence(self.CAMERA_LAYER_LAST))
end


function renderqueue:Queue(layer, node)
  --LOG("renderqueue:Queue " .. tostring(layer))
  if layer < self.CAMERA_LAYER_FIRST or layer > self.CAMERA_LAYER_LAST then
    error("out of range")
    return
  end
  node:SetSequence(self:_GetSequence(layer))
  self:_Activate(layer)
  self:Update()
end

function renderqueue:Activate(layer)
  if layer < self.CAMERA_LAYER_FIRST or layer > self.CAMERA_LAYER_LAST then
    error("out of range")
    return
  end
  self:_Activate(layer)
  self:Update()
end

function renderqueue:Deactivate(layer)
  if layer < self.CAMERA_LAYER_FIRST or layer > self.CAMERA_LAYER_LAST then
    error("out of range")
    return
  end
  self:_Deactivate(layer)
  self:Update()
end

function renderqueue:GetTexture(layer)
  if layer < self.CAMERA_LAYER_FIRST or layer >= self.CAMERA_LAYER_LAST then
    error("out of range")
    return nil
  end
  return self:_GetTexture(layer)
end

function renderqueue:GetLink(layer)
  if layer <= self.CAMERA_LAYER_FIRST or layer > self.CAMERA_LAYER_LAST then
    error("out of range")
    return nil
  end
  return self.linksMap[layer]
end

--得到对应layer的纹理,0代表before
function renderqueue:_GetSequence(layer)
  return self.sequences[layer + defined.camera_zero]
end

function renderqueue:_GetTexture(layer)
  return self.textures[layer + defined.camera_zero]
end

function renderqueue:_GetRender(layer)
  return self.renders[layer + defined.camera_zero - 1]
end

function renderqueue:GetCamera(layer)
  return self.cameras[layer + defined.camera_zero]
end

function renderqueue:Clear(from, to)--每次更换场景的时候需要将旧的队列清除

  LOG("renderqueue:Clear " .. tostring(from) .. " " .. tostring(to))

  local activatesLen = #self.activates

  if from == nil then
    from = self.CAMERA_LAYER_FIRST
  end

  if to == nil then
    to = self.CAMERA_LAYER_LAST
  end

  for i = from, to do
    self.activates[i + defined.camera_zero] = false
  end

  self:Update()

end



function renderqueue:_Activate(layer)
  self.activates[layer + defined.camera_zero] = true
end

function renderqueue:_Deactivate(layer)
  self.activates[layer + defined.camera_zero] = false
end

function renderqueue:_ConcatSwap()

  local concatActives = {}
  local activatesLen = #self.activates
  local lastActiveLayer = nil
  for i = 1, activatesLen do
    local layer = i - defined.camera_zero
    local active = self.activates[i]
    table.insert(concatActives, active)
    if active then
      if lastActiveLayer ~= nil then
        local diff = layer - lastActiveLayer
        if diff % 2 == 0 then
          concatActives[i - 1] = true
        end
      end
      lastActiveLayer = layer
    end
  end
  return concatActives
end

function renderqueue:Update()

  self:_Activate(self.CAMERA_LAYER_FIRST)
  self:_Activate(self.CAMERA_LAYER_ZERO)
  self:_Activate(self.CAMERA_LAYER_LAST)

  self:_Activate(self.CAMERA_LAYER_FIRST + 3) --编辑器滤镜前一层（锐化）
  self:_Activate(self.CAMERA_LAYER_ZERO + 9) --编辑器贴纸前一层（商汤美妆）
  
  local activates = self:_ConcatSwap()

  local links = {}

  local activatesLen = #activates

  for i = 1, activatesLen do
    local layer = i - defined.camera_zero
    local camera = self:GetCamera(layer)
    local render = self:_GetRender(layer)
    local active = activates[i]
    if render then
      render:SetShow(active);
    end    
    if active then
      camera:Activate()
      table.insert(links, layer)
    else
      camera:Deactivate()
    end
  end

  local linksLen = #links
  self.linksMap = {}

  for i = 2, linksLen do
    local layer = links[i]
    local prevLayer = links[i - 1]
    self.linksMap[layer] = prevLayer
    if prevLayer >= self.CAMERA_LAYER_FIRST then
      local render = self:_GetRender(layer)
      local prevTex = self:_GetTexture(prevLayer)
      render:SetParameter(
              apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
              prevTex)
    end
  end

  local strLinks = "Links: "
  local strSeqs = "Seqs: "
  for i = 1, linksLen do
    strLinks = strLinks .. " " .. tostring(links[i])
    strSeqs = strSeqs .. " " .. tostring(self:_GetSequence(links[i]))
  end
  LOG("SUNTYLOG: " .. strLinks)
  LOG("SUNTYLOG: " .. strSeqs)

  for srcLayer, insertion in pairs(self.insertions) do
    self:_UpdateInsertion(insertion)
    self:_ReLinkInsertion(insertion)
  end

end

function renderqueue:AddObjs(obj)
  table.insert(self.objs, obj);
end

 

function renderqueue:UpdateObjQueue(beginLayer)
  
  table.sort(self.objs, function (a,b)
    return (a:GetRenderOrder()) < (b:GetRenderOrder());
  end);

  
  local layer = self.CAMERA_LAYER_ZERO;
  if beginLayer then
    layer = layer + beginLayer
  end
  local lastRenderOrder = nil
  local len = #self.objs;
  for i = 1, len do

    local obj = self.objs[i]

    local currRenderOrder = obj:GetRenderOrder()


    if lastRenderOrder ~= nil and currRenderOrder - lastRenderOrder > 0.000001 then
      layer = layer + 1
      layer = math.min(layer, self.CAMERA_LAYER_LAST)
    end

    lastRenderOrder = currRenderOrder

    self:Queue(layer, obj)

    obj.renderLayer = layer

  end

  self.objs = {}

end

function renderqueue:SetMaxFaceOrder(maxz)
  self.facemaxz = maxz;
end

function renderqueue:GetMaxFaceOrder()
  return self.facemaxz;
end

function renderqueue:_GetRevLink(layer)
  for i = self.CAMERA_LAYER_FIRST, self.CAMERA_LAYER_LAST do
    if self.linksMap[i] == layer then
      return i
    end
  end
  return nil
end

--layer: the layer after which the cameras are inserting.
--cameras: the inserting cameras
--funcSettingInputTexture: to set the input texture of the cameras. funcSettingInputTexture(texture) -> void
--funcGettingOutputTexture: to get the output texture of the cameras. funcGettingOutputTexture(void) -> texture
--funcOnSequenceChanged: to notify the camera sequence is changed
--return: handle of the cameras, nil for failure
function renderqueue:InsertCameras(
    layer, 
    cameras, 
    funcSettingInputTexture, 
    funcGettingOutputTexture,
    funcOnSequenceChanged)
  
  local insertion = self:_AddInsertion(layer)

  if insertion == nil then
    LOG("renderqueue:InsertCameras failed: invalid layer: " .. tostring(layer))
    return nil
  end
  LOG(string.format(
    "SUNTYLOG: renderqueue:InsertCameras between %s %s",
    tostring(insertion.srcLayer),
    tostring(insertion.sinkLayer)
  ))

  local handle = self:_AddCameraQueue(
    insertion, 
    { unpack(cameras) }, 
    funcSettingInputTexture, 
    funcGettingOutputTexture,
    funcOnSequenceChanged)

  if handle == nil then
    LOG("renderqueue:InsertCameras failed: incorrect cameras or funcs");
    return nil
  end

  self:_ReLinkInsertion(insertion)
  self:_ReSeqInsertion(insertion)

  return handle
end

function renderqueue:RemoveCameras(handle)
  local insertion = self:_RemoveCameraQueue(handle)
  if insertion ~= nil then
    self:_ReLinkInsertion(insertion)
    self:_ReSeqInsertion(insertion)
    self:_RemoveInsertion(insertion)
  end
end

function renderqueue:_AddInsertion(srcLayer)
  if srcLayer == nil then
    return nil
  end
  if srcLayer < self.CAMERA_LAYER_FIRST or srcLayer >= self.CAMERA_LAYER_LAST then
    return nil
  end
  local sinkLayer = self:_GetRevLink(srcLayer)
  if sinkLayer == nil then
    return nil
  end
  if self.insertions[srcLayer] ~= nil then
    return self.insertions[srcLayer]
  end
  local insertion = {
    srcLayer = srcLayer,
    sinkLayer = sinkLayer,
    cameraQueues = {},
    cameraCount = 0,
  }
  self.insertions[srcLayer] = insertion
  return insertion
end

function renderqueue:_RemoveInsertion(insertion)
  if #insertion.cameraQueues == 0 then
    self.insertions[insertion.srcLayer] = nil
  end
end

function renderqueue:_AddCameraQueue(insertion, cameras, fnSetTex, fnGetTex, funOnSeqChanged)
  if cameras == nil or fnSetTex == nil or fnGetTex == nil or funOnSeqChanged == nil then
    return nil
  end
  if insertion.cameraCount + #cameras >= SEQ_INTERVAL then
    return nil
  end

  table.sort(
    cameras,
    function(a, b)
      return a:GetSequence() < b:GetSequence()
    end)

  local cameraQueue = {
    insertion = insertion,
    cameras = cameras,
    fnSetTex = fnSetTex,
    fnGetTex = fnGetTex,
    funOnSeqChanged = funOnSeqChanged
  }
  table.insert(insertion.cameraQueues, cameraQueue)
  insertion.cameraCount = insertion.cameraCount + #cameras
  return cameraQueue
end

function renderqueue:_RemoveCameraQueue(cameraQueue)
  if cameraQueue == nil then
    return nil
  end
  local insertion = cameraQueue.insertion
  if insertion == nil then
    return nil
  end
  local cameraQueueIndex = 0
  for i = 1, #insertion.cameraQueues do
    if insertion.cameraQueues[i] == cameraQueue then
      cameraQueueIndex = i
      break
    end
  end
  if cameraQueueIndex == 0 then
    return nil
  end
  table.remove(insertion.cameraQueues, cameraQueueIndex)
  insertion.cameraCount = insertion.cameraCount - #cameraQueue.cameras
  return insertion
end

function renderqueue:_UpdateInsertion(insertion)
  insertion.sinkLayer = self:_GetRevLink(insertion.srcLayer)
end

function renderqueue:_ReLinkInsertion(insertion)
  local prevCameraQueue = nil
  for i = 1, #insertion.cameraQueues do
    local cameraQueue = insertion.cameraQueues[i]
    if i == 1 then
      cameraQueue.fnSetTex(self:_GetTexture(insertion.srcLayer))
    else
      cameraQueue.fnSetTex(prevCameraQueue.fnGetTex())
    end
    prevCameraQueue = cameraQueue
  end
  if prevCameraQueue ~= nil then
    self:_GetRender(insertion.sinkLayer):SetParameter(
        apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
        prevCameraQueue.fnGetTex())
  else
    self:_GetRender(insertion.sinkLayer):SetParameter(
        apolloengine.ShaderEntity.TEXTURE_DIFFUSE,
        self:_GetTexture(insertion.srcLayer))
  end
end

function renderqueue:_ReSeqInsertion(insertion)
  local seq = self:_GetSequence(insertion.srcLayer)
  for i = 1, #insertion.cameraQueues do
    local cameraQueue = insertion.cameraQueues[i]
    for j = 1, #cameraQueue.cameras do
      local camera = cameraQueue.cameras[j]
      seq = seq + 1
      camera:SetSequence(seq)
    end
    cameraQueue.funOnSeqChanged();
  end
  
end

return renderqueue;
