local BlueNode = require "bluecore.bluenode"
local BlueUtility = require "bluecore.blueutility"
local BD = require "bluecore.bluedefined"

local SetVariableTemplate = BlueNode:extend();

SetVariableTemplate.myExtend = SetVariableTemplate.extend

-- 模板节点的特化(根据外部提供的特化信息)
-- e.g spec_data_tbl = { "ref_variable_name" = "enableSetPosition" }
--     SetVariableTemplate:extend(graph, spec_data_tbl , rtti = nil)
function SetVariableTemplate:extend(graph, specDataTbl)

  -- 不是新的类型,不用在RttiManager记录, 但是属于这个蓝图类中在编辑状态动态生成的节点类
  local clazz = SetVariableTemplate.myExtend(self)

  -- 这个特化节点类所属的蓝图
  clazz.graph = graph

  -- 目前extend类序列化出去的是BlueNode但是为空table,参考MemberIterator的实现是从metatable中获取__members
  -- BaseClass SpecData 实际作为实例的属性序列化出去
  -- 作为这个动态生成的节点类的反序列化信息
  -- spec_data其实更像是类的信息 而不是成员
  clazz[BD.BaseClass]= "bluecore.variable.setvariable"
  clazz[BD.SpecData] = specDataTbl

  local refVarName = clazz[BD.SpecData][BD.RefVarName]

  clazz:MemberRegister(BD.SpecData)
  clazz:MemberRegister(BD.BaseClass)
  clazz:SetFunctionName("Set "..tostring(refVarName)); -- default'name is idName

  assert(refVarName ~= nil, "[SetVariableTemplate] RefVarName nil ");

  local varinfo = graph:GetVariableInfo(refVarName);
  if varinfo then
    clazz:_GenerateDataPin(varinfo);
    clazz:SetFunctionName("Set "..varinfo:GetDisplayName());
  elseif _KRATOSEDITOR then
    WARNING("[extend] no rtti info(during deserialization, it's ok)");
  end
        
  clazz:RegisterExecInput(1, "exec", "执行");
  clazz:RegisterExecOutput(1, "exec", "执行");

  -- Class持有这个Class实例的弱引用
  clazz.classNodeMap = {}
  setmetatable(clazz.classNodeMap, {__mode='v'})
  clazz.new = function(node, ...)
    clazz.super.new(node, ...)
    table.insert(clazz.classNodeMap, node); -- 弱引用
  end

  return clazz ;
end

function SetVariableTemplate:_OnDuplicate() -- TODO FIXME(hhl) EditorCreate

  local luaTbl = self[BD.LUA_CLASS_PATH];
  local bluePrintNode = luaTbl(self.graph); 
  bluePrintNode[BD.LUA_CLASS_PATH] = luaTbl;
  bluePrintNode:EditorCreate();
  return bluePrintNode;

end


-- 重构考虑作为模板节点类的基类
function SetVariableTemplate:GetSpecData()
  return self[BD.SpecData];
end

function SetVariableTemplate:GetTemplateClass()
  return self[BD.BaseClass]
end

-- 动态节点类 处理反序列化完成
function SetVariableTemplate:_OnClassDeserializePost()
  -- 重新设置rtti类型
  local clazz = self;

  if not next(clazz.infoTable[BD.PIN_DATA_INPUT]) then
    -- 输入引脚还没有确定
    local idName = clazz[BD.SpecData][BD.RefVarName]
    if _KRATOSEDITOR then WARNING("[_OnClassDeserializePost]   "..tostring(idName)) end
    local varinfo = self.graph:GetVariableInfo(idName);
    if varinfo then
      clazz:_GenerateDataPin(varinfo);
      clazz:SetFunctionName("Set "..tostring(varinfo:GetDisplayName()));
    else
      assert("[_OnClassDeserializePost] no rtti info");
    end
  else
    LOG("[_OnDeserializePost] done yet "..tostring(clazz[BD.SpecData][BD.RefVarName]))
  end
end

function SetVariableTemplate:onPropertyChange(publisher, property, oldValue, newValue)

  local clazz = self ;

  if property == BD.VarProp.Name then
    local displayName = tostring(newValue)
    clazz:SetFunctionName("Set "..displayName);
    WARNING("[onPropertyChange] displayName " .. displayName)
  elseif property == BD.VarProp.BaseType or property == BD.VarProp.IsArray then
  else
    assert(false, "[onPropertyChange] missing property change ");
  end

  local varinfo = clazz.graph:GetVariableInfo(clazz[BD.SpecData][BD.RefVarName]);
  if varinfo then
    clazz:_GenerateDataPin(varinfo); -- 重新生成数据引脚
  else
    assert("[onPropertyChange] no rtti info");
  end

  -- 类型修改 节点实例字面值重新修改
  if property == BD.VarProp.BaseType or property == BD.VarProp.IsArray then
    local rtti = varinfo:GetRtti()
    for _, node in pairs(clazz.classNodeMap) do
      WARNING("[onPropertyChange] node "..tostring(node.uid).." change literal")
      local literal
      if rtti:isArray() then
        literal = {}
      else
        literal = BlueUtility:GetDefaultByRtti(rtti)
      end
      node.inputs[1].literal = literal
    end
  end

end

-- 反序列化'节点实例'之后重新设置LUA_CLASS_PATH
function SetVariableTemplate:_OnDeserialize()
  self[BD.LUA_CLASS_PATH] = getmetatable(self)
  self[BD.SpecData] = nil
end

function SetVariableTemplate:_GenerateDataPin(varInfo)

  local clazz = self ;

  local refVarName = tostring(clazz[BD.SpecData][BD.RefVarName])

  if _KRATOSEDITOR then
    LOG("[_GenerateDataPin] "..refVarName..":"  .. varInfo.base_rtti:GetTypeName()..","..tostring(varInfo.is_array))
  end

  local rtti = varInfo:GetRtti()
  local defaultValue
  if rtti:isArray() then
    defaultValue = {} -- 数组类型的变量 default = {ArrayItemWrap(), ArrayItemWrap() }
  else
    defaultValue = BlueUtility:GenerateLiteral(rtti, varInfo:GetDefault())
  end


  local displayName = varInfo:GetDisplayName()

  clazz:UnRegisterInput(1);
  clazz:RegisterInput(1, rtti, displayName, "Set "..displayName, defaultValue);

  clazz:UnRegisterOutput(1);
  clazz:RegisterOutput(1, rtti, "result", "result");
end

function SetVariableTemplate:_OnCompile()
  if BD.STRICT_MODE then
    self.compileResult = nil
    local myRtti = self.infoTable[BD.PIN_DATA_INPUT][1].rtti
    for _, link in pairs(self.inputs[1].links) do
      local pinInfo = self.graph:GetPinInfoById(link.pinUid);
      local node = self.graph:GetNode(link.nodeUid);
      if not pinInfo.rtti:isTypeOrDriverType(myRtti) then
        self.compileResult = "input pin 1 not match with ".. node:GetName().. " of pin" .. pinInfo.argId
        return false
      end
    end

    myRtti = self.infoTable[BD.PIN_DATA_OUTPUT][1].rtti
    for _, link in pairs(self.outputs[1].links) do -- 参考 getvariable.lua
      local pinInfo = self.graph:GetPinInfoById(link.pinUid);
      local node = self.graph:GetNode(link.nodeUid);
      if not myRtti:isTypeOrDriverType(pinInfo.rtti) then
        local flag = false
        if myRtti:isArray() then
          local compatible = node:IsInputArrayCompatible(pinInfo); -- refer to _DoInputDependency
          if compatible then
            local elemType = myRtti:GetElementType();
            flag = elemType:isTypeOrDriverType(pinInfo.rtti) ;
          end
        end
        if not flag then
          self.compileResult = "output pin 1 not match with ".. node:GetName().. " of pin" .. pinInfo.argId
          return false
        end
      end
    end
  end

  return true ;
end

function SetVariableTemplate:GetVariableName()
  return self[BD.SpecData][BD.RefVarName]
end

function SetVariableTemplate:GetRefBluegraph()
  return self.graph --
end

function SetVariableTemplate:IsVarArrayEd()
  local varInfo = self.graph:GetVariableInfo(self[BD.SpecData][BD.RefVarName]);
  return varInfo:IsArray();
end

---- 游戏运行模式
function SetVariableTemplate:new(...)
  SetVariableTemplate.super.new(self, ...)
end

function SetVariableTemplate:_OnUpdate(args)
  local varName = self[BD.SpecData][BD.RefVarName]
  self.graph[varName] = args[1]
  return self.graph[varName]
end

function SetVariableTemplate:GetOutputByIndex(index)
  return self.graph[self[BD.SpecData][BD.RefVarName]];
end

---- 编辑器模式

if _KRATOSEDITOR then
  function SetVariableTemplate:_OnUpdateByEditor()
      self:_OnCompile() -- 实时做编译/检查
  end
end

SetVariableTemplate:SetFunctionType(BD.VARIABLE_NODE);

SetVariableTemplate:SetFunctionName("Set")

return SetVariableTemplate ;

