local BlueFunctionNode = require "bluecore.functioncore.blue_func_node"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local AE = require "apolloengine"
local BR  = require "bluecore.core.blue_rtti"
local BlueprintSelf = require "bluecore.core.blueprint_self"

local FuncCallNode = BlueFunctionNode:extend();
FuncCallNode:MemberRegister("_clsId");
FuncCallNode:MemberRegister("ftnUid");
FuncCallNode:MemberRegister("libUid");
FuncCallNode:MemberRegister("ftnType");
FuncCallNode:MemberRegister("eventId");
FuncCallNode:MemberRegister("dispId")
FuncCallNode:MemberRegister("refName")

FuncCallNode:MemberRegister(BD.INPUTS_INFO);
FuncCallNode:MemberRegister(BD.OUTPUTS_INFO);

FuncCallNode.INVALID_NAME = "n/a"

-- inputs input extend   扩展 增加 argUid 属性
-- outputs output extend 扩展 增加 argUid 属性

-- graph node所属的blueprint/bluefunction/bluescene
function FuncCallNode:new(graph)
  FuncCallNode.super.new(self, graph)

  -- 从类的信息 转为 实例的属性 需要在new中  
  local originExecInputsInfo = self.infoTable[BD.PIN_EXEC_INPUT]
  local originExecOutputsInfo = self.infoTable[BD.PIN_EXEC_OUTPUT]

  self.infoTable = {}
  self.infoTable[BD.PIN_DATA_INPUT] = {}
  self.infoTable[BD.PIN_DATA_OUTPUT] = {}
  self.infoTable[BD.PIN_EXEC_INPUT]  = originExecInputsInfo
  self.infoTable[BD.PIN_EXEC_OUTPUT] = originExecOutputsInfo

  self[BD.INPUTS_INFO] = self.infoTable[BD.PIN_DATA_INPUT];
  self[BD.OUTPUTS_INFO]= self.infoTable[BD.PIN_DATA_OUTPUT];

  self.refName = FuncCallNode.INVALID_NAME

  self._failReason = nil ;
end

function FuncCallNode:_UpdateNodeType()
  if self.ftnType == BD.FuncType.LOCAL then
    self.nodeType = BD.FUNCTION_CALL_NODE
    self._OnUpdate = FuncCallNode._OnUpdateForLocal
    self._OnCreateWithEditor = FuncCallNode._OnCreateWithEditorForLocal
    self._GetInput = FuncCallNode._GetInputForThis
  elseif self.ftnType == BD.FuncType.GLOBAL then
    self.nodeType = BD.GLOBAL_CALL_NODE
    self._OnUpdate = FuncCallNode._OnUpdateForGlobal
    self._OnCreateWithEditor = FuncCallNode._OnCreateWithEditorForGlobal
    self._GetInput = FuncCallNode.super._GetInput -- Use baseClass's _GetInput
  elseif self.ftnType == BD.FuncType.EVENT then
    self.nodeType = BD.EVENT_CALL_NODE
    self._OnUpdate = FuncCallNode._OnUpdateForEvent
    self._OnCreateWithEditor = FuncCallNode._OnCreateWithEditorForEvent
    self._GetInput = FuncCallNode._GetInputForThis
  elseif self.ftnType == BD.FuncType.DISPATCHER then
    self.nodeType = BD.DISPATCHER_CALL_NODE
    self._OnUpdate = FuncCallNode._OnUpdateForDispatcher
    self._OnCreateWithEditor = FuncCallNode._OnCreateWithEditorForDispatcher
    self._GetInput = FuncCallNode._GetInputForThis
  else
    error("_UpdateNodeType not match, ftnType = "..tostring(self.ftnType))
  end
end

function FuncCallNode:_OnBeginCreateEd(ftnType, classId, ftnUid, libUid, eventId, dispatcherId)

  -- 调用指定类classId的成员函数ftnUid
  self._clsId = classId;
  self.ftnUid = ftnUid
  self.libUid = libUid
  self.dispId = dispatcherId
  self.eventId = eventId  -- may nil -- eventId is nodeId for CustomEvent
  self.ftnType = ftnType
  self:_UpdateNodeType();                      -- 更新成具体的节点类型

  if eventId ~= nil then self.eventName = BD.EVENT_PREFIX..self.eventId end -- 调用'自定义事件'
  self.globalFunc = setmetatable({} , { __mode = "v"})    -- 全局函数

  LOG("FuncCallNode:new _clsId "..tostring(classId)..", ftnUid "..tostring(ftnUid)..", libUid "..tostring(libUid));

end

--function FuncCallNode:_OnDuplicate()
--
--  local luaPath = self[BD.LUA_CLASS_PATH];
--  local clazz = require(luaPath);
--  local bluePrintNode = clazz(self.graph, self.funcType, self.ftnUid, self.libUid, self.comp); --new实例
--  bluePrintNode[BD.LUA_CLASS_PATH] = luaPath;
--  return bluePrintNode;
--  -- body
--end

-- 第一个输入参数默认是蓝图类类型
function FuncCallNode:_FirstThis()
  return self.ftnType ~= BD.FuncType.GLOBAL
end

-- 1.用户可知调用节点-参数个数和类型修改 2.prefab实例不应该 修改 '调用函数' 的序列化数据
function FuncCallNode:_OnEditorPinAdd(isInput, reflectInfo)
  LOG("FuncCallNode pin add isInput:"..tostring(isInput))
  -- keep empty, don't modify serialization data, especially,  prefab instance
end

function FuncCallNode:_OnEditorPinDelete(isInput, idx)
  LOG("FuncCallNode pin del idx:" ..tostring(idx)..", isInput:"..tostring(isInput))
end

function FuncCallNode:_OnInfoChange(reflectIdx, type, val, isInput)
  LOG("FuncCallNode change idx:" ..tostring(reflectIdx)..", type:"..tostring(type)..", isInput:"..tostring(isInput))
end

function FuncCallNode:NameChanged(oldName, newName)

end

function FuncCallNode:_PinAdd(isInput, argInfo)
  if isInput then
    self:AddInput(argInfo);
  else
    self:AddOutput(argInfo);
  end
end

-- isInput: input parameter or output
-- idx: parameter index, if isInput = true, ignore first input(*THIS*)
function FuncCallNode:_PinDelete(isInput, idx)
  if isInput then
    if self:_FirstThis() then idx = idx + 1 end
    local toDelInsPin = self.inputs[idx];
    self.graph:DeleteNodePin(toDelInsPin.uid);
    self:RemoveInput(idx);
  else
    local toDelInsPin = self.outputs[idx];
    self.graph:DeleteNodePin(toDelInsPin.uid);
    self:RemoveOutput(idx);
  end
end



-- override
function FuncCallNode:AddInput(reflectInfo)
  local pin = FuncCallNode.super.AddInput(self,reflectInfo)
  pin.argUid = reflectInfo.argUid -- 增加保存参数唯一号
  if pin.argUid == nil then error("pin.argUid == nil"); end
  self.graph:AddBlueNodePinToAll(pin);
end

-- override
function FuncCallNode:AddOutput(reflectInfo)
  local pin = FuncCallNode.super.AddOutput(self,reflectInfo)
  pin.argUid = reflectInfo.argUid -- 增加保存参数唯一号
  if pin.argUid == nil then error("pin.argUid == nil"); end
  self.graph:AddBlueNodePinToAll(pin);
end

function BlueFunctionNode:_AddDefaultInputEd(rtti, name, tips)
  local newArgId = 1
  for _ , __ in pairs(self.infoTable[BD.PIN_DATA_INPUT]) do
    newArgId = newArgId + 1
  end
  local pinInfo = self:DynamicRegisterInput(newArgId, rtti, name, tips, BlueprintSelf());
  local inputPin = {
    uid     = self.graph and self.graph:_NextID() or -1,
    nodeUid = self.uid,
    pinType = BD.PIN_DATA_INPUT,
    argId   = pinInfo.argId,
    links   = {},
    useLiteral = true ,
    literal = BU:GenerateLiteral(pinInfo.rtti, pinInfo.default),
    delAble = false
  }
  table.insert(self.inputs, inputPin);
  return inputPin;
end

function FuncCallNode:_OnCreateWithEditorForLocal()

  if self._clsId == nil or self._clsId == 0 then
    self._failReason = "Class id is invalid [Local]"
  elseif self.ftnUid == nil then
    self._failReason = "Function id is invalid"
  else
    local clsInfo = BR.GetClassesByIdEd(self._clsId);
    local ftnList = clsInfo:GetBlueFtnList()
    local funcInfo = ftnList[self.ftnUid]

    local blueType = BR.GetRttiByClsId(self._clsId);

    -- FuncInfo 实现了基本和BlueFunction一样的获取函数信息接口
    local inputsInfo = funcInfo:GetInputInfo();
    local outputsInfo = funcInfo:GetOutputInfo();

    self:_NameChanged(FuncCallNode.INVALID_NAME, funcInfo:GetName())

    self:_AddDefaultInputEd(blueType, "class", "BlueprintClass" );

    if inputsInfo ~= nil then
      for _, info in pairs(inputsInfo) do
        self:AddInput(info);
      end
    end

    if outputsInfo ~= nil then
      for _, info in pairs(outputsInfo) do
        self:AddOutput(info);
      end
    end
    funcInfo:AddRefactor(self);
  end
  if self._failReason ~= nil then
    ERROR(self._failReason);
  end
end

function FuncCallNode:_OnCreateWithEditorForGlobal()
  local node = self.graph[BD.THIS]
  local scene = node:GetHostScene();
  local rootNode = scene:GetRootNode();

  local blueprintComp = rootNode:GetComponent(AE.Node.CT_BLUEPRINT);
  if blueprintComp then
    local blueScene = BU.GetIns(blueprintComp);
    if blueScene then

      local libs = blueScene:GetLibs()
      local lib = libs[self.libUid]
      local ftns = lib:GetLibFtns()
      local ftn = ftns[self.ftnUid]

      local inputsInfo = ftn:GetInputInfo();
      local outputsInfo = ftn:GetOutputInfo();

      self:_NameChanged(FuncCallNode.INVALID_NAME, ftn:GetName())

      -- 全局函数 第一个参数不用class_id
      if inputsInfo ~= nil then
        for _, info in pairs(inputsInfo) do
          self:AddInput(info);
        end
      end

      if outputsInfo ~= nil then
        for _, info in pairs(outputsInfo) do
          self:AddOutput(info);
        end
      end

      self.globalFunc = setmetatable({} , { __mode = "v"})
      self.globalFunc[1] = ftn

      ftn:AddRefactor(self);
    else
      self._failReason = "blueScene in rootNode NOT FOUND"
    end
  else
    self._failReason = "prefab/blueprint is deleted"
  end

  if self._failReason ~= nil then
    ERROR(self._failReason);
  end
end

function FuncCallNode:_OnCreateWithEditorForEvent()

  if self._clsId == nil or self._clsId == 0 then
    self._failReason = "Class id is invalid [Event]"
  elseif self.eventName == nil then
    self._failReason = "Event id is invalid"
  else
    local clsInfo = BR.GetClassesByIdEd(self._clsId);
    local customEvents = clsInfo:GetCustomEventNodes();
    local customEvent = customEvents[self.eventId]

    local blueType = BR.GetRttiByClsId(self._clsId);

    -- FuncInfo 实现了基本和BlueFunction一样的获取函数信息接口
    local inputsInfo = customEvent:GetInputInfo();

    self:_NameChanged(FuncCallNode.INVALID_NAME, customEvent:GetName());

    self:_AddDefaultInputEd(blueType, "class", "BlueprintClass" );

    if inputsInfo ~= nil then
      for _, info in pairs(inputsInfo) do
        self:AddInput(info);
      end
    end

    customEvent:AddRefactor(self);
  end

  if self._failReason ~= nil then
    ERROR(self._failReason);
  end

end

function FuncCallNode:_OnCreateWithEditorForDispatcher()

  if self._clsId == nil or self._clsId == 0 then
    self._failReason = "Class id is invalid [Dispatcher]"
  elseif self.dispId == nil then
    self._failReason = "Dispatcher id is invalid"
  else
    local clsInfo = BR.GetClassesByIdEd(self._clsId);
    local dispatchers = clsInfo:GetEventDispatchers()
    local dispatcherInfo = dispatchers[self.dispId]

    local blueType = BR.GetRttiByClsId(self._clsId);

    local inputsInfo = dispatcherInfo:GetInputInfo();

    self:_NameChanged(FuncCallNode.INVALID_NAME,  dispatcherInfo:GetName())

    self:_AddDefaultInputEd(blueType, "class", "BlueprintClass" );

    if inputsInfo ~= nil then
      for _, info in pairs(inputsInfo) do
        self:AddInput(info);
      end
    end

    dispatcherInfo:AddRefactor(self);
  end

  if self._failReason ~= nil then
    ERROR(self._failReason);
  end

end

-- implement
function FuncCallNode:_OnCreateWithEditor()
  error("_OnUpdate Not Callable, it should NOT call _OnUpdate")
end


function FuncCallNode:_GetInputForThis()
  self.inputArgs = {}
  local passArgs = {}
  local arg
  for argIdx, input in pairs(self.dependInputs) do
    input[1]:Update();
    arg = input[1]:GetOutputByIndex(input[2]);
    self.inputArgs[argIdx] = arg
    if argIdx ~= 1 then passArgs[argIdx - 1] =  arg end
  end
  return passArgs
end

function FuncCallNode:_OnUpdateForLocal(args)
  local variation =  self.inputArgs[1]
  if variation.GetTypeName then
    local func = variation.blueFunctionList[ self.ftnUid ];
    func["_OnUpdate"](func, args);
    return func:GetFunctionOutput();
  else
    local index, blueprint
    if next(variation, index) then
      while true do
        index, blueprint = next(variation, index)
        local func = blueprint.blueFunctionList[ self.ftnUid ];
        func["_OnUpdate"](func, args);
        if next(variation, index) == nil then
          return func:GetFunctionOutput(); -- just return last call result, same with unreal
        end
      end -- if empty array, no result, take care of this
    end
  end
end

function FuncCallNode:_OnUpdateForGlobal(args)
  local func = self.globalFunc[1]
  if func ~= nil then
    func["_OnUpdate"](func, args)
  end
  return func:GetFunctionOutput()
end

function FuncCallNode:_OnUpdateForEvent(args)

  local variation = self.inputArgs[1] -- should be blueprint or bluescene, not bluefunction

  if variation.GetTypeName then
    local blueprint = variation
    blueprint:SetCustomEventRunLabel(true, self.eventName);
    blueprint[self.eventName](blueprint, args);
    blueprint:SetCustomEventRunLabel(false);
  else
    for _, blueprint in pairs(variation) do
      blueprint:SetCustomEventRunLabel(true, self.eventName);
      blueprint[self.eventName](blueprint, args);
      blueprint:SetCustomEventRunLabel(false);
    end
  end
  -- nothing return for event call
end


function FuncCallNode:_OnUpdateForDispatcher(args)

  local variation = self.inputArgs[1] -- should be blueprint or bluescene, not bluefunction

  if variation.GetTypeName then
    local blueprint = variation
    local dispatcher = blueprint:GetEventDispatcherByUid(self.dispId)
    dispatcher:Dispatch(args);
  else
    for _, blueprint in pairs(variation) do
      local dispatcher = blueprint:GetEventDispatcherByUid(self.dispId)
      dispatcher:Dispatch(args);
    end
  end
  -- nothing return for dispatcher call
end

-- implement,
function FuncCallNode:_OnUpdate(args)
  error("_OnUpdate Not Callable, it should NOT call _OnUpdate");
end

-- override
function FuncCallNode:IsInputArrayCompatible(pinInfo)
  -- 只有调用分发器和自定义事件call(没有返回值) 才支持数组/集合输入
  if (self.ftnType == BD.FuncType.EVENT or self.ftnType == BD.FuncType.DISPATCHER)
          and pinInfo.argId == 1 and pinInfo.pinType == BD.PIN_DATA_INPUT then
    return true -- 第一个数据引脚是BlueprintType
  end
  return false
end

--暂时不允许复制
function FuncCallNode:IsAllowDuplicate()
  return false
end

-- override
function FuncCallNode:IsInputArrayCompatible(pinInfo)
  if FuncCallNode:_FirstThis() and pinInfo.argId == 1 and pinInfo.pinType == BD.PIN_DATA_INPUT then
    return true -- 第一个数据引脚是BlueprintType
  end
  return false
end

-- 只有显示蓝图的时候，才会调用_Sync同步，并提示修改的地方
-- 不管是蓝图类还是Prefab类 和全局函数 , 在没有显示之前，都不会自动修改
function FuncCallNode:SyncEd()

  local inputsInfo = {}
  local outputsInfo = {}

  if self:_FirstThis() then

    local classInfo = BR.GetClassesByIdEd(self._clsId);

    if classInfo ~= nil then

      if self.ftnType == BD.FuncType.LOCAL then

        local ftnList = classInfo:GetBlueFtnList()
        local blueFtn = ftnList[self.ftnUid]
        if blueFtn ~= nil then

          local name =  blueFtn:GetName()
          if self.refName ~= name then
            self:AppendModifyMessage(string.format("function name change %s -> %s ", self.refName, name))
            self:_NameChanged(self.refName, name)
          end

          inputsInfo = blueFtn:GetInputInfo();
          outputsInfo = blueFtn:GetOutputInfo();

        else
          self._failReason = string.format("function %s is deleted in class %s", self.refName, classInfo:GetClassName())
          return
        end

      elseif self.ftnType == BD.FuncType.DISPATCHER then

        local dispatchers = classInfo:GetEventDispatchers()
        local dispatcherInfo = dispatchers[self.dispId]

        if dispatcherInfo ~= nil then
          local name =  dispatcherInfo:GetName()
          if self.refName ~= name then
            self:AppendModifyMessage(string.format("dispatcher name change %s -> %s ", self.refName, name));
            self:_NameChanged(self.refName, name);
          end

          inputsInfo = dispatcherInfo:GetInputInfo();
        else
          self._failReason = string.format("dispatcher %s is deleted in class %s", self.refName, classInfo:GetClassName())
          return
        end

      elseif self.ftnType == BD.FuncType.EVENT then

        --if self.eventId ~= nil then self.eventName = BD.EVENT_PREFIX..self.eventId end -- FIXME(放在反序列化)

        local eventNodes = classInfo:GetCustomEventNodes()
        local customEvent = eventNodes[self.eventId]

        if customEvent ~= nil then

          local name = customEvent:GetName()
          if self.refName ~= name then
            self:AppendModifyMessage(string.format("dispatcher name change %s -> %s ", self.refName, name));
            self:_NameChanged(self.refName, name)
          end

          inputsInfo = customEvent:GetInputInfo();

        else
          self._failReason = string.format("custom event %s is deleted in class %s", self.refName, classInfo:GetClassName())
          return
        end
      end

    else
      self._failReason = "prefab/blueprint is deleted"  -- 类被删除了 读取不到类的信息
      return
    end
  else -- global function call
    local node = self.graph[BD.THIS]
    local scene = node:GetHostScene();
    local rootNode = scene:GetRootNode();

    self.globalFunc = setmetatable({} , { __mode = "v"})

    local blueprintComp = rootNode:GetComponent(AE.Node.CT_BLUEPRINT);
    if blueprintComp then
      local blueScene = BU.GetIns(blueprintComp);
      if blueScene then

        local libs = blueScene:GetLibs()
        local lib = libs[self.libUid]
        if lib ~= nil then

          local ftns = lib:GetLibFtns()
          local ftn = ftns[self.ftnUid]
          if ftn ~= nil then

            self.globalFunc[1] = ftn

            local name = ftn:GetName() ;
            if self.refName ~= name then
              self:AppendModifyMessage(string.format("global function name change %s -> %s ", self.refName, name));
              self:_NameChanged(self.refName, name)
            end
            inputsInfo = ftn:GetInputInfo();
            outputsInfo = ftn:GetOutputInfo();
          else
            self._failReason = "global function is deleted"
          end
        else
          self._failReason = "library is deleted"
        end
      else
        self._failReason = "blue scene is deleted"
      end
    else
      self._failReason = "blue scene's component is deleted"
    end
  end

  local iInfoTable = self.infoTable[BD.PIN_DATA_INPUT]
  local oInfoTable = self.infoTable[BD.PIN_DATA_OUTPUT]

  -- input
  local firstIsThis = self:_FirstThis() -- 类成员的调用 第一个参数是this
  local argId = 1;
  local toBeDelete = {}  -- 删除已不存在的引脚
  for thisArgId, input in pairs(self.inputs) do
    if thisArgId ~= 1 or not firstIsThis then
      local inputInfo =  inputsInfo[argId]
      if inputInfo == nil or input.argUid ~= inputInfo.argUid then
        table.insert(toBeDelete, {
                argId = firstIsThis and thisArgId - 1 or thisArgId, -- 函数参数序号(不含this)
                name = iInfoTable[thisArgId].name}
        )
      else
        local curInputInfo = iInfoTable[thisArgId]
        if not curInputInfo.rtti:isType(inputInfo.rtti) then
          local oldRtti = curInputInfo.rtti;
          local rtti = inputInfo:GetRtti(); -- 引脚存在类型修改了,重新同步
          local default = inputInfo:GetDefault()
          curInputInfo:ChangeRtti(rtti, default);
          input.literal = BU:GenerateLiteral(rtti, default)
          self:AppendModifyMessage(string.format("pin[in][%d] parameter %s type changed %s -> %s",
                  thisArgId, inputInfo.name, BR.GetRealTypeName(oldRtti), BR.GetRealTypeName(rtti)))
        end
        if inputInfo.name ~= curInputInfo.name then
          self:AppendModifyMessage(string.format("pin[in][%d] parameter name changed %s -> %s",
                  thisArgId, curInputInfo.name, inputInfo.name))
        end
        curInputInfo:ChangeNameAndHint(inputInfo.name, inputInfo.tips);
        argId = argId + 1;
      end
    end
  end

  if next(toBeDelete) then
    for offset, deleteItem in pairs(toBeDelete) do
      self:_PinDelete(true, deleteItem.argId - (offset - 1))
      self:AppendModifyMessage(
              string.format("pin[in][%d] %s is deleted",
              firstIsThis and deleteItem.argId + 1 or deleteItem.argId, deleteItem.name))
    end
  end


  local argInfo = inputsInfo[argId] -- argId 从这个以后的都是没有创建的
  while argInfo ~= nil do
    self:_PinAdd(true, argInfo);
    self:AppendModifyMessage(
            string.format( "pin[in][%d] %s is add",
            firstIsThis and argId + 1 or argId, argInfo.name))
    argId = argId + 1
    argInfo = inputsInfo[argId]
  end

  -- output
  argId = 1;
  toBeDelete = {}
  for thisArgId, output in pairs(self.outputs) do
    local outputInfo = outputsInfo[argId]
    if outputInfo == nil or output.argUid ~= outputInfo.argUid then
      table.insert(toBeDelete, {argId = thisArgId, name = oInfoTable[thisArgId].name} )
    else
      local curOutputInfo = oInfoTable[thisArgId];
      if not curOutputInfo.rtti:isType(outputInfo.rtti) then
        local oldRtti = curOutputInfo.rtti;
        local rtti = outputInfo:GetRtti()
        curOutputInfo:ChangeRtti(rtti);
        self:AppendModifyMessage(string.format("pin[out][%d] parameter %s type changed %s -> %s",
                thisArgId, outputInfo.name, BR.GetRealTypeName(oldRtti), BR.GetRealTypeName(rtti)))
      end
      if outputInfo.name ~= curOutputInfo.name then
        self:AppendModifyMessage(string.format("pin[out][%d] parameter name changed %s -> %s",
                thisArgId, curOutputInfo.name, outputInfo.name))
      end
      curOutputInfo:ChangeNameAndHint(outputInfo.name, outputInfo.tips);
      argId = argId + 1;
    end
  end

  if next(toBeDelete) then
    for offset, deleteItem in pairs(toBeDelete) do
      self:_PinDelete(false, deleteItem.argId - (offset - 1) )
      self:AppendModifyMessage("pin[out]["..tostring(deleteItem.argId).."] ".. deleteItem.name .." is deleted");
    end
  end

  argInfo = outputsInfo[argId]
  while argInfo ~= nil do
    self:_PinAdd(false, argInfo);
    self:AppendModifyMessage("pin[out]["..tostring(argId).."] ".. argInfo.name .." is add");

    argId = argId + 1
    argInfo = outputsInfo[argId]
  end
  return
end


function FuncCallNode:_OnDeserialize()
  self:_UpdateNodeType();
  self.infoTable[BD.PIN_DATA_INPUT] = self[BD.INPUTS_INFO]
  self.infoTable[BD.PIN_DATA_OUTPUT]= self[BD.OUTPUTS_INFO]
  self:_NameChanged(FuncCallNode.INVALID_NAME, self.refName)
  if self.eventId ~= nil then self.eventName = BD.EVENT_PREFIX..self.eventId end
end

function FuncCallNode:_AddToRefactorEd()

  if self.ftnType == BD.FuncType.LOCAL then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    if classInfo ~= nil then
      local ftnList = classInfo:GetBlueFtnList()
      local blueFtn = ftnList[self.ftnUid]
      if blueFtn ~= nil then   -- 加入函数的引用列表
        blueFtn:AddRefactor(self);
      end
    end
  elseif self.ftnType == BD.FuncType.EVENT then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    if classInfo ~= nil then
      local eventNodes = classInfo:GetCustomEventNodes()
      local customEvent = eventNodes[self.eventId]
      if customEvent ~= nil then
        customEvent:AddRefactor(self)
      end
    end
  elseif self.ftnType == BD.FuncType.DISPATCHER then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    if classInfo ~= nil then
      local dispatchers = classInfo:GetEventDispatchers()
      local dispatcherInfo = dispatchers[self.dispId]
      if dispatcherInfo ~= nil then
        dispatcherInfo:AddRefactor(self);
      end
    end
  end
end

function FuncCallNode:_OnDeserializePost()

  FuncCallNode.super:_OnDeserializePost()

  if _KRATOSEDITOR then
    self:_AddToRefactorEd();
  end

  if self.ftnType == BD.FuncType.GLOBAL then
    local node = self.graph[BD.THIS]
    local scene = node:GetHostScene();
    local rootNode = scene:GetRootNode();
    self.globalFunc = setmetatable({} , { __mode = "v"})
    local blueprintComp = rootNode:GetComponent(AE.Node.CT_BLUEPRINT);
    if blueprintComp then
      local blueScene = BU.GetIns(blueprintComp);
      if blueScene then

        local libs = blueScene:GetLibs()
        local lib = libs[self.libUid]
        if lib ~= nil then

          local ftns = lib:GetLibFtns()
          local ftn = ftns[self.ftnUid]
          if ftn ~= nil then
            -- 加入全局函数的引用列表 -- 全局函数没有 FuncInfo
            ftn:AddRefactor(self);
            self.globalFunc[1] = ftn
            return
          end -- if ftn ~= nil
        end -- if lib ~= nil then
      end -- if blueScene
    end -- if blueprintComp
  end

end


-- 不同步只检查:只要参数数目/类型(其他不考虑) 就必须打开蓝图同步 才能编译通过
function FuncCallNode:_OnCompile()

  if _KRATOSEDITOR then

    self.compileResult = nil
    self._failReason = nil

    local inputsInfo = {}
    local outputsInfo = {}

    if self:_FirstThis() then

      local classInfo = BR.GetClassesByIdEd(self._clsId);

      if classInfo ~= nil then

        if self.ftnType == BD.FuncType.LOCAL then

          local ftnList = classInfo:GetBlueFtnList()
          local blueFtn = ftnList[self.ftnUid]
          if blueFtn ~= nil then
            inputsInfo = blueFtn:GetInputInfo();
            outputsInfo = blueFtn:GetOutputInfo();
          else
            self._failReason = string.format("function %s is deleted in class %s", self.refName, classInfo:GetClassName())

          end

        elseif self.ftnType == BD.FuncType.DISPATCHER then

          local dispatchers = classInfo:GetEventDispatchers()
          local dispatcherInfo = dispatchers[self.dispId]
          if dispatcherInfo ~= nil then
            inputsInfo = dispatcherInfo:GetInputInfo();
          else
            self._failReason = string.format("dispatcher %s is deleted in class %s", self.refName, classInfo:GetClassName())

          end

        elseif self.ftnType == BD.FuncType.EVENT then

          local eventNodes = classInfo:GetCustomEventNodes()
          local customEvent = eventNodes[self.eventId]
          if customEvent ~= nil then
            inputsInfo = customEvent:GetInputInfo();
          else
            self._failReason = string.format("custom event %s is deleted in class %s", self.refName, classInfo:GetClassName())

          end
        end

      else -- class not found
        self._failReason = "prefab/blueprint is deleted"
      end
    else -- global function call
      local node = self.graph[BD.THIS]
      local scene = node:GetHostScene();
      local rootNode = scene:GetRootNode();

      local blueprintComp = rootNode:GetComponent(AE.Node.CT_BLUEPRINT);
      if blueprintComp then
        local blueScene = BU.GetIns(blueprintComp);
        if blueScene then
          local libs = blueScene:GetLibs()
          local lib = libs[self.libUid]
          if lib ~= nil then
            local ftns = lib:GetLibFtns()
            local ftn = ftns[self.ftnUid]
            if ftn ~= nil then
              inputsInfo = ftn:GetInputInfo();
              outputsInfo = ftn:GetOutputInfo();
            else
              self._failReason = "global function is deleted"
            end
          else
            self._failReason = "library is deleted"
          end
        else
          self._failReason = "blue scene is deleted"
        end
      else
        self._failReason = "blue scene's component is deleted"
      end
    end

    if self._failReason ~= nil then
      self.compileResult = self._failReason .. ", open and check it"
      return false
    end

    local iInfoTable = self.infoTable[BD.PIN_DATA_INPUT]
    local oInfoTable = self.infoTable[BD.PIN_DATA_OUTPUT]

    local argId = 1;
    for thisArgId, input in pairs(self.inputs) do
      if thisArgId ~= 1 or not self:_FirstThis() then  -- FIXME 类成员函数
        local inputInfo =  inputsInfo[argId]
        if inputInfo == nil or input.argUid ~= inputInfo.argUid then
          self.compileResult = "parameter is deleted, open and check it"
          return false ;
        else
          local curInputInfo = iInfoTable[thisArgId]
          if not curInputInfo.rtti:isType(inputInfo.rtti) then
            self.compileResult = "parameter type is changed, open and check it"
            return false ;
          end
          argId = argId + 1;
        end
      end
    end

    if inputsInfo[argId] ~= nil then
      self.compileResult = "parameter is add, open and check it"
      return false ;
    end

    argId = 1;
    for thisArgId, output in pairs(self.outputs) do
      local outputInfo =  outputsInfo[argId]
      if outputInfo == nil or output.argUid ~= outputInfo.argUid then
        self.compileResult = "parameter is deleted, open and check it"
        return false ;
      else
        local curInputInfo = oInfoTable[thisArgId]
        if not curInputInfo.rtti:isType(outputInfo.rtti) then
          self.compileResult = "parameter type is changed, open and check it"
          return false ;
        end
        argId = argId + 1;
      end
    end

    if outputsInfo[argId] ~= nil then
      self.compileResult = "parameter is add, open and check it"
      return false ;
    end

  end -- if _KRATOSEDITOR then

  return true
end


function FuncCallNode:_OnUpdateByEditor()
  -- 如果显示当前蓝图类,那么开启实时编译/检查
  self:_OnCompile()
end


function FuncCallNode:ToBeDeletedEd()
  self._failReason = self.refName.." is deleted"
  -- 全局函数/蓝图间成员函数 只是标记被删除
end

-- Node delete by editor
function FuncCallNode:_OnDeleteEd()

  if self.ftnType == BD.FuncType.LOCAL then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    if classInfo ~= nil then
      local ftnList = classInfo:GetBlueFtnList()
      local blueFtn = ftnList[self.ftnUid]
      if blueFtn ~= nil then blueFtn:RemoveRefactor(self) end -- FuncInfo
    else
      WARNING("function declaration deleted")
    end
  elseif self.ftnType == BD.FuncType.EVENT then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    if classInfo ~= nil then
      local customEvents = classInfo:GetCustomEventNodes();
      local customEvent = customEvents[self.eventId]
      if customEvent~= nil then customEvent:RemoveRefactor(self) end -- EventInfo
      -- ToBeDeletedEd, just flag
    else
      WARNING("event declaration deleted")
    end
  elseif self.ftnType == BD.FuncType.DISPATCHER then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    if classInfo ~= nil then
      local dispatchers = classInfo:GetEventDispatchers()
      local dispatcherInfo = dispatchers[self.dispId]
      if dispatcherInfo~= nil then dispatcherInfo:RemoveRefactor(self) end
    else
      WARNING("dispatcher declaration deleted")
    end
  elseif self.ftnType == BD.FuncType.GLOBAL then
    local func = self.globalFunc[1]
    if func ~= nil then
      func:RemoveRefactor(self) -- BlueFunction
      self.globalFunc[1] = nil
    else
      WARNING("library function deleted before");
    end
  end

  self.ftnUid = nil
  self.dispId = nil
  self.eventId = nil
end

function FuncCallNode:_NameChanged(oldName, newName)
  self.refName = newName
  if self.ftnType == BD.FuncType.LOCAL then
    self:SetFunctionName("Call Function "..self.refName)
  elseif self.ftnType == BD.FuncType.GLOBAL then
    self:SetFunctionName("Call Global "..self.refName)
  elseif self.ftnType == BD.FuncType.EVENT then
    self:SetFunctionName("Call Event "..self.refName)
  elseif self.ftnType == BD.FuncType.DISPATCHER then
    self:SetFunctionName("Call Dispatcher "..self.refName)
  end
end

function FuncCallNode:ClsIdChgEd(oldClassId, newClassId)

  if oldClassId ==  self._clsId then
    if self.ftnType == BD.FuncType.GLOBAL then
      error("Global Function Do Not has ClassID")
    end

    -- 从原来的refactor移除 后面blueprint重新注册会 在新的refactor上添加
    if self.ftnType == BD.FuncType.LOCAL then
      local classInfo = BR.GetClassesByIdEd(self._clsId);
      local ftnList = classInfo:GetBlueFtnList()
      local blueFtn = ftnList[self.ftnUid]
      blueFtn:RemoveRefactor(self) -- FuncInfo
    elseif self.ftnType == BD.FuncType.GLOBAL then

    elseif self.ftnType == BD.FuncType.EVENT then
      local classInfo = BR.GetClassesByIdEd(self._clsId);
      local customEvents = classInfo:GetCustomEventNodes();
      local customEvent = customEvents[self.eventId]
      customEvent:RemoveRefactor(self) -- EventInfo
    elseif self.ftnType == BD.FuncType.DISPATCHER then
      local classInfo = BR.GetClassesByIdEd(self._clsId);
      local dispatchers = classInfo:GetEventDispatchers()
      local dispatcherInfo = dispatchers[self.dispId]
      dispatcherInfo:RemoveRefactor(self)
    end

    self._clsId =  newClassId
    local blueType = BR.GetRttiByClsId(self._clsId);
    local bluePinInfo = self.infoTable[BD.PIN_DATA_INPUT][1]
    bluePinInfo:ChangeRtti(blueType, BlueprintSelf())
  end
end

function FuncCallNode:GetInfo() -- return const, do NOT modify
  if self.ftnType == BD.FuncType.LOCAL then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    local ftnList = classInfo:GetBlueFtnList()
    local blueFtn = ftnList[self.ftnUid]
    if blueFtn == nil then
      ERROR(string.format("GetInfo LOCAL fail %s, %s", tostring(self._clsId), tostring(self.ftnUid)));
    end
    return blueFtn
  elseif self.ftnType == BD.FuncType.GLOBAL then
    if self.globalFunc[1] == nil then
      ERROR(string.format("GetInfo GLOBAL fail %s, %s", tostring(self.libUid), tostring(self.ftnUid)));
    end
    return self.globalFunc[1]
  elseif self.ftnType == BD.FuncType.EVENT then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    local customEvents = classInfo:GetCustomEventNodes();
    local customEvent = customEvents[self.eventId]
    if customEvent == nil then
      ERROR(string.format("GetInfo EVENT fail %s, %s", tostring(self._clsId), tostring(self.eventId)));
    end
    return customEvent ;
  elseif self.ftnType == BD.FuncType.DISPATCHER then
    local classInfo = BR.GetClassesByIdEd(self._clsId);
    local dispatchers = classInfo:GetEventDispatchers()
    local dispatcherInfo = dispatchers[self.dispId]
    if dispatcherInfo == nil then
      ERROR(string.format("GetInfo DISPATCHER fail %s, %s", tostring(self._clsId), tostring(self.dispId)));
    end
    return dispatcherInfo;
  end
  ERROR("GetInfo self.ftnType unknown "..tostring(self.ftnType) );
end

FuncCallNode:RegisterExecInput(1, "exec", "Execute");
FuncCallNode:RegisterExecOutput(1, "exec", "Execute");
return FuncCallNode;


