local venuscore = require "libvenuscore"
local Types = require "venuscore.rtti.types"
local cv = require "computervisionfunction"
local mf = require "mathfunction"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local BlueNode = require "bluecore.bluenode"
local BlueSelf = require "bluecore.blueself"
local EventDispatcher = require "bluecore.eventdispatcher"
local BluePinInfo = require "bluecore.bluepininfo"

local Classify = BlueNode:extend();


function Classify:new(...)
  Classify.super.new(self, ...)
  self.output = {
     humanID = 0,
     point = mf.vector2(0,0),
     size = mf.vector2(0,0),
     isHigh = false,
     isEdge = false,
  }
  self.isCfgEdge = false ;

  self.component = nil ;

  -- override Types.UnFixedInputType
  self.myUnFixComboType = Types.ClassifyType:extend();
  self.myUnFixComboType:SetData({},{});

end


function Classify:_OnUpdate(args)
  self.output = args ; -- args is event.params
  return self.output.humanId, self.output.point, self.output.size, self.output.isHigh, self.output.isEdge
end

function Classify:_OnNextBranch()
  if self.isCfgEdge and not self.output.isEdge then
    return 0
  end
  if self.output.isHigh then
    return 1
  else
    return 2
  end
end

function Classify:RegisterTriggerFunc(func, bluePrintSelf)

  self:_GetInput()

  local comp = self.component or self.inputArgs[1] -- 优先反射面板 后链接

  local type = self.inputArgs[2];

  self.isCfgEdge = self.inputArgs[3];

  WARNING("Classify " .. tostring(type) );

  EventDispatcher:RegisterCallback(comp:GetContentPath(), type, func, bluePrintSelf)

end

function Classify:LiteralMismatchOnCompile(pin, pinInfo)
  -- 第一个引脚使用的是反射面板的值
  if pin.argId ==  1 and BU:isBlueSelf(pin.literal) and  not venuscore.isNil(self.component) then
    return true
  end
  return false
end

function Classify:_OnCompile()
  self.compileResult = nil
  if Classify.super._OnCompile(self) then

    local comp = self.component

    if comp ~= nil then
      if venuscore.isNil(comp) then
        self.compileResult = BD.EHINT.COMP_LOST
        return false
      end
    end

    if comp == nil then

      if  #self.inputs[1].links ~= 0 then
        local bluePinLink = self.inputs[1].links[1] -- 输入引脚只允许一个连接
        local otherNode = self.graph:GetNode(bluePinLink.nodeUid);

        -- 确保上一个节点是 引用Component
        if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp == nil then
          self.compileResult = BD.EHINT.NOT_LINK_TO_COMP
          return false
        end

        if venuscore.isNil(otherNode.comp) then
          self.compileResult = BD.EHINT.COMP_LOST
          return false
        end

        comp = otherNode.comp

      else
        self:_Reset();
        self.compileResult = BD.EHINT.PIN_1_EMPTY;
        return false
      end
    end

    -- 节点配置值(引脚字面值)不能超过当前反射面板配置值 在编译时(和当前蓝图显示时)自动去除超出范围的
    local configs, hints = self:_GetCompCurrentDetectType(comp);

    local config =   self.inputs[2].literal
    local found = false ;
    for _, cfg in pairs(configs) do
      if config == cfg then
        found = true
      end
    end
    if not found then
      self.inputs[2].literal = 0 -- 存放节点配置值(会序列化)
    end

    self.myUnFixComboType:SetData(configs, hints);

    return true ;
  end

  return false
end


function Classify:_Reset()
  if venuscore.isNil(self.component) then
    -- 引脚类型
    self.myUnFixComboType:SetData({},{});
    -- 字面值
    self.inputs[2].literal = 0
  end
end

function Classify:_OnDeserialize()
  self:_SetupInstanceInfoTable();
end

function Classify:_OnDeserializePost()
  if not venuscore.isNil(self.component) then
    local configs, hints = self:_GetCompCurrentDetectType(self.component);
    self.myUnFixComboType:SetData(configs, hints);
  end
end

function Classify:_SetupInstanceInfoTable()

  -- override Recognition infoTable
  local staticExecInputsInfo = self.infoTable[BD.PIN_EXEC_INPUT]
  local staticExecOutputsInfo = self.infoTable[BD.PIN_EXEC_OUTPUT]
  local staticDataOutputsInfo = self.infoTable[BD.PIN_DATA_OUTPUT]
  local staticDataInputsInfo = self.infoTable[BD.PIN_DATA_INPUT]
  self.infoTable = {}
  self.infoTable[BD.PIN_DATA_INPUT] = {}
  self.infoTable[BD.PIN_DATA_OUTPUT] = staticDataOutputsInfo -- 并不会动态修改 data output info
  self.infoTable[BD.PIN_EXEC_INPUT]  = staticExecInputsInfo
  self.infoTable[BD.PIN_EXEC_OUTPUT] = staticExecOutputsInfo

  local pinInfo = staticDataInputsInfo[1]
  self.infoTable[BD.PIN_DATA_INPUT][1] =  staticDataInputsInfo[1]
  pinInfo = staticDataInputsInfo[2]
  self.infoTable[BD.PIN_DATA_INPUT][2] = BluePinInfo(BD.PIN_DATA_INPUT, pinInfo.argId, self.myUnFixComboType, pinInfo.name,pinInfo.tips,pinInfo.default,pinInfo.delAble, pinInfo.groupName);
  pinInfo = staticDataInputsInfo[3]
  self.infoTable[BD.PIN_DATA_INPUT][3] =  staticDataInputsInfo[3]
end

function Classify:_GetCompCurrentDetectType(comp)
  local configs = {};
  local hints = {};
  if comp.FaceExpression then
    for k,v in pairs(cv.ClassifyComponent:GetFaceTypes()) do
      table.insert(configs, k)
      table.insert(hints, v)
    end
  end
  if comp.HandClassify then
    for k,v in pairs(cv.ClassifyComponent:GetHandTypes()) do
      table.insert(configs, k)
      table.insert(hints, v)
    end
  end
  return configs, hints ;
end

function Classify:_IsCompPin(pin)
  return pin.pinType == BD.PIN_DATA_INPUT and pin.argId == 1
end

-- 蓝图节点反射面板相关
function Classify:_SetComp(comp)
  self.component = comp ;

  if not venuscore.isNil(comp) then
    local configs, hints = self:_GetCompCurrentDetectType(comp);
    self.myUnFixComboType:SetData(configs, hints);
  end

end

function Classify:_GetComp()
  return self.component
end


function Classify:ReflectFilterFtnOnPrefab(obj, member, value)
  return member == "component"
end

if _KRATOSEDITOR then

  function Classify:_OnLink(selfPin, otherNode, otherPin)
    if self:_IsCompPin(selfPin) then
      -- 上一个节点必须是组件引用节点
      if otherNode:GetFunctionType() == BD.COMPONENT_NODE and otherNode.comp ~= nil then
        if venuscore.isNil(otherNode.comp) then
          ERROR(BD.EHINT.COMP_LOST)
        else
          local configs, hints = self:_GetCompCurrentDetectType(otherNode.comp);
          self.myUnFixComboType:SetData(configs, hints);
          self.inputs[2].literal = 0
        end
      else
        ERROR(BD.EHINT.NOT_LINK_TO_COMP)
        ERROR("[Classify] error "..tostring(otherNode:GetFunctionType())..","..tostring(otherNode.comp))
      end
    end
  end

  function Classify:_OnUnLink(unlinkPin)
    if self:_IsCompPin(unlinkPin) then -- 断开是数据输入第一个引脚
      self:_Reset();
    end
  end


  function Classify:_OnCreateWithEditor()
    Classify.super._OnCreateWithEditor(self);

    self:_SetupInstanceInfoTable();
  end

  function Classify:_OnUpdateByEditor()

    Classify.super._OnUpdateByEditor(self);

    -- 如果编辑器显示当前蓝图,即时显示检查结果
    self:_OnCompile();
  end

end

Classify:RegisterInput(1, cv.ClassifyComponent:RTTI(), "component", "监听组件", BlueSelf());
Classify:RegisterInput(2, Types.ClassifyType, "eventType", "监听事件类型", 0);
Classify:RegisterInput(3, Types.BoolType, "edge", "只在事件触发时执行(非持续)" , true)

Classify:RegisterOutput(1, Types.IntType , "HumanID", "人物序号");
Classify:RegisterOutput(2, mf.vector2:RTTI() , "Position", "框位置");
Classify:RegisterOutput(3, mf.vector2:RTTI() , "Size", "框大小");
Classify:RegisterOutput(4, Types.BoolType , "isHigh", "是否高电平");
Classify:RegisterOutput(5, Types.BoolType , "isEdge", "是否边沿");


Classify:RegisterExecOutput(1, "OnTrigger", "执行");
Classify:RegisterExecOutput(2, "OnRelease", "释放");

Classify:SetFunctionName("Classify");

Classify:SetFunctionType(BD.EVENT_FUNCTION_NODE);

Classify:SetEventType(BD.EventType.RegisterCallback);



-- 反射面板(支援prefab下的多bvt事件节点)
-- 优先顺序 反射面板component --(Override)-> 引脚连接 ; 如果是nil的话, 代表使用链接
Classify:MemberRegister("component", Types.ReferenceType(cv.ClassifyComponent:RTTI(), Classify._GetComp, Classify._SetComp)); -- 检测对象



return Classify;

