--@author      : Zhu Fei
--@date        : 2020-03-25
--@description : lua script for FluidEffect
--@version     : 1.0

local venuscore    = require "venuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"

local fluid = {}
fluid.Queue = 100   --sequence of fluid effect

--FBOs: render targets of simulation substeps
fluid.fbo_velocity_read  = 0
fluid.fbo_velocity_write = 0
fluid.fbo_pressure_read  = 0
fluid.fbo_pressure_write = 0
fluid.fbo_color_read     = 0
fluid.fbo_color_write    = 0
fluid.fbo_curl           = 0
fluid.fbo_divergence     = 0
fluid.fbo_last_frame     = 0
fluid.fbo_motion_vec     = 0

--render objects
-- fluid.ro_motion_vec_step     = 0 --cur_frame - last_frame -> motion_vec
-- fluid.ro_blur_step           = 0 --motion_vec -> motion_vec
-- fluid.ro_ext_force_step      = 0 --motion_vec, velocity -> velocity
-- fluid.ro_dye_color_step      = 0 --motion_vec, custom color, color -> color
-- fluid.ro_curl_step           = 0 --velocity -> curl
-- fluid.ro_vorticity_step      = 0 --velocity, curl -> velocity
-- fluid.ro_divergence_step     = 0 --velocity -> divergence
-- fluid.ro_clear_pressure_step = 0 --pressure -> pressure
-- fluid.ro_pressure_step       = 0 --divergence, pressure -> pressure
-- fluid.ro_pressure_force_step = 0 --pressure, velocity -> velocity
-- fluid.ro_advection_step      = 0 --velocity, dissipation -> velocity
--                                  --color, velocity, dissipation -> color
-- fluid.ro_copy_step           = 0 --cur_frame -> last_frame
-- fluid.ro_blend_step          = 0 --color, cur_frame -> color
-- fluid.ro_display_step        = 0 --final display

--liuqian add material for renderObject
fluid.ro_motion_vec_step_material     = 0 --cur_frame - last_frame -> motion_vec
fluid.ro_blur_step_material           = 0 --motion_vec -> motion_vec
fluid.ro_ext_force_step_material      = 0 --motion_vec, velocity -> velocity
fluid.ro_dye_color_step_material      = 0 --motion_vec, custom color, color -> color
fluid.ro_curl_step_material           = 0 --velocity -> curl
fluid.ro_vorticity_step_material      = 0 --velocity, curl -> velocity
fluid.ro_divergence_step_material     = 0 --velocity -> divergence
fluid.ro_clear_pressure_step_material = 0 --pressure -> pressure
fluid.ro_pressure_step_material       = 0 --divergence, pressure -> pressure
fluid.ro_pressure_force_step_material = 0 --pressure, velocity -> velocity
fluid.ro_advection_step_material      = 0 --velocity, dissipation -> velocity
                                 --color, velocity, dissipation -> color
fluid.ro_copy_step_material           = 0 --cur_frame -> last_frame
fluid.ro_blend_step_material          = 0 --color, cur_frame -> color
fluid.ro_display_step_material        = 0 --final display

--liuqian modify interface
fluid.ro_motion_vec_step_renderObj     = 0 --cur_frame - last_frame -> motion_vec
fluid.ro_blur_step_renderObj           = 0 --motion_vec -> motion_vec
fluid.ro_ext_force_step_renderObj      = 0 --motion_vec, velocity -> velocity
fluid.ro_dye_color_step_renderObj      = 0 --motion_vec, custom color, color -> color
fluid.ro_curl_step_renderObj           = 0 --velocity -> curl
fluid.ro_vorticity_step_renderObj      = 0 --velocity, curl -> velocity
fluid.ro_divergence_step_renderObj     = 0 --velocity -> divergence
fluid.ro_clear_pressure_step_renderObj = 0 --pressure -> pressure
fluid.ro_pressure_step_renderObj       = 0 --divergence, pressure -> pressure
fluid.ro_pressure_force_step_renderObj = 0 --pressure, velocity -> velocity
fluid.ro_advection_step_renderObj      = 0 --velocity, dissipation -> velocity
                                 --color, velocity, dissipation -> color
fluid.ro_copy_step_renderObj           = 0 --cur_frame -> last_frame
fluid.ro_blend_step_renderObj          = 0 --color, cur_frame -> color
fluid.ro_display_step_renderObj        = 0 --final display


----------------------------interfaces----------------------------
function fluid:Initialize(host, size)
    --public configs
    self.simulation_quality     = 2    --[0, 3] -> [very low, low, medium, high]
    self.visual_quality         = 2    --[0, 3] -> [very low, low, medium, high]
    self.display_dissipation    = 1.0  --[0, 4], controls how fast color field dissipates
    self.velocity_dissipation   = 0.02 --[0, 4], controls how fast velocity dissipates
    self.pressure               = 0.8  --[0, 1], fluid pressure
    self.curl                   = 80   --[0, 100], controls the vorticiy of fluid
    self.display_mask           = apolloengine.TextureEntity()  --only display fluid in mask
    self.use_display_mask       = 0    --indicate whether display mask is used
    self.invert_display_mask    = 0    --switch between background and forground of display mask
    self.fluid_init_mask        = apolloengine.TextureEntity()  --only init fluid in mask
    self.use_fluid_init_mask    = 0    --indicate whether init mask is used
    self.invert_fluid_init_mask = 0    --switch between background and forground of init mask
    self.blend_mode             = 1    --[1, 7] -> [Normal, Add, Lighten, Multiply, Overlay, Screen, Lighter]
    self.color_mode             = 0    --[0, 1], controls fluid color -> [customized color, background texture color]
    self.fluid_init_color       = mathfunction.vector3(255.0, 100.0, 0.0) --[0,255], customized fluid color when first generated
    self.fluid_phase2_color     = mathfunction.vector3(255.0, 0.0, 0.0) --[0,255], customized fluid color in phase 2
    self.fluid_phase3_color     = mathfunction.vector3(0.0, 0.0, 0.0) --[0,255], customized fluid color in phase 3
    self.use_multi_phase_color  = 0.0  --0/1, whether to use multi-phase color or not
    self.fluid_init_difficulty  = 0.6  --[0, 1.0], indicate how difficult it is to init fluid motion
    self.mv_force_scale         = 100  --force scale to generate force from motion vector
    self.float_max              = 255.0   --max value of floating point values, used to normalize values in shaders
    self.wind_dir               = mathfunction.vector2(0.0, 0.0)  --wind field direction
    self.wind_strength          = 0.0  --strength of wind field
    self.display_background     = 1.0  --whether to display background frame or not
    --private configs
    self.pressure_iter_num      = 20   --number of iterations to apply pressure force
    self.time_step              = 0.01 --simulation time step
    self.sim_resolution_base    = 32   --short-side simulation resolution corresponds to 'very low' quality
    self.vis_resolution_base    = 128  --short-side visualization resolution corresponds to 'very low' quality
    self.float_min              = -(self.float_max + 1)  --min value of floating point values, used to normalize values in shaders
    self.normalized_zero        = -self.float_min / (self.float_max - self.float_min) -- normalized value that corresponds to zero
    self.blur_motion_vector     = 0.0  --whether blur motion vector or not
    self.motion_detect_interval = 1    --motion vector detection frame interval
    self.motion_detect_idx      = 0    --current frame index in a motion detect circle

    self.first_frame            = 1    --indicate whether current frame is first frame
    --determine resolutions according to configs && screen resolution
    self.simulation_resolution = 0
    self.visual_resolution     = 0
    self.frame_resolution      = 0
    self.sim_texel_size        = 0
    self.vis_texel_size        = 0
    self:_UpdateResolutions(size)
    --backup current state to trigger render parameter update
    self:_BackupStates()

    --backup host for use in process
    self.host = host
    --create render targets
    self:_CreateSimResFBOs(host)
    self:_CreateVisResFBOs(host)

    --create render objects and set parameters that donot update frequently
    --self.ro_motion_vec_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/motion_vector_step.material")
    -- local motion_vec_material = self.ro_motion_vec_step.MaterialEntities[1]
    -- motion_vec_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_motion_vec_step_material = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/motion_vector_step.material");
    self.ro_motion_vec_step_renderObj = host:CreateRenderObject();
    self.ro_motion_vec_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)

    if self.blur_motion_vector > 0.0 then
        -- self.ro_blur_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/blur_step.material")
        -- local blur_material = self.ro_blur_step.MaterialEntities[1]
        -- blur_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_blur_step_material   = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/blur_step.material")
        self.ro_blur_step_renderObj = host:CreateRenderObject();
        self.ro_blur_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    end

    -- self.ro_ext_force_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/ext_force_step.material")
    -- local ext_force_material = self.ro_ext_force_step.MaterialEntities[1]
    -- ext_force_material:SetParameter("MV_FORCE_SCALE", mathfunction.vector1(self.mv_force_scale))
    -- ext_force_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- ext_force_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_ext_force_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/ext_force_step.material")
    self.ro_ext_force_step_renderObj = host:CreateRenderObject();
    self.ro_ext_force_step_material:SetParameter("MV_FORCE_SCALE", mathfunction.vector1(self.mv_force_scale))
    self.ro_ext_force_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_ext_force_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    local wind_dir = mathfunction.vector2(self.wind_dir:x(), self.wind_dir:y())
    if self.wind_dir:LengthPow() > 0.0 then
        wind_dir:NormalizeSelf()
    end
    wind_dir = wind_dir * self.wind_strength
    self.ro_ext_force_step_material:SetParameter("WIND_FIELD", wind_dir)

    -- self.ro_dye_color_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/dye_color_step.material")
    -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
    -- local normalized_fluid_init_color = self.fluid_init_color / 255.0
    -- dye_color_material:SetParameter("COLOR_MODE", mathfunction.vector1(self.color_mode))
    -- dye_color_material:SetParameter("FLUID_COLOR", normalized_fluid_init_color)
    -- dye_color_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_fluid_init_mask))
    -- dye_color_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_fluid_init_mask))
    self.ro_dye_color_step_material   = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/dye_color_step.material")
    self.ro_dye_color_step_renderObj = host:CreateRenderObject();
    local normalized_fluid_init_color = self.fluid_init_color / 255.0
    self.ro_dye_color_step_material:SetParameter("COLOR_MODE", mathfunction.vector1(self.color_mode))
    self.ro_dye_color_step_material:SetParameter("FLUID_COLOR", normalized_fluid_init_color)
    self.ro_dye_color_step_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_fluid_init_mask))
    self.ro_dye_color_step_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_fluid_init_mask))

    -- self.ro_curl_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/curl_step.material")
    -- local curl_material = self.ro_curl_step.MaterialEntities[1]
    -- curl_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    -- curl_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- curl_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_curl_step_material   = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/curl_step.material")
    self.ro_curl_step_renderObj = host:CreateRenderObject();
    self.ro_curl_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_curl_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_curl_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    -- self.ro_vorticity_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/vorticity_step.material")
    -- local vorticity_material = self.ro_vorticity_step.MaterialEntities[1]
    -- vorticity_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    -- vorticity_material:SetParameter("CURL_CONFIG", mathfunction.vector1(self.curl))
    -- vorticity_material:SetParameter("TIME_STEP", mathfunction.vector1(self.time_step))
    -- vorticity_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- vorticity_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_vorticity_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/vorticity_step.material")
    self.ro_vorticity_step_renderObj = host:CreateRenderObject();
    self.ro_vorticity_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_vorticity_step_material:SetParameter("CURL_CONFIG", mathfunction.vector1(self.curl))
    self.ro_vorticity_step_material:SetParameter("TIME_STEP", mathfunction.vector1(self.time_step))
    self.ro_vorticity_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_vorticity_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    -- self.ro_divergence_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/divergence_step.material")
    -- local divergence_material = self.ro_divergence_step.MaterialEntities[1]
    -- divergence_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    -- divergence_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- divergence_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_divergence_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/divergence_step.material")
    self.ro_divergence_step_renderObj = host:CreateRenderObject();
    self.ro_divergence_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_divergence_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_divergence_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    -- self.ro_clear_pressure_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/clear_pressure_step.material")
    -- local clear_pressure_material = self.ro_clear_pressure_step.MaterialEntities[1]
    -- clear_pressure_material:SetParameter("PRESSURE_CONFIG", mathfunction.vector1(self.pressure))
    -- clear_pressure_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- clear_pressure_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_clear_pressure_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/clear_pressure_step.material")
    self.ro_clear_pressure_step_renderObj = host:CreateRenderObject();
    self.ro_clear_pressure_step_material:SetParameter("PRESSURE_CONFIG", mathfunction.vector1(self.pressure))
    self.ro_clear_pressure_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_clear_pressure_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    -- self.ro_pressure_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/pressure_step.material")
    -- local pressure_material = self.ro_pressure_step.MaterialEntities[1]
    -- pressure_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    -- pressure_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- pressure_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_pressure_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/pressure_step.material")
    self.ro_pressure_step_renderObj = host:CreateRenderObject();
    self.ro_pressure_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_pressure_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_pressure_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    -- self.ro_pressure_force_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/pressure_force_step.material")
    -- local pressure_force_material = self.ro_pressure_force_step.MaterialEntities[1]
    -- pressure_force_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    -- pressure_force_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- pressure_force_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_pressure_force_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/pressure_force_step.material")
    self.ro_pressure_force_step_renderObj = host:CreateRenderObject();
    self.ro_pressure_force_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_pressure_force_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_pressure_force_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))

    -- self.ro_advection_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/advection_step.material")
    -- local advection_material = self.ro_advection_step.MaterialEntities[1]
    -- advection_material:SetParameter("TIME_STEP", mathfunction.vector1(self.time_step))
    -- advection_material:SetParameter("VEL_FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- advection_material:SetParameter("VEL_FLOAT_MAX", mathfunction.vector1(self.float_max))
    -- advection_material:SetParameter("SRC_PHASE_1", self.fluid_init_color / 255.0)
    -- advection_material:SetParameter("SRC_PHASE_2", self.fluid_phase2_color / 255.0)
    -- advection_material:SetParameter("SRC_PHASE_3", self.fluid_phase3_color / 255.0)
    self.ro_advection_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/advection_step.material")
    self.ro_advection_step_renderObj = host:CreateRenderObject();
    self.ro_advection_step_material:SetParameter("TIME_STEP", mathfunction.vector1(self.time_step))
    self.ro_advection_step_material:SetParameter("VEL_FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_advection_step_material:SetParameter("VEL_FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_advection_step_material:SetParameter("SRC_PHASE_1", self.fluid_init_color / 255.0)
    self.ro_advection_step_material:SetParameter("SRC_PHASE_2", self.fluid_phase2_color / 255.0)
    self.ro_advection_step_material:SetParameter("SRC_PHASE_3", self.fluid_phase3_color / 255.0)

    -- self.ro_copy_step = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/copy_step.material")
    self.ro_copy_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/copy_step.material")
    self.ro_copy_step_renderObj = host:CreateRenderObject();

    -- self.ro_blend_step   = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/blend_step.material")
    -- local blend_material = self.ro_blend_step.MaterialEntities[1]
    -- blend_material:SetParameter("BLEND_MODE", mathfunction.vector1(self.blend_mode))
    self.ro_blend_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/blend_step.material")
    self.ro_blend_step_renderObj = host:CreateRenderObject();
    self.ro_blend_step_material:SetParameter("BLEND_MODE", mathfunction.vector1(self.blend_mode))

    -- self.ro_display_step = host:CreateRenderObject("comm:documents/shaders/posteffect/fluid/display_step.material")
    -- local display_material = self.ro_display_step.MaterialEntities[1]
    -- display_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_display_mask))
    -- display_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_display_mask))
    self.ro_display_step_material  = host:CreateMaterial("comm:documents/shaders/posteffect/fluid/display_step.material")
    self.ro_display_step_renderObj = host:CreateRenderObject();
    self.ro_display_step_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_display_mask))
    self.ro_display_step_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_display_mask))

    --register configs
    host:RegisterScriptParameter(self, "simulation_quality")
    host:RegisterScriptParameter(self, "visual_quality")
    host:RegisterScriptParameter(self, "display_dissipation")
    host:RegisterScriptParameter(self, "velocity_dissipation")
    host:RegisterScriptParameter(self, "pressure")
    host:RegisterScriptParameter(self, "curl")
    host:RegisterScriptParameter(self, "display_mask")
    host:RegisterScriptParameter(self, "use_display_mask")
    host:RegisterScriptParameter(self, "invert_display_mask")
    host:RegisterScriptParameter(self, "fluid_init_mask")
    host:RegisterScriptParameter(self, "use_fluid_init_mask")
    host:RegisterScriptParameter(self, "invert_fluid_init_mask")
    host:RegisterScriptParameter(self, "blend_mode")
    host:RegisterScriptParameter(self, "color_mode")
    host:RegisterScriptParameter(self, "fluid_init_color")
    host:RegisterScriptParameter(self, "fluid_phase2_color")
    host:RegisterScriptParameter(self, "fluid_phase3_color")
    host:RegisterScriptParameter(self, "use_multi_phase_color")
    host:RegisterScriptParameter(self, "fluid_init_difficulty")
    host:RegisterScriptParameter(self, "mv_force_scale")
    host:RegisterScriptParameter(self, "float_max")
    host:RegisterScriptParameter(self, "wind_dir")
    host:RegisterScriptParameter(self, "wind_strength")
    host:RegisterScriptParameter(self, "display_background")
    return self.Queue
end

function fluid:Resizeview(size)
    self:_UpdateResolutions(size)
    self:_Reset()
end

--function fluid:Process(pipeline, Original, Scene, Output)
function fluid:Process(context, Original, Scene, Output)
    self:_UpdateRenderParameters(self.host)
    self:_BackupStates()
    -- set read buffer value to zero at first frame
    local first_frame = math.floor(self.first_frame + 0.5)
    local motion_detect_idx = math.floor(self.motion_detect_idx + 0.5)
    if first_frame > 0.0 then
        local zero_val = mathfunction.Color(self.normalized_zero, self.normalized_zero, 0.0, 1.0)
        --self.fbo_velocity_read:PushRenderTarget()
        --self.fbo_velocity_read:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR, zero_val)
        context:BeginRenderPass(self.fbo_velocity_read,apolloengine.RenderTargetEntity.CF_COLOR,zero_val);
        context:EndRenderPass();
        -- self.fbo_pressure_read:PushRenderTarget()
        -- self.fbo_pressure_read:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR, zero_val)
        context:BeginRenderPass(self.fbo_pressure_read,apolloengine.RenderTargetEntity.CF_COLOR, zero_val);
        context:EndRenderPass();
        zero_val = mathfunction.Color(0.5, 0.5, 0.0, 1.0) --motion_vector range is [-1, 1]
        -- self.fbo_motion_vec:PushRenderTarget()
        -- self.fbo_motion_vec:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR, zero_val) --motion_vector == 0 when current frame is first frame
        context:BeginRenderPass(self.fbo_motion_vec,apolloengine.RenderTargetEntity.CF_COLOR, zero_val);
        context:EndRenderPass();
        zero_val = mathfunction.Color(0.0, 0.0, 0.0, 0.0)
        -- self.fbo_color_read:PushRenderTarget()
        -- self.fbo_color_read:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR, zero_val)
        context:BeginRenderPass(self.fbo_color_read,apolloengine.RenderTargetEntity.CF_COLOR, zero_val);
        context:EndRenderPass();
        self.first_frame = 0.0 --update first frame flag
    elseif motion_detect_idx == 0.0 then
        -- get motion vector between last frame and current frame
        -- self.fbo_motion_vec:PushRenderTarget()
        -- self.fbo_motion_vec:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
        context:BeginRenderPass(self.fbo_motion_vec,apolloengine.RenderTargetEntity.CF_COLOR);
        --local motion_vec_material = self.ro_motion_vec_step.MaterialEntities[1]
        self.ro_motion_vec_step_material:SetParameter("LAST_FRAME", self.fbo_last_frame:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        self.ro_motion_vec_step_material:SetParameter("CUR_FRAME", Original:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        -- self.ro_motion_vec_step:Draw(pipeline)
        context:Draw(self.ro_motion_vec_step_renderObj,self.ro_motion_vec_step_material);
        context:EndRenderPass();
    else
        local zero_val = mathfunction.Color(0.5, 0.5, 0.0, 1.0) --motion_vector range is [-1, 1]
        -- self.fbo_motion_vec:PushRenderTarget()
        -- self.fbo_motion_vec:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR, zero_val) --motion_vector == 0 when not detecting
        context:BeginRenderPass(self.fbo_motion_vec,apolloengine.RenderTargetEntity.CF_COLOR, zero_val);
        context:EndRenderPass();
    end

    self.motion_detect_idx = (self.motion_detect_idx + 1) % self.motion_detect_interval

    -- blur motion vectors
    if self.blur_motion_vector > 0.0 then
        -- LOG("Blur motion vector")
        -- self.fbo_divergence:PushRenderTarget()   --reuse fbo_divergence as tmp buffer for blurred motion vector
        -- self.fbo_divergence:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
        context:BeginRenderPass(self.fbo_divergence,apolloengine.RenderTargetEntity.CF_COLOR);
        -- local blur_material = self.ro_blur_step.MaterialEntities[1]
        -- blur_material:SetParameter("BLUR_SOURCE", self.fbo_motion_vec:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        self.ro_blur_step_material:SetParameter("BLUR_SOURCE", self.fbo_motion_vec:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        -- self.ro_blur_step:Draw(pipeline)
        context:Draw(self.ro_blur_step_renderObj,self.ro_blur_step_material);
        self.fbo_motion_vec, self.fbo_divergence = self:_Swap(self.fbo_motion_vec, self.fbo_divergence)
        context:EndRenderPass();
    end

    -- apply external forces
    -- LOG("External force")
    -- self.fbo_velocity_write:PushRenderTarget()
    -- self.fbo_velocity_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_velocity_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local ext_force_material = self.ro_ext_force_step.MaterialEntities[1]
    -- ext_force_material:SetParameter("MOTION_VECTOR", self.fbo_motion_vec:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- ext_force_material:SetParameter("FORCE_TARGET", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_ext_force_step_material:SetParameter("MOTION_VECTOR", self.fbo_motion_vec:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_ext_force_step_material:SetParameter("FORCE_TARGET", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_ext_force_step:Draw(pipeline)
    context:Draw(self.ro_ext_force_step_renderObj,self.ro_ext_force_step_material);
    self.fbo_velocity_read, self.fbo_velocity_write = self:_Swap(self.fbo_velocity_read, self.fbo_velocity_write)
    context:EndRenderPass();
    -- dye color due to motion vector
    -- LOG("Dye color")
    -- self.fbo_color_write:PushRenderTarget()
    -- self.fbo_color_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_color_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
    -- dye_color_material:SetParameter("MOTION_VECTOR", self.fbo_motion_vec:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- dye_color_material:SetParameter("CUR_FRAME", Original:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- dye_color_material:SetParameter("CUR_COLOR", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- dye_color_material:SetParameter("DYE_DIFFICULTY", mathfunction.vector1(self.fluid_init_difficulty))
    self.ro_dye_color_step_material:SetParameter("MOTION_VECTOR", self.fbo_motion_vec:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_dye_color_step_material:SetParameter("CUR_FRAME", Original:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_dye_color_step_material:SetParameter("CUR_COLOR", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_dye_color_step_material:SetParameter("DYE_DIFFICULTY", mathfunction.vector1(self.fluid_init_difficulty))
    if self.use_fluid_init_mask > 0.0 then
        self.ro_dye_color_step_material:SetParameter("DYE_MASK", self.fluid_init_mask)
    end
    -- self.ro_dye_color_step:Draw(pipeline)
    context:Draw(self.ro_dye_color_step_renderObj,self.ro_dye_color_step_material);
    self.fbo_color_read, self.fbo_color_write = self:_Swap(self.fbo_color_read, self.fbo_color_write)
    context:EndRenderPass();
    -- curl step
    -- LOG("Compute curl")
    -- self.fbo_curl:PushRenderTarget()
    -- self.fbo_curl:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_curl,apolloengine.RenderTargetEntity.CF_COLOR);
    --local curl_material = self.ro_curl_step.MaterialEntities[1]
    --curl_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_curl_step_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_curl_step:Draw(pipeline)
    context:Draw(self.ro_curl_step_renderObj,self.ro_curl_step_material);
    context:EndRenderPass();

    -- vorticiy step
    -- LOG("Apply vorticity")
    -- self.fbo_velocity_write:PushRenderTarget()
    -- self.fbo_velocity_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass( self.fbo_velocity_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local vorticity_material = self.ro_vorticity_step.MaterialEntities[1]
    -- vorticity_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- vorticity_material:SetParameter("CURL", self.fbo_curl:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_vorticity_step_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_vorticity_step_material:SetParameter("CURL", self.fbo_curl:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_vorticity_step:Draw(pipeline)
    context:Draw(self.ro_vorticity_step_renderObj,self.ro_vorticity_step_material);
    self.fbo_velocity_read, self.fbo_velocity_write = self:_Swap(self.fbo_velocity_read, self.fbo_velocity_write)
    context:EndRenderPass();

    -- divergence step
    -- LOG("Compute divergence")
    -- self.fbo_divergence:PushRenderTarget()
    -- self.fbo_divergence:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_divergence,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local divergence_material = self.ro_divergence_step.MaterialEntities[1]
    -- divergence_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_divergence_step_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_divergence_step:Draw(pipeline)
    context:Draw(self.ro_divergence_step_renderObj,self.ro_divergence_step_material);
    context:EndRenderPass();

    -- clear pressure step
    -- LOG("Clear pressure")
    -- self.fbo_pressure_write:PushRenderTarget()
    -- self.fbo_pressure_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_pressure_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local clear_pressure_material = self.ro_clear_pressure_step.MaterialEntities[1]
    -- clear_pressure_material:SetParameter("PRESSURE", self.fbo_pressure_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_clear_pressure_step_material:SetParameter("PRESSURE", self.fbo_pressure_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_clear_pressure_step:Draw(pipeline)
    context:Draw(self.ro_clear_pressure_step_renderObj,self.ro_clear_pressure_step_material);
    self.fbo_pressure_read, self.fbo_pressure_write = self:_Swap(self.fbo_pressure_read, self.fbo_pressure_write)
    context:EndRenderPass();
    -- pressure step
    -- LOG("Compute pressure")
    -- local pressure_material = self.ro_pressure_step.MaterialEntities[1]
    -- pressure_material:SetParameter("DIVERGENCE", self.fbo_divergence:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_pressure_step_material:SetParameter("DIVERGENCE", self.fbo_divergence:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    for i = 1, self.pressure_iter_num do
        -- self.fbo_pressure_write:PushRenderTarget()
        -- self.fbo_pressure_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
        context:BeginRenderPass(self.fbo_pressure_write,apolloengine.RenderTargetEntity.CF_COLOR);
        -- pressure_material:SetParameter("PRESSURE", self.fbo_pressure_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        self.ro_pressure_step_material:SetParameter("PRESSURE", self.fbo_pressure_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        -- self.ro_pressure_step:Draw(pipeline)
        context:Draw(self.ro_pressure_step_renderObj,self.ro_pressure_step_material);
        self.fbo_pressure_read, self.fbo_pressure_write = self:_Swap(self.fbo_pressure_read, self.fbo_pressure_write)
        context:EndRenderPass();
    end

    -- pressure force step
    -- LOG("Compute pressure force")
    -- self.fbo_velocity_write:PushRenderTarget()
    -- self.fbo_velocity_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_velocity_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local pressure_force_material = self.ro_pressure_force_step.MaterialEntities[1]
    -- pressure_force_material:SetParameter("PRESSURE", self.fbo_pressure_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- pressure_force_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_pressure_force_step_material:SetParameter("PRESSURE", self.fbo_pressure_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_pressure_force_step_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_pressure_force_step:Draw(pipeline)
    context:Draw(self.ro_pressure_force_step_renderObj,self.ro_pressure_force_step_material);
    self.fbo_velocity_read, self.fbo_velocity_write = self:_Swap(self.fbo_velocity_read, self.fbo_velocity_write)
    context:EndRenderPass();

    -- velocity advection step
    -- LOG("Advect velocity")
    -- self.fbo_velocity_write:PushRenderTarget()
    -- self.fbo_velocity_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_velocity_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local advection_material = self.ro_advection_step.MaterialEntities[1]
    -- advection_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    -- advection_material:SetParameter("DISSIPATION", mathfunction.vector1(self.velocity_dissipation))
    -- advection_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- advection_material:SetParameter("ADVECT_SOURCE", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- advection_material:SetParameter("SRC_FLOAT_MIN", mathfunction.vector1(self.float_min))
    -- advection_material:SetParameter("SRC_FLOAT_MAX", mathfunction.vector1(self.float_max))
    -- advection_material:SetParameter("SRC_MULTI_PHASE", mathfunction.vector1(0.0))
    self.ro_advection_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
    self.ro_advection_step_material:SetParameter("DISSIPATION", mathfunction.vector1(self.velocity_dissipation))
    self.ro_advection_step_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_advection_step_material:SetParameter("ADVECT_SOURCE", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_advection_step_material:SetParameter("SRC_FLOAT_MIN", mathfunction.vector1(self.float_min))
    self.ro_advection_step_material:SetParameter("SRC_FLOAT_MAX", mathfunction.vector1(self.float_max))
    self.ro_advection_step_material:SetParameter("SRC_MULTI_PHASE", mathfunction.vector1(0.0))
    -- self.ro_advection_step:Draw(pipeline)
    context:Draw(self.ro_advection_step_renderObj,self.ro_advection_step_material);
    self.fbo_velocity_read, self.fbo_velocity_write = self:_Swap(self.fbo_velocity_read, self.fbo_velocity_write)
    context:EndRenderPass();
    
    -- color advection step
    -- LOG("Advect color")
    -- self.fbo_color_write:PushRenderTarget()
    -- self.fbo_color_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_color_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- advection_material:SetParameter("TEXEL_SIZE", self.vis_texel_size)
    -- advection_material:SetParameter("DISSIPATION", mathfunction.vector1(self.display_dissipation))
    -- advection_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- advection_material:SetParameter("ADVECT_SOURCE", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- advection_material:SetParameter("SRC_FLOAT_MIN", mathfunction.vector1(0.0))
    -- advection_material:SetParameter("SRC_FLOAT_MAX", mathfunction.vector1(1.0))
    self.ro_advection_step_material:SetParameter("TEXEL_SIZE", self.vis_texel_size)
    self.ro_advection_step_material:SetParameter("DISSIPATION", mathfunction.vector1(self.display_dissipation))
    self.ro_advection_step_material:SetParameter("VELOCITY", self.fbo_velocity_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_advection_step_material:SetParameter("ADVECT_SOURCE", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_advection_step_material:SetParameter("SRC_FLOAT_MIN", mathfunction.vector1(0.0))
    self.ro_advection_step_material:SetParameter("SRC_FLOAT_MAX", mathfunction.vector1(1.0))
    if self.color_mode == 0.0 and self.use_multi_phase_color > 0.0 then
        -- advection_material:SetParameter("SRC_MULTI_PHASE", mathfunction.vector1(1.0))
        self.ro_advection_step_material:SetParameter("SRC_MULTI_PHASE", mathfunction.vector1(1.0))
    end
    -- self.ro_advection_step:Draw(pipeline)
    context:Draw(self.ro_advection_step_renderObj,self.ro_advection_step_material);
    self.fbo_color_read, self.fbo_color_write = self:_Swap(self.fbo_color_read, self.fbo_color_write)
    context:EndRenderPass();

    -- update last frame
    -- LOG("Backup last frame")
    -- self.fbo_last_frame:PushRenderTarget()
    -- self.fbo_last_frame:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_last_frame,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local copy_material = self.ro_copy_step.MaterialEntities[1]
    -- copy_material:SetParameter("CUR_FRAME", Original:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_copy_step_material:SetParameter("CUR_FRAME", Original:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_copy_step:Draw(pipeline)
    context:Draw(self.ro_copy_step_renderObj,self.ro_copy_step_material);
    context:EndRenderPass();

    -- blend step, blend fluid color with background frame
    -- LOG("Blend fluid with background")
    -- self.fbo_color_write:PushRenderTarget()
    -- self.fbo_color_write:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR)
    context:BeginRenderPass(self.fbo_color_write,apolloengine.RenderTargetEntity.CF_COLOR);
    -- local blend_material = self.ro_blend_step.MaterialEntities[1]
    -- blend_material:SetParameter("COLOR", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- blend_material:SetParameter("CUR_FRAME", Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_blend_step_material:SetParameter("COLOR", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    self.ro_blend_step_material:SetParameter("CUR_FRAME", Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_blend_step:Draw(pipeline)
    context:Draw(self.ro_blend_step_renderObj,self.ro_blend_step_material);
    self.fbo_color_read, self.fbo_color_write = self:_Swap(self.fbo_color_read, self.fbo_color_write)
    context:EndRenderPass();

    -- display with mask, alpha blend with background frame
    -- LOG("Display")
	-- Output:PushRenderTarget()
    -- Output:ClearBuffer(apolloengine.RenderTargetEntity.CF_COLOR, mathfunction.Color(0.0, 0.0, 0.0, 1.0))
    context:BeginRenderPass(Output,apolloengine.RenderTargetEntity.CF_COLOR, mathfunction.Color(0.0, 0.0, 0.0, 1.0));
    -- local display_material = self.ro_display_step.MaterialEntities[1]
    if self.display_background > 0.0 then
        -- copy_material:SetParameter("CUR_FRAME", Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        self.ro_copy_step_material:SetParameter("CUR_FRAME", Scene:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
        --self.ro_copy_step:Draw(pipeline)
        context:Draw(self.ro_copy_step_renderObj,self.ro_copy_step_material);
    end
    if self.use_display_mask > 0.0 then
        self.ro_display_step_material:SetParameter("DISPLAY_MASK", self.display_mask)
    end
    self.ro_display_step_material:SetParameter("COLOR", self.fbo_color_read:GetAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0))
    -- self.ro_display_step:Draw(pipeline)
    context:Draw(self.ro_display_step_renderObj,self.ro_display_step_material);
    context:EndRenderPass();
end

----------------------------internal functions----------------------------

function fluid:_GetResolution(screen_size, short_side_resolution)
    if screen_size:x() < screen_size:y() then
        local aspect_ratio = screen_size:y() / screen_size:x()
        return mathfunction.vector2(short_side_resolution, math.ceil(aspect_ratio * short_side_resolution))
    else
        local aspect_ratio = screen_size:x() / screen_size:y()
        return mathfunction.vector2(math.ceil(aspect_ratio * short_side_resolution), short_side_resolution)
    end
end

function fluid:_Swap(a, b)
    return b, a
end

--update resolutions
function fluid:_UpdateResolutions(screen_size)
    local sim_quality = math.ceil(self.simulation_quality)
    local sim_resolution_short = 2^sim_quality * self.sim_resolution_base
    self.simulation_resolution = self:_GetResolution(screen_size, sim_resolution_short)
    local vis_quality = math.ceil(self.visual_quality)
    local vis_resolution_short = 2^vis_quality * self.vis_resolution_base
    self.visual_resolution = self:_GetResolution(screen_size, vis_resolution_short)
    self.frame_resolution  = mathfunction.vector2(screen_size:x(), screen_size:y())
    --texel size
    self.sim_texel_size = mathfunction.vector2(1.0 / self.simulation_resolution:x(), 1.0 / self.simulation_resolution:y())
    self.vis_texel_size = mathfunction.vector2(1.0 / self.visual_resolution:x(), 1.0 / self.visual_resolution:y())
end

function fluid:_UpdateSimResolution(screen_size)
    local sim_quality = math.ceil(self.simulation_quality)
    local sim_resolution_short = 2^sim_quality * self.sim_resolution_base
    self.simulation_resolution = self:_GetResolution(screen_size, sim_resolution_short)
    self.sim_texel_size = mathfunction.vector2(1.0 / self.simulation_resolution:x(), 1.0 / self.simulation_resolution:y())
end

function fluid:_UpdateVisResolution(screen_size)
    local vis_quality = math.ceil(self.visual_quality)
    local vis_resolution_short = 2^vis_quality * self.vis_resolution_base
    self.visual_resolution = self:_GetResolution(screen_size, vis_resolution_short)
    self.vis_texel_size = mathfunction.vector2(1.0 / self.visual_resolution:x(), 1.0 / self.visual_resolution:y())
end

--update render object parameters
function fluid:_UpdateRenderParameters(host)
    --update resolutions if simulation/visual quality changes
    if self.simulation_quality_bak ~= self.simulation_quality then
        -- LOG("Simulation quality change")
        self:_UpdateSimResolution(self.frame_resolution)
    end
    if self.visual_quality_bak ~= self.visual_quality then
        -- LOG("Visualization quality change")
        self:_UpdateVisResolution(self.frame_resolution)
    end
    --recreate render targets and update parameter if resolution changes
    if self.simulation_resolution_bak ~= self.simulation_resolution then
        -- LOG("Simulation resolution change")
        self:_CreateSimResFBOs(host)
        self:_Reset()
        -- local curl_material = self.ro_curl_step.MaterialEntities[1]
        -- curl_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_curl_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        -- local vorticity_material = self.ro_vorticity_step.MaterialEntities[1]
        -- vorticity_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_vorticity_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        -- local divergence_material = self.ro_divergence_step.MaterialEntities[1]
        -- divergence_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_divergence_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        -- local pressure_material = self.ro_pressure_step.MaterialEntities[1]
        -- pressure_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_pressure_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        -- local pressure_force_material = self.ro_pressure_force_step.MaterialEntities[1]
        -- pressure_force_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_pressure_force_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        -- local motion_vec_material = self.ro_motion_vec_step.MaterialEntities[1]
        -- motion_vec_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        self.ro_motion_vec_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        if self.blur_motion_vector > 0.0 then
            -- local blur_material = self.ro_blur_step.MaterialEntities[1]
            -- blur_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
            self.ro_blur_step_material:SetParameter("TEXEL_SIZE", self.sim_texel_size)
        end
    end
    if self.visual_resolution_bak ~= self.visual_resolution then
        -- LOG("Visualization resolution change")
        self:_CreateVisResFBOs(host)
        self:_Reset()
    end
    --update parameters if config changes
    if self.pressure_bak ~= self.pressure then
        -- local clear_pressure_material = self.ro_clear_pressure_step.MaterialEntities[1]
        -- clear_pressure_material:SetParameter("PRESSURE_CONFIG", mathfunction.vector1(self.pressure))
        self.ro_clear_pressure_step_material:SetParameter("PRESSURE_CONFIG", mathfunction.vector1(self.pressure))
    end
    if self.curl_bak ~= self.curl then
        -- local vorticity_material = self.ro_vorticity_step.MaterialEntities[1]
        -- vorticity_material:SetParameter("CURL_CONFIG", mathfunction.vector1(self.curl))
        self.ro_vorticity_step_material:SetParameter("CURL_CONFIG", mathfunction.vector1(self.curl))
    end
    if self.use_display_mask_bak ~= self.use_display_mask then
        -- local display_material = self.ro_display_step.MaterialEntities[1]
        -- display_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_display_mask))
        self.ro_display_step_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_display_mask))
    end
    if self.invert_display_mask_bak ~= self.invert_display_mask then
        -- local display_material = self.ro_display_step.MaterialEntities[1]
        -- display_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_display_mask))
        self.ro_display_step_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_display_mask))
    end
    if self.use_fluid_init_mask_bak ~= self.use_fluid_init_mask then
        -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
        -- dye_color_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_fluid_init_mask))
        self.ro_dye_color_step_material:SetParameter("MASK_VALID", mathfunction.vector1(self.use_fluid_init_mask))
    end
    if self.invert_fluid_init_mask_bak ~= self.invert_fluid_init_mask then
        -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
        -- dye_color_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_fluid_init_mask))
        self.ro_dye_color_step_material:SetParameter("INVERT_MASK", mathfunction.vector1(self.invert_fluid_init_mask))
    end
    if self.blend_mode_bak ~= self.blend_mode then
        -- local blend_material = self.ro_blend_step.MaterialEntities[1]
        -- blend_material:SetParameter("BLEND_MODE", mathfunction.vector1(self.blend_mode))
        self.ro_blend_step_material:SetParameter("BLEND_MODE", mathfunction.vector1(self.blend_mode))
    end
    if self.color_mode_bak ~= self.color_mode then
        -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
        -- dye_color_material:SetParameter("COLOR_MODE", mathfunction.vector1(self.color_mode))
        self.ro_dye_color_step_material:SetParameter("COLOR_MODE", mathfunction.vector1(self.color_mode))
    end
    if self.fluid_init_color_bak ~= self.fluid_init_color then
        local normalized_fluid_init_color = self.fluid_init_color / 255.0
        -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
        -- dye_color_material:SetParameter("FLUID_COLOR", normalized_fluid_init_color)
        self.ro_dye_color_step_material:SetParameter("FLUID_COLOR", normalized_fluid_init_color)
        -- local advection_material = self.ro_advection_step.MaterialEntities[1]
        -- advection_material:SetParameter("SRC_PHASE_1", normalized_fluid_init_color)
        self.ro_advection_step_material:SetParameter("SRC_PHASE_1", normalized_fluid_init_color)
    end
    if self.fluid_phase2_color_bak ~= self.fluid_phase2_color then
        local normalized_fluid_phase2_color = self.fluid_phase2_color / 255.0
        -- local advection_material = self.ro_advection_step.MaterialEntities[1]
        -- advection_material:SetParameter("SRC_PHASE_2", normalized_fluid_phase2_color)
        self.ro_advection_step_material:SetParameter("SRC_PHASE_2", normalized_fluid_phase2_color)
    end
    if self.fluid_phase3_color_bak ~= self.fluid_phase3_color then
        local normalized_fluid_phase3_color = self.fluid_phase3_color / 255.0
        -- local advection_material = self.ro_advection_step.MaterialEntities[1]
        -- advection_material:SetParameter("SRC_PHASE_3", normalized_fluid_phase3_color)
        self.ro_advection_step_material:SetParameter("SRC_PHASE_3", normalized_fluid_phase3_color)
    end
    if self.fluid_init_difficulty_bak ~= self.fluid_init_difficulty then
        -- local dye_color_material = self.ro_dye_color_step.MaterialEntities[1]
        -- dye_color_material:SetParameter("DYE_DIFFICULTY", mathfunction.vector1(self.fluid_init_difficulty))
        self.ro_dye_color_step_material:SetParameter("DYE_DIFFICULTY", mathfunction.vector1(self.fluid_init_difficulty))
    end
    if self.mv_force_scale_bak ~= self.mv_force_scale then
        -- local ext_force_material = self.ro_ext_force_step.MaterialEntities[1]
        -- ext_force_material:SetParameter("MV_FORCE_SCALE", mathfunction.vector1(self.mv_force_scale))
        self.ro_ext_force_step_material:SetParameter("MV_FORCE_SCALE", mathfunction.vector1(self.mv_force_scale))
    end
    if self.wind_dir_bak ~= self.wind_dir or self.wind_strength_bak ~= self.wind_strength then
        local wind_dir = mathfunction.vector2(self.wind_dir:x(), self.wind_dir:y())
        if wind_dir:LengthPow() > 0.0 then
            wind_dir:NormalizeSelf()
        end
        wind_dir = wind_dir * self.wind_strength
        -- local ext_force_material = self.ro_ext_force_step.MaterialEntities[1]
        -- ext_force_material:SetParameter("WIND_FIELD", wind_dir)
        self.ro_ext_force_step_material:SetParameter("WIND_FIELD", wind_dir)
    end
    if self.float_max_bak ~= self.float_max then
        self.float_min       = -(self.float_max + 1)
        self.normalized_zero = -self.float_min / (self.float_max - self.float_min)
        -- local ext_force_material = self.ro_ext_force_step.MaterialEntities[1]
        -- ext_force_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- ext_force_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_ext_force_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_ext_force_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local curl_material = self.ro_curl_step.MaterialEntities[1]
        -- curl_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- curl_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_curl_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_curl_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local vorticity_material = self.ro_vorticity_step.MaterialEntities[1]
        -- vorticity_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- vorticity_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_vorticity_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_vorticity_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local divergence_material = self.ro_divergence_step.MaterialEntities[1]
        -- divergence_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- divergence_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_divergence_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_divergence_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local clear_pressure_material = self.ro_clear_pressure_step.MaterialEntities[1]
        -- clear_pressure_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- clear_pressure_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_clear_pressure_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_clear_pressure_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local pressure_material = self.ro_pressure_step.MaterialEntities[1]
        -- pressure_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- pressure_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_pressure_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_pressure_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local pressure_force_material = self.ro_pressure_force_step.MaterialEntities[1]
        -- pressure_force_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- pressure_force_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_pressure_force_step_material:SetParameter("FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_pressure_force_step_material:SetParameter("FLOAT_MAX", mathfunction.vector1(self.float_max))
        -- local advection_material = self.ro_advection_step.MaterialEntities[1]
        -- advection_material:SetParameter("VEL_FLOAT_MIN", mathfunction.vector1(self.float_min))
        -- advection_material:SetParameter("VEL_FLOAT_MAX", mathfunction.vector1(self.float_max))
        self.ro_advection_step_material:SetParameter("VEL_FLOAT_MIN", mathfunction.vector1(self.float_min))
        self.ro_advection_step_material:SetParameter("VEL_FLOAT_MAX", mathfunction.vector1(self.float_max))
    end
end

--backup current states to trigger render parameter update
function fluid:_BackupStates()
    self.pressure_bak               = self.pressure
    self.curl_bak                   = self.curl
    self.use_display_mask_bak       = self.use_display_mask
    self.invert_display_mask_bak    = self.invert_display_mask
    self.use_fluid_init_mask_bak    = self.use_fluid_init_mask
    self.invert_fluid_init_mask_bak = self.invert_fluid_init_mask
    self.simulation_quality_bak     = self.simulation_quality
    self.visual_quality_bak         = self.visual_quality
    self.simulation_resolution_bak  = self.simulation_resolution
    self.visual_resolution_bak      = self.visual_resolution
    self.blend_mode_bak             = self.blend_mode
    self.color_mode_bak             = self.color_mode
    self.fluid_init_color_bak       = self.fluid_init_color
    self.fluid_phase2_color_bak     = self.fluid_phase2_color
    self.fluid_phase3_color_bak     = self.fluid_phase3_color
    self.fluid_init_difficulty_bak  = self.fluid_init_difficulty
    self.mv_force_scale_bak         = self.mv_force_scale
    self.float_max_bak              = self.float_max
    self.wind_dir_bak               = self.wind_dir
    self.wind_strength_bak          = self.wind_strength
end

function fluid:_CreateSimResFBOs(host)
    self.fbo_velocity_read  = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.simulation_resolution)
    self.fbo_velocity_write = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.simulation_resolution)
    self.fbo_pressure_read  = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.simulation_resolution)
    self.fbo_pressure_write = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.simulation_resolution)
    self.fbo_curl           = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.simulation_resolution)
    self.fbo_divergence     = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_A, self.simulation_resolution)
    self.fbo_last_frame     = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_A, self.simulation_resolution)
    self.fbo_motion_vec     = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.simulation_resolution)
end

function fluid:_CreateVisResFBOs(host)
    self.fbo_color_read  = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.visual_resolution)
    self.fbo_color_write = host:CreateRenderTarget(apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE, self.visual_resolution)
end

function fluid:_Reset()
    self.first_frame = 1.0
    self.motion_detect_idx = 0
end

return fluid