--@author      : Chen Wei
--@date        : 2019-06-25
--@description : lua script for LineEffect
--@version     : 1.0

require "utility"

local apolloengine          = require "apolloengine"
local mathfunction          = require "mathfunction"
local apollonode            = require "apolloutility.apollonode"
local defined               = require "apolloutility.defiend"
local renderqueue           = require "apolloutility.renderqueue"
local HistoryFrameManager   = require "apolloutility.basehistoricalframe"
local LineStripRender       = require "apolloutility.linestriprender"
local LineSystem            = require "lineeffect"
local venuscore             = require "venuscore"
local venusjson             = require "venusjson"
local videodecet            = require "videodecet"

-- Note on the camera sequences:
-- HistoryFrameManager -> motion_detect_camera -> line strip render
-- Don't mess with it.

local LineEffect = {}

function LineEffect:Initialize(cameraLayer)
    self.cameraLayer = cameraLayer
    self.camera = renderqueue:GetCamera(cameraLayer)
    self.initialized = false
end

function LineEffect:LoadConfig(configPath)

    local rootconfig = venusjson.LaodJsonFile(configPath);
    if rootconfig == nil then
        return false
    end
    if rootconfig.line == nil then
        return false
    end
    local config = rootconfig.line.config;
    if config == nil then
        return false
    end

    local rootDir = string.match(configPath, "(.+)/[^/]*%.%w+$")
    venuscore.IFileSystem:SetResourcePath(rootDir .. "/")

    local line_num = config.line_num
    local vertex_num_per_line = config.vertex_num_per_line
    local line_system_num = config.line_system_num
    local is_vertical = config.is_vertical

    -- create line system
    self.line_num            = line_num;
    self.vertex_num_per_line = vertex_num_per_line;
    self.line_system_num     = line_system_num;
    self.is_vertical         = is_vertical;

    self.line_system         = LineSystem.LineEffect(line_num, vertex_num_per_line, line_system_num, is_vertical);

    -- set history frame manger
    self.history_frame_manager = HistoryFrameManager(
                                 self.camera,
                                 2,
                                 self.camera:GetSequence() - 2,
                                 defined.flip_material_path);
    --self.history_frame_manager:SetVideoTexture(
    --        renderqueue:GetTexture(
    --                renderqueue:GetLink(self.cameraLayer)))

    -- set motion detect camera
    local fbo_size = mathfunction.vector2(vertex_num_per_line, line_num);
    self.motion_detect_camera = apollonode.CameraNode(fbo_size);
    self.motion_detect_camera:Activate();
    self.motion_detect_camera:SetSequence(self.camera:GetSequence() - 1);

    -- create render target & tex & tex_stream
    self.rt, self.tex = self:_CreateRenderTarget();
    self.tex_stream = nil

    -- attach render target to camera
    self.motion_detect_camera:AttachRenderTarget(self.rt);

    -- create uniform slots
    apolloengine.ShaderEntity.UNIFORM_CUR_FRAME = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
            "UNIFORM_CUR_FRAME");
    apolloengine.ShaderEntity.UNIFORM_PRE_FRAME = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
            "UNIFORM_PRE_FRAME");
    apolloengine.ShaderEntity.UNIFORM_TEX_SIZE  = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
            "UNIFORM_TEX_SIZE");
    apolloengine.ShaderEntity.UNIFORM_IS_HORIZONTAL_LINE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,
            "UNIFORM_IS_HORIZONTAL_LINE");

    -- set quad node to draw fbo(camera)
    local material = "docs:lineeffect/material/line.material"

    self.quad_node = apollonode.QuadNode();
    self.quad_node:SetShow(true);
    self.quad_node:CreateResource(material, true);
    self.quad_node:SetSequence(self.camera:GetSequence() - 1);


    self.line_strip_render_queue = {};

    for i = 1, line_system_num, 1 do
        --NOTE(Chen Wei): need modify line_num / line_system_num
        local line_strip_render = LineStripRender(
                line_num / line_system_num,
                vertex_num_per_line,
                mathfunction.vector4(1.0, 1.0, 1.0, 1.0),
                self.camera:GetSequence());
        table.insert(self.line_strip_render_queue, line_strip_render);
    end

    self.initialized = true
    return true
end

function LineEffect:ReleaseResource()
    self.line_system = nil
    self.history_frame_manager = nil
    self.motion_detect_camera = nil
    self.tex_stream = nil
    self.quad_node = nil
    self.initialized = false
    self.line_strip_render_queue = nil
    self.rt = nil
    self.tex = nil
end

function LineEffect:_CreateRenderTarget()
    local clear_color = mathfunction.Color(0.0, 0.0, 0.0, 0.0);
    local resolution  = self.motion_detect_camera:GetCameraResolution();
    
    local rt = apolloengine.RenderTargetEntity();
    rt:PushMetadata(apolloengine.RenderTargetMetadata(apolloengine.RenderTargetEntity.RT_RENDER_TARGET_2D,
                                                      apolloengine.RenderTargetEntity.ST_SWAP_UNIQUE,
                                                      clear_color,
                                                      mathfunction.vector4(0, 0, resolution:x(), resolution:y()),
                                                      resolution)
                    );

    local tex = rt:MakeTextureAttachment(apolloengine.RenderTargetEntity.TA_COLOR_0);
    tex:PushMetadata(apolloengine.TextureBufferMetadata(resolution,
                                                        apolloengine.TextureEntity.TT_TEXTURE2D,
                                                        apolloengine.TextureEntity.TU_READ,
                                                        apolloengine.TextureEntity.PF_R8G8B8A8,
                                                        1, false,
                                                        apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                        apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                                                        apolloengine.TextureEntity.TF_LINEAR,
                                                        apolloengine.TextureEntity.TF_LINEAR));
                    
    
    rt:CreateResource();

    return rt, tex;
end

function LineEffect:Update()

    if not self.initialized then
        return
    end

    -- update history frame manager
    self.history_frame_manager:SetVideoTexture(videodecet:GetVideoTexture())
    self.history_frame_manager:SetShow(true);
    self.history_frame_manager:Update();

    -- draw to calculate motion info
    local tex_queue = self.history_frame_manager:GetFrame();
    self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_CUR_FRAME, tex_queue[1]);
    self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_PRE_FRAME, tex_queue[2]);

    -- get resolution
    local tex_size = self.camera:GetCameraResolution();
    self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_TEX_SIZE, tex_size);

    if self.is_vertical then
        self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_IS_HORIZONTAL_LINE, mathfunction.vector1(0.0));
    else
        self.quad_node:SetParameter(apolloengine.ShaderEntity.UNIFORM_IS_HORIZONTAL_LINE, mathfunction.vector1(1.0));
    end
    
    
    -- read motion data from texture to texture stream
    self.tex_stream = self.rt:GetAttachmentStream(apolloengine.RenderTargetEntity.TA_COLOR_0);

    -- add force based on motion info
    self.line_system:SetForceFromVerticesMotionInfo(self.tex_stream);

    -- simulate line effect system
    -- for i = 1, self.line_system_num, 1 do
    --     self.line_system:Step(i - 1, 0.005, 6);
    -- end

    self.line_system:StepAll(0.005, 6);

    -- draw lines
    for i = 1, self.line_system_num, 1 do
        local vertex_stream = self.line_system:VertexPositionStream(i - 1);
        self.line_strip_render_queue[i]:Update(vertex_stream);
    end
    
end

function LineEffect:SetShow(show)
    if self.line_strip_render_queue ~= nil then
        for i = 1, #self.line_strip_render_queue do
            self.line_strip_render_queue[i]:SetShow(show)
        end
    end
end

return LineEffect
