require "utility"
local ae = require "apolloengine"
local vc = require "venuscore"
local mf = require "mathfunction"
local vj = require "venusjson"

local obj = {}

local BLEND_NORMAL = "normal"
local BLEND_SCREEN = "screen"


local function _CreateRenderTarget(size)
    local fbo = ae.RenderTargetEntity()
    fbo:PushMetadata(
            ae.RenderTargetMetadata(
                    ae.RenderTargetEntity.RT_RENDER_TARGET_2D,
                    ae.RenderTargetEntity.ST_SWAP_UNIQUE,
                    mf.Color(0, 0, 0, 0),
                    mf.vector4(
                            0,
                            0,
                            size:x(),
                            size:y()),
                    mf.vector2(
                            size:x(),
                            size:y())))
    local tex = fbo:MakeTextureAttachment(
            ae.RenderTargetEntity.TA_COLOR_0)
    tex:PushMetadata(
            ae.TextureRenderMetadata(
                    ae.RenderTargetEntity.ST_SWAP_UNIQUE,
                    mf.vector2(size:x(), size:y())))
    fbo:SetJobType(vc.IJob.JT_SYNCHRONOUS)
    fbo:CreateResource()
    return fbo
end

function obj:Initialize(host, size)

    self.host = host
    self.targetSize = { size:x(), size:y() }
    self.order = 100

    self.CLIPART_TRANSFORM = ae.IMaterialSystem:NewParameterSlot(
            ae.ShaderEntity.UNIFORM,
            "CLIPART_TRANSFORM");

    self.CLIPART_BLEND = ae.IMaterialSystem:NewParameterSlot(
            ae.ShaderEntity.UNIFORM,
            "CLIPART_BLEND");

    self.TEXTURE_CLIPART = ae.IMaterialSystem:NewParameterSlot(
            ae.ShaderEntity.UNIFORM,
            "TEXTURE_CLIPART");

    self.transformRenderer = host:CreateRenderObject("comm:documents/material/clipart_transform.material")

    self.blendMaterials = {}

    self.blendMaterials[BLEND_NORMAL] = "comm:documents/material/clipart_blend_alpha.material"
    self.blendMaterials[BLEND_SCREEN] = "comm:documents/material/clipart_blend_screen.material"

    self.blendRenderer = host:CreateRenderObject(self.blendMaterials[BLEND_NORMAL])

    --self.transformTarget = host:CreateRenderTarget(ae.RenderTargetEntity.ST_SWAP_UNIQUE, size)
    self.transformTarget = _CreateRenderTarget(size)

    self:_InitializeParameters()

    self:_UpdateParameters()

    return self.order
end

function obj:Resizeview(size)

    self.transformTarget = _CreateRenderTarget(size)

    self.lastClipartConfig = nil
    self.targetSize = { size:x(), size:y() }

    self:_UpdateParameters()
end

function obj:Process(pipeline, Original, Scene, Output)

    self:_UpdateParameters()

    self.transformTarget:PushRenderTarget()
    self.transformTarget:ClearBuffer(ae.RenderTargetEntity.CF_COLOR)

    self.transformRenderer:Draw(pipeline)

    Output:PushRenderTarget()
    Output:ClearBuffer(ae.RenderTargetEntity.CF_COLOR)

    self.blendRenderer:SetParameter(
            ae.ShaderEntity.TEXTURE_DIFFUSE,
            Scene:GetAttachment(ae.RenderTargetEntity.TA_COLOR_0))

    self.blendRenderer:SetParameter(
            self.TEXTURE_CLIPART,
            self.transformTarget:GetAttachment(ae.RenderTargetEntity.TA_COLOR_0))

    self.blendRenderer:Draw(pipeline)
end

local function _CreateEmptyTexture()
    local tex = ae.TextureEntity()
    tex:PushMetadata(ae.TextureBufferMetadata(mf.vector2(4, 4)))
    tex:SetJobType(vc.IJob.JT_SYNCHRONOUS)
    tex:CreateResource()
    return tex
end

local function _CreateDefaultConfig()
    local json = {
        size = { 4, 4 },
        transform = "match",
        blend = "normal"
    }
    return vj.StoreJsonString(json)
end

local function FitIn(w1, h1, w2, h2)
    local aspect1 = w1 / h1
    local aspect2 = w2 / h2
    local alignW = aspect1 >= aspect2
    if alignW then
        return w2, w2 / aspect1
    else
        return h2 * aspect1, h2
    end
end

local function FillIn(w1, h1, w2, h2)
    local aspect1 = w1 / h1
    local aspect2 = w2 / h2
    local alignW = aspect1 <= aspect2
    if alignW then
        return w2, w2 / aspect1
    else
        return h2 * aspect1, h2
    end
end

function obj:_InitializeParameters()
    self.clipartTexture = _CreateEmptyTexture()
    self.lastClipartTexture = nil

    self.clipartConfig = _CreateDefaultConfig()
    self.lastClipartConfig = nil

end

function obj:_UpdateParameters()
    if self.lastClipartTexture ~= self.clipartTexture then
        self.transformRenderer:SetParameter(
                ae.ShaderEntity.TEXTURE_DIFFUSE,
                self.clipartTexture)
        self.lastClipartTexture = self.clipartTexture
    end

    if self.lastClipartConfig ~= self.clipartConfig then

        local config = vj.LoadJsonString(self.clipartConfig)

        local material = self.blendMaterials[config.blend]

        if material ~= nil then
            self.blendRenderer = self.host:CreateRenderObject(material)
        else
            self.blendRenderer = self.host:CreateRenderObject(self.blendMaterials[BLEND_NORMAL])
        end

        local originSize = config.size

        local imageW = originSize[1]
        local imageH = originSize[2]

        local viewW = self.targetSize[1]
        local viewH = self.targetSize[2]

        local transform = config.transform

        if transform == "match" then
            self.transformRenderer:SetParameter(
                    self.CLIPART_TRANSFORM,
                    mf.Matrix44:CreateScaleMatrix(mf.vector3(1.0, -1.0, 1.0)))
        elseif transform == "pixel" then
            local scaleHorz = imageW / viewW
            local scaleVert = imageH / viewH
            self.transformRenderer:SetParameter(
                    self.CLIPART_TRANSFORM,
                    mf.Matrix44:CreateScaleMatrix(mf.vector3(scaleHorz, -scaleVert, 1.0)))
        elseif transform == "fill" then
            local newW, newH = FillIn(imageW, imageH, viewW, viewH)
            local scaleHorz = newW / viewW
            local scaleVert = newH / viewH
            self.transformRenderer:SetParameter(
                    self.CLIPART_TRANSFORM,
                    mf.Matrix44:CreateScaleMatrix(mf.vector3(scaleHorz, -scaleVert, 1.0)))
        elseif transform == "fit" then
            local newW, newH = FitIn(imageW, imageH, viewW, viewH)
            local scaleHorz = newW / viewW
            local scaleVert = newH / viewH
            self.transformRenderer:SetParameter(
                    self.CLIPART_TRANSFORM,
                    mf.Matrix44:CreateScaleMatrix(mf.vector3(scaleHorz, -scaleVert, 1.0)))
        else
            self.transformRenderer:SetParameter(
                    self.CLIPART_TRANSFORM,
                    mf.Matrix44:CreateScaleMatrix(mf.vector3(1.0, -1.0, 1.0)))
        end

        self.lastClipartConfig = self.clipartConfig
    end
end

return obj;
