require "venusdebug"
require "utility"
local Object = require "classic"
local ae = require "apolloengine"
local an = require "apolloutility.apollonode"
local mf = require "mathfunction"
local vc = require "venuscore"
local vp = require "videoprocess"

local VideoFrameNode = an.RenderNode:extend()

local function GetMaterialPath(pixelFormat, alphaFormat)
  local pathBase = "comm:documents/material/video_frames/"
  if pixelFormat == vp.Defines.PF_I420 or
          pixelFormat == vp.Defines.PF_I444 then
    if alphaFormat == nil then
      return pathBase .. "y_cr_cb.material"
    else
      return pathBase .. "y_cr_cb_a.material"
    end
  elseif pixelFormat == vp.Defines.PF_I420SP then
    if alphaFormat == nil then
      return pathBase .. "y_cr_cb_sp.material"
    else
      return pathBase .. "y_cr_cb_sp_a.material"
    end
  elseif pixelFormat == vp.Defines.PF_RGB then
    return pathBase .. "r_g_b.material"
  elseif pixelFormat == vp.Defines.PF_RGBA then
    return pathBase .. "r_g_b_a.material"
  elseif pixelFormat == vp.Defines.PF_GRAY then
    return pathBase .. "gray.material"
  else
    return pathBase .. "r_g_b.material"
  end
end

local function GetColorOffsetAndMatrix(pixelFormat, uvOrder, rbOrder, colorMatrixId, range)
  LOG(string.format("GetColorOffsetAndMatrix %d %d %d %d %d",
          pixelFormat, uvOrder, rbOrder, colorMatrixId, range))
  local offset = nil
  local matrix = nil
  if pixelFormat == vp.Defines.PF_I420 or
          pixelFormat == vp.Defines.PF_I420SP or
          pixelFormat == vp.Defines.PF_I444 then
    if colorMatrixId == vp.Defines.BT_601 then
      if range == vp.Defines.FULL_RANGE then
        offset = mf.vector3(0.0, -0.5, -0.5)
        matrix = mf.Matrix33(
                1.0,     1.0,    1.0,
                0.0,    -0.344,  1.772,
                1.402,  -0.714,  0.0)
      elseif range == vp.Defines.VIDEO_RANGE then
        offset = mf.vector3(-16.0 / 255.0, -0.5, -0.5)
        matrix = mf.Matrix33(
                1.164, 1.164, 1.164,
                0.0, -0.392, 2.017,
                1.596, -0.813, 0.0)
      end
    elseif colorMatrixId == vp.Defines.BT_709 then
      if range == vp.Defines.FULL_RANGE then
        offset = mf.vector3(0.0, -0.5, -0.5)
        matrix = mf.Matrix33(
                1.0, 1.0, 1.0,
                0.0, -0.1873, 1.8556,
                1.5748, -0.4681, 0.0)
      elseif range == vp.Defines.VIDEO_RANGE then
        offset = mf.vector3(-16.0 / 255.0, -0.5, -0.5)
        matrix = mf.Matrix33(
                1.1644, 1.1644, 1.1644,
                0.0, -0.2132, 2.1124,
                1.7927, -0.5329, 0.0)
      end
    end
    if uvOrder == vp.Defines.UV_YVU then
      matrix = matrix * mf.Matrix33(
              1, 0, 0,
              0, 0, 1,
              0, 1, 0)
    end
  elseif pixelFormat == vp.Defines.PF_RGB or
          pixelFormat == vp.Defines.PF_RGBA then
    offset = mf.vector3(0, 0, 0)
    matrix = mf.Matrix33(
            1, 0, 0,
            0, 1, 0,
            0, 0, 1)
    if rbOrder == vp.Defines.RB_BGR then
      matrix = matrix * mf.Matrix33(
              0, 0, 1,
              0, 1, 0,
              1, 0, 0)
    end
  elseif pixelFormat == vp.Defines.PF_GRAY then
    offset = mf.vector3(0, 0, 0)
    matrix = mf.Matrix33(
            1, 0, 0,
            0, 1, 0,
            0, 0, 1)
  end
  return offset, matrix
end

local function GetVideoMatrix(rotation, flip)
  local flp = nil
  local rot = nil
  if rotation == vp.Defines.ROTATE_NONE then
    rot = mf.Matrix44()
  elseif rotation == vp.Defines.ROTATE_90 then
    rot = mf.Matrix44(
            mf.vector4( 0, 1, 0, 0),
            mf.vector4(-1, 0, 0, 0),
            mf.vector4( 0, 0, 1, 0),
            mf.vector4( 0, 0, 0, 1))
  elseif rotation == vp.Defines.ROTATE_180 then
    rot = mf.Matrix44(
            mf.vector4(-1, 0, 0, 0),
            mf.vector4( 0,-1, 0, 0),
            mf.vector4( 0, 0, 1, 0),
            mf.vector4( 0, 0, 0, 1))
  elseif rotation == vp.Defines.ROTATE_270 then
    rot = mf.Matrix44(
            mf.vector4( 0,-1, 0, 0),
            mf.vector4( 1, 0, 0, 0),
            mf.vector4( 0, 0, 1, 0),
            mf.vector4( 0, 0, 0, 1))
  end
  if flip == vp.Defines.FLIP_NONE then
    flp = mf.Matrix44()
  elseif flip == vp.Defines.FLIP_HORZ then
    flp = mf.Matrix44(
            mf.vector4(-1, 0, 0, 0),
            mf.vector4( 0, 1, 0, 0),
            mf.vector4( 0, 0, 1, 0),
            mf.vector4( 0, 0, 0, 1))
  elseif flip == vp.Defines.FLIP_VERT then
    flp = mf.Matrix44(
            mf.vector4( 1, 0, 0, 0),
            mf.vector4( 0,-1, 0, 0),
            mf.vector4( 0, 0, 1, 0),
            mf.vector4( 0, 0, 0, 1))
  end
  return flp * rot
end

function VideoFrameNode:new(videoParam, alphaParam)

  VideoFrameNode.super.new(self)

  local verticesStream = ae.VertexStream()
  local indicesStream = ae.IndicesStream()
  verticesStream:SetVertexType(
          ae.ShaderEntity.ATTRIBUTE_POSITION,
          ae.VertexBufferEntity.DT_FLOAT,
          ae.VertexBufferEntity.DT_HALF_FLOAT,
          4);
  verticesStream:SetVertexType(
          ae.ShaderEntity.ATTRIBUTE_COORDNATE0,
          ae.VertexBufferEntity.DT_FLOAT,
          ae.VertexBufferEntity.DT_HALF_FLOAT,
          2);
  indicesStream:SetIndicesType(
          ae.IndicesBufferEntity.IT_UINT16);


  local meshCoord = {
    mf.vector4( 1.0,  1.0, 0.0, 1.0),
    mf.vector4(-1.0,  1.0, 0.0, 1.0),
    mf.vector4(-1.0, -1.0, 0.0, 1.0),
    mf.vector4( 1.0, -1.0, 0.0, 1.0),
  }

  local texCoord = {
    mf.vector2(1.0, 0.0),
    mf.vector2(0.0, 0.0),
    mf.vector2(0.0, 1.0),
    mf.vector2(1.0, 1.0),
  }

  local indices = {
    0, 1, 3, 2
  }

  for i = 1, #meshCoord do
    verticesStream:PushVertexData(
            ae.ShaderEntity.ATTRIBUTE_POSITION,
            meshCoord[i])
  end
  for i = 1, #texCoord do
    verticesStream:PushVertexData(
            ae.ShaderEntity.ATTRIBUTE_COORDNATE0,
            texCoord[i])
  end
  for i = 1, #indices do
    indicesStream:PushIndicesData(indices[i])
  end

  self.VIEW_PROJECT_MATRIX = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "VF_VIEW_PROJECT_MATRIX")

  self.MODEL_MATRIX = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "VF_MODEL_MATRIX")

  self.VIDEO_MATRIX = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "VF_VIDEO_MATRIX")

  self.COLOR_OFFSET = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "COLOR_OFFSET")

  self.COLOR_MATRIX = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "COLOR_MATRIX")

  self.TEXTURE_1 = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "TEXTURE_1")

  self.TEXTURE_2 = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "TEXTURE_2")

  self.TEXTURE_3 = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "TEXTURE_3")

  self.TEXTURE_4 = ae.IMaterialSystem:NewParameterSlot(
          ae.ShaderEntity.UNIFORM,
          "TEXTURE_4")

  local materialPath = GetMaterialPath(
          videoParam:GetPixelFormat(),
          alphaParam ~= nil and alphaParam:GetPixelFormat() or nil)

  self:CreateResource(
          materialPath,
          ae.RenderObjectEntity.RM_TRIANGLE_STRIP,
          verticesStream,
          indicesStream)

  local colorOffset, colorMatrix = GetColorOffsetAndMatrix(
          videoParam:GetPixelFormat(),
          videoParam:GetUVOrder(),
          videoParam:GetRBOrder(),
          videoParam:GetColorMatrix(),
          videoParam:GetColorRange())

  self:SetViewProjectMatrix(mf.Matrix44())

  self:SetModelMatrix(mf.Matrix44())

  local videoMatrix = GetVideoMatrix(
          videoParam:GetRotation(),
          videoParam:GetFlip())

  self:SetParameter(
          self.VIDEO_MATRIX,
          videoMatrix)

  self:SetParameter(
          self.COLOR_OFFSET,
          colorOffset)

  self:SetParameter(
          self.COLOR_MATRIX,
          colorMatrix)

  self:_SetupTextures(videoParam, alphaParam)

end

function VideoFrameNode:SetModelMatrix(m)
  self:SetParameter(
          self.MODEL_MATRIX,
          m)
end

function VideoFrameNode:SetViewProjectMatrix(m)
  self:SetParameter(
          self.VIEW_PROJECT_MATRIX,
          m)
end

function VideoFrameNode:_SetupTextures(videoParam, alphaParam)

  self.textures = {}

  local video2Tex = nil
  local videoTexCount = 0

  if videoParam ~= nil then

    video2Tex = vp.VideoFrameToTextureStream()

    if not video2Tex:Setup(videoParam) then
      LOG("video2Tex:Setup failed")
    end

    self.video2Tex = video2Tex

    videoTexCount = video2Tex:GetTextureStreamCount()
  end

  local alpha2Tex = nil
  local alphaTexCount = 0

  if alphaParam ~= nil then

    alpha2Tex = vp.VideoFrameToTextureStream()

    if not alpha2Tex:Setup(alphaParam) then
      LOG("alpha2Tex:Setup failed")
    end

    self.alpha2Tex = alpha2Tex

    alphaTexCount = video2Tex:GetTextureStreamCount()

  end

  LOG("_SetupTextures video tex count is " .. tostring(videoTexCount))
  LOG("_SetupTextures alpha tex count is " .. tostring(alphaTexCount))

  for i = 1, videoTexCount do
    if #self.textures == 4 then
      break
    end
    local tex = ae.TextureEntity()
    tex:PushMetadata(
            ae.TextureBufferMetadata(
                    ae.TextureEntity.TU_WRITE,
                    1,
                    false,
                    ae.TextureEntity.TW_CLAMP_TO_BORDER,
                    ae.TextureEntity.TW_CLAMP_TO_BORDER,
                    ae.TextureEntity.TF_LINEAR,
                    ae.TextureEntity.TF_LINEAR,
                    video2Tex:GetTextureStream(i - 1)))
    tex:CreateResource()
    table.insert(self.textures, tex)
  end

  for i = 1, alphaTexCount do
    if #self.textures == 4 then
      break
    end
    local tex = ae.TextureEntity()
    tex:PushMetadata(
            ae.TextureBufferMetadata(
                    ae.TextureEntity.TU_WRITE,
                    1,
                    false,
                    ae.TextureEntity.TW_CLAMP_TO_BORDER,
                    ae.TextureEntity.TW_CLAMP_TO_BORDER,
                    ae.TextureEntity.TF_LINEAR,
                    ae.TextureEntity.TF_LINEAR,
                    alpha2Tex:GetTextureStream(i - 1)))
    tex:CreateResource()
    table.insert(self.textures, tex)
  end

  local textureSlots = {
    self.TEXTURE_1,
    self.TEXTURE_2,
    self.TEXTURE_3,
    self.TEXTURE_4,
  }

  for i = 1, #self.textures do
    self:SetParameter(
            textureSlots[i],
            self.textures[i])
  end

end

function VideoFrameNode:PutVideoFrame(videoFrame, alphaFrame)

  local video2Tex = self.video2Tex
  local alpha2Tex = self.alpha2Tex
  local videoTexCount = 0
  local alphaTexCount = 0

  if video2Tex ~= nil then
    videoTexCount = video2Tex:GetTextureStreamCount()
    if not video2Tex:PutVideoFrame(videoFrame) then
      LOG("video2Tex:PutVideoFrame failed")
    end
  end

  if alpha2Tex ~= nil then
    alphaTexCount = alpha2Tex:GetTextureStreamCount()
    if not alpha2Tex:PutVideoFrame(alphaFrame) then
      LOG("alpha2Tex:PutVideoFrame failed")
    end
  end

  local index = 1

  for i = 1, videoTexCount do
    --self.textures[index]:SubstituteTextureBuffer(video2Tex:GetTextureStream(i - 1))
    self.textures[index]:ChangeTextureBuffer(video2Tex:GetTextureStream(i - 1))
    index = index + 1
  end

  for i = 1, alphaTexCount do
    --self.textures[index]:SubstituteTextureBuffer(alpha2Tex:GetTextureStream(i - 1))
    self.textures[index]:ChangeTextureBuffer(alpha2Tex:GetTextureStream(i - 1))
    index = index + 1
  end

end


return VideoFrameNode
