local vc = require "libvenuscore"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local ArgInfo = require "bluecore.core.arg_info"

local EventInfo = vc.Object:extend("EventInfo");


function EventInfo:new(classInfo)
  self.owner = classInfo ;
  self._RefactorList = setmetatable({}, { __mode = 'v' })
end

function EventInfo:GetUid()
  return self._nodeId
end

function EventInfo:GetEventName()
  return self._eventName
end

function EventInfo:GetInputInfo()
  return self._inputInfos
end

function EventInfo:GetName()
  return self._nodeName ;
end

-- register from bluefunction instance
function EventInfo:RegisterCustomEventNode(customEventNode)

  self._inputInfos = {}

  -- node uid
  self._nodeId = customEventNode:GetUid();

  -- import! not changeable! generated function name!
  self._eventName = customEventNode:GetEventName();

  -- just for display, it's changeable
  self._nodeName =  customEventNode:GetName();

  local inputInfos = customEventNode:GetInputInfo();

  -- input argument,  the first blueprint pin about 'this' is exclusive
  if inputInfos ~= nil then
    for _, reflectInfo in pairs(inputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterBlueFunPinInfo(reflectInfo);
      table.insert(self._inputInfos, argInfo);
    end
  end

  customEventNode:AddRefactor(self);
end

function EventInfo:AddRefactor(callback)

  if _KRATOSEDITOR then

    local cb = callback ;

    if (cb._OnInfoChange == nil ) then error("_OnInfoChange not implement") end
    if (cb._OnEditorPinDelete == nil) then error("_OnEditorPinDelete not implement") end
    if (cb._OnEditorPinAdd == nil) then error("_OnEditorPinAdd not implement") end

    if (cb._OnDeserializePost == nil ) then error("_OnDeserializePost not implement") end
    if (cb._OnCreateWithEditor == nil ) then error("_OnCreateWithEditor not implement") end
    if (cb._OnDeleteEd == nil ) then error("_OnDeleteEd not implement") end

    table.insert(self._RefactorList, callback);

  end -- 非编辑器启动不加入refactor列表
end


function EventInfo:RemoveRefactor(callback)
  if _KRATOSEDITOR then
    for idx, cb in pairs(self._RefactorList) do
      if cb == callback then
        self._RefactorList[idx] = nil
      end
    end
  end
end


function EventInfo:InfoChange(reflectIdx, type, val, isInput)

  if not isInput then error("EventInfo InfoChange false") end

  local argInfo = self._inputInfos
  argInfo = argInfo[reflectIdx]

  if type == BD.FuncInfoProp.Name then
    argInfo:ChangeName(val)
  elseif type == BD.FuncInfoProp.BaseType then
    argInfo:ChangeRtti(val, BU:GetDefaultByRtti(val));
  elseif type == BD.FuncInfoProp.Value then
    argInfo:ChangeDefault(val);
  end

  for _, cb in pairs(self._RefactorList) do
    cb:InfoChange(reflectIdx, type, val, isInput);
  end

end

function EventInfo:EditorPinDelete(isInput, idx)

  if not isInput then error("EventInfo EditorPinDelete false") end

  table.remove(self._inputInfos, idx)

  for _, cb in pairs(self._RefactorList) do
    cb:EditorPinDelete(isInput, idx);
  end

end


function EventInfo:EditorPinAdd(isInput, blueFuncPinInfo)

  if not isInput then error("EventInfo EditorPinAdd false") end

  local argInfo = ArgInfo()
  argInfo:RegisterBlueFunPinInfo(blueFuncPinInfo);

  table.insert(self._inputInfos,  argInfo);

  for _, cb in pairs(self._RefactorList) do
    cb:EditorPinAdd(isInput, blueFuncPinInfo);
  end

end


function EventInfo:NameChanged(oldName, newName )
  self._nodeName = newName
  for _, cb in pairs(self._RefactorList) do
    cb:NameChanged(oldName, newName);
  end
end


function EventInfo:RegisterPrefab(prefabJson)

  self._nodeId = prefabJson.uid

  self._eventName = prefabJson.eventName

  self._nodeName = prefabJson.name

  self._inputInfos = {}

  local inputInfos = prefabJson.inputInfos;

  -- input argument
  if inputInfos ~= nil then
    for _, reflectInfo in pairs(inputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterPrefab(reflectInfo);
      table.insert(self._inputInfos, argInfo);
    end
  end


end

function EventInfo:ToBeDeletedEd()
  for _, cb in pairs(self._RefactorList) do
    cb:ToBeDeletedEd();
  end
  self.owner:ToBeDeletedEd(self)
  self.owner = nil
end


function EventInfo:Dump()
  LOG("----event:"..tostring(self:GetEventName()))
  LOG("----  uid:"..tostring(self:GetUid()))
  LOG("---- name:"..tostring(self:GetName()))
  local inputInfos = self:GetInputInfo()
  for i, argInfo in pairs(inputInfos) do
    LOG("---- arg "..tostring(i)..":")
    argInfo:Dump();
  end
end

function EventInfo:GetRefactorList()
  return self._RefactorList;
end

return EventInfo