local vc = require "libvenuscore"
local BU = require "bluecore.blueutility"
local BD = require "bluecore.bluedefined"
local AE = require "apolloengine"
local FuncInfo = require "bluecore.core.func_info"
local EventInfo = require "bluecore.core.event_info"
local DispatcherInfo = require "bluecore.core.dispatcher_info"
local MemberIterator = require "venuscore.rtti.mpairs"
local ClassInfo = vc.Object:extend("ClassInfo");

local gIndex = 1

function ClassInfo:new()

  self._ftnList = {}
  self._eventList = {}
  self._dispatcherList = {}
  self._blueprint = setmetatable({}, { __mode = 'v' });
  -- list is "uid" as index
  -- array is order as index
  self._prefabMetaFileFullPath = nil
  -- from Prefab file, Prefab Class
  -- from Blueprint, Blueprint Class,
  -- in PrefabEditor, the current Prefab file's Class is BlueprintClass
  self._refCount = 0

if BD.LEAK_DEBUG then
  local index = gIndex
  gIndex = gIndex + 1
  self.index = index -- not used
  ERROR("^^^^^^^^^ ClassInfo new called:" ..tostring(index));
  local prox = newproxy(true)
  getmetatable(prox).__gc = function() ERROR("^^^^^^^^^ ClassInfo CG called:" ..tostring(index)); end
  self[prox] = true
end


end

function ClassInfo:AddRef()
  self._refCount = self._refCount + 1
end

function ClassInfo:UnRef()
  self._refCount =  self._refCount - 1
  if self._refCount <= 0 then WARNING("Class "..tostring(self._clsName)..", Ref "..tostring(self._refCount)) end
end

function ClassInfo:GetRef()
  return self._refCount;
end

function ClassInfo:IsBlueprintClass()
  -- BlueprintClass 在当前场景(大场景/prefab场景) 该类有且只有一个实例
  return self._prefabMetaFileFullPath == nil
end

function ClassInfo:ClearPrefabClass(blueprint)
  local metaFullPath = self._prefabMetaFileFullPath
  self._prefabMetaFileFullPath = nil
  return metaFullPath ;
end

function ClassInfo:RegisterBlueprint(blueprint)
  -- class info
  self._clsId = blueprint:GetClassId() ;
  self._clsName = blueprint:GetClassName() ;

  -- variable info


  -- function info
  self._ftnList = {}
  local ftnList = blueprint:GetBlueFtnList()
  for ftnUid, blueFtn in pairs(ftnList) do
    local funcInfo = FuncInfo(self)
    funcInfo:RegisterBlueFunction(blueFtn);
    self._ftnList[ftnUid] = funcInfo;
  end

  -- custom event info
  self._eventList = {}
  local eventList = blueprint:GetCustomEventNodes();
  for _, eventNode in pairs(eventList) do
    self:AppendCustomEvent(eventNode);
  end

  -- event dispatcher
  self._dispatcherList = {}
  local dispatchers =  blueprint:GetEventDispatchers();
  for _, dispatcher in pairs(dispatchers) do
    self:AppendEventDispatcher(dispatcher)
  end

  -- blueprint reference
  self._blueprint[1] = blueprint;

end

-- maybe nil
function ClassInfo:GetBluePrint()
  return self._blueprint[1];
end


function ClassInfo:RegisterPrefab(prefabJson, prefabMetaFileFullPath)

  self._prefabMetaFileFullPath = prefabMetaFileFullPath;

  self._clsId = prefabJson.clsId ;
  self._clsName = prefabJson.clsName ;

  local clsInfo = prefabJson.clsInfo ;
  self._ftnList = {}
  local ftnArray = clsInfo.ftnArray
  if ftnArray ~= nil then
    for _, ftnInfoJson in pairs(ftnArray) do
      local funcInfo = FuncInfo(self)
      -- convert
      ftnInfoJson.funcUid = tonumber(ftnInfoJson.funcUid)

      funcInfo:RegisterPrefab(ftnInfoJson);

      self._ftnList[ftnInfoJson.funcUid] = funcInfo;
    end
  end

  local eventArray = clsInfo.eventArray
  if eventArray ~= nil then
    for _, eventInfoJson in pairs(eventArray) do
      local eventInfo = EventInfo(self)
      -- convert
      eventInfoJson.uid = tonumber(eventInfoJson.uid)

      eventInfo:RegisterPrefab(eventInfoJson);
      self._eventList[eventInfoJson.uid] = eventInfo;
    end
  end

  local dispatcherArray = clsInfo.dispatcherArray
  if dispatcherArray ~= nil then
    for _, dispatcherJson in pairs(dispatcherArray) do
      local dispatcherInfo = DispatcherInfo(self)
      -- convert
      dispatcherJson.uid = tonumber(dispatcherJson.uid)

      dispatcherInfo:RegisterPrefab(dispatcherJson);
      self._dispatcherList[dispatcherJson.uid] = dispatcherInfo;
    end
  end

end

function ClassInfo:GetCustomEventNodes()
    return self._eventList
end

function ClassInfo:GetBlueFtnList()
  return self._ftnList
end

function ClassInfo:GetEventDispatchers()
  return self._dispatcherList
end

function ClassInfo:GetClassId()
  return self._clsId
end

-- 与blueprint一样的接口
function ClassInfo:GetClassName()
  return self._clsName
end

function ClassInfo:SetClassNameEd(newName)

  if newName == self._clsName then
    return true
  end

  local BlueRtti = require "bluecore.core.blue_rtti"
  local classInfos = BlueRtti.GetClassesEd()
  -- 名字冲突
  for _, classInfo in pairs(classInfos) do
    if classInfo ~= self then
      local name = classInfo:GetClassName();
      if name == newName then
        return false
      end
    end
  end

  self._clsName = newName

  -- 目前用到类信息的地方只有blueprintType
  local blueprintType = BlueRtti.GetRttiByClsId(self._clsId)
  blueprintType:UpdateDisplayName(newName)


  return true
end


-- static function
function ClassInfo.BlueprintClassInfoToJson(rootnode)
  --把蓝图函数信息序列化出来
  --Name inputinfo outputinfo
  --读prefab信息是不是一定要反序列化回来成为node才能读？  不  每次反序列化回来效率太低 必须序列化到meta文件中 而且逻辑不好融合到别的地方
  local SerializeInfoList = {};
  local bluecomp = rootnode:GetComponent(AE.Node.CT_BLUEPRINT);
  if not bluecomp then
    return nil;
  end
  local blueprint = BU.GetIns(bluecomp);
  local classId = blueprint:GetClassId();-- classId唯一标记
  local clsName = blueprint:GetClassName();

  -- serialize blue-function
  local funcList = blueprint:GetBlueFtnList();
  if next(funcList) ~= nil then

    local ftnArray = {}

    for _, func in pairs(funcList) do

      local funcInfo = {};

      funcInfo.funcName = func:GetName();
      funcInfo.funcUid = func:GetFuncId();

      funcInfo.inputInfos = {};
      local inputInfos = func:GetInputInfo();
      for _, argInfo in pairs(inputInfos) do
        local saveInfo = {};
        for k,v in MemberIterator(argInfo) do
          if k == "default_value" then
            local BundleSystem = require "venuscore.bundle.bundlesystem"
            local info = {}
            BundleSystem:_SerializeTraverseNode(v, nil, nil,info)
            local _, result = next(info)
            saveInfo[k] = result -- lua is table <--> json is array
          else
            saveInfo[k] = tostring(v);
          end
        end
        table.insert(funcInfo.inputInfos, saveInfo);
      end
      funcInfo.outputInfos = {};
      local outputInfos = func:GetOutputInfo();
      for _, argInfo in pairs(outputInfos) do
        local saveInfo = {};
        for k,v in MemberIterator(argInfo) do
          if k == "default_value" then
            local BundleSystem = require "venuscore.bundle.bundlesystem"
            local info = {}
            BundleSystem:_SerializeTraverseNode(v, nil, nil,info)
            local _, result = next(info)
            saveInfo[k] = result -- lua is table <--> json is array
          else
            saveInfo[k] = tostring(v);
          end

        end
        table.insert(funcInfo.outputInfos, saveInfo);
      end
      table.insert(ftnArray, funcInfo);
    end
    SerializeInfoList.ftnArray = ftnArray ;
  end

  -- serialize EventNode
  local eventNodeArray = blueprint:GetCustomEventNodes();
  if next(eventNodeArray) ~= nil then

    local eventArray = {}

    for _, eventNode in pairs(eventNodeArray) do

      local eventInfo = {};

      eventInfo.name = eventNode:GetName();
      eventInfo.eventName = eventNode:GetEventName();
      eventInfo.uid = eventNode:GetUid();

      eventInfo.inputInfos = {};
      local inputInfos = eventNode:GetInputInfo();
      for _, argInfo in pairs(inputInfos) do -- argInfo is BlueFuncPinInfo
        local saveInfo = {};
        for k,v in MemberIterator(argInfo) do
          if k == "default_value" then
            local BundleSystem = require "venuscore.bundle.bundlesystem"
            local info = {}
            BundleSystem:_SerializeTraverseNode(v, nil, nil,info)
            local _, result = next(info)
            saveInfo[k] = result -- lua is table <--> json is array
          else
            saveInfo[k] = tostring(v);
          end
        end
        table.insert(eventInfo.inputInfos, saveInfo);
      end

      table.insert(eventArray, eventInfo);
    end

    SerializeInfoList.eventArray = eventArray ;
  end


  local dispatcherList = blueprint:GetEventDispatchers();
  if next(dispatcherList) ~= nil then

    local dispatcherArray = {}

    for _, dispatcher in pairs(dispatcherList) do

      local dispatcherInfo = {};

      dispatcherInfo.name = dispatcher:GetName();
      dispatcherInfo.uid = dispatcher:GetUid();

      dispatcherInfo.inputInfos = {};
      local inputInfos = dispatcher:GetInputInfo();
      for _, argInfo in pairs(inputInfos) do
        local saveInfo = {};
        for k,v in MemberIterator(argInfo) do
          if k == "default_value" then
            local BundleSystem = require "venuscore.bundle.bundlesystem"
            local info = {}
            BundleSystem:_SerializeTraverseNode(v, nil, nil,info)
            local _, result = next(info)
            saveInfo[k] = result -- lua is table <--> json is array
          else
            saveInfo[k] = tostring(v);
          end
        end
        table.insert(dispatcherInfo.inputInfos, saveInfo);
      end

      table.insert(dispatcherArray, dispatcherInfo);

    end

    SerializeInfoList.dispatcherArray = dispatcherArray ;

  end


  return classId, clsName, SerializeInfoList;

end

function ClassInfo:Dump()
  WARNING("--ClsId: "..tostring(self._clsId));
  WARNING("--CName: "..tostring(self._clsName));
  for ftnUid, ftnInfo in pairs(self._ftnList) do
    WARNING("---- Function   ID: "..tostring(ftnUid))
    ftnInfo:Dump();
  end

  for uid, eventInfo in pairs(self._eventList) do
    WARNING("---- EventNode  ID: "..tostring(uid))
    eventInfo:Dump();
  end

  for uid, dispatcherInfo in pairs(self._dispatcherList) do
    WARNING("---- Dispatcher ID: "..tostring(uid))
    dispatcherInfo:Dump();
  end

end


function ClassInfo:AppendCustomEvent(eventNode)

  if self._eventList[eventNode:GetUid()] then
    LOG("AppendCustomEvent again "..tostring(eventNode:GetUid()) );
    return
  end
  LOG("AppendCustomEvent "..tostring(eventNode:GetUid()) );
  local eventInfo = EventInfo(self)
  eventInfo:RegisterCustomEventNode(eventNode)
  self._eventList[eventNode:GetUid()] = eventInfo

end


function ClassInfo:AppendEventDispatcher(dispatcher)

  local uid = dispatcher:GetUid() ;

  if self._dispatcherList[uid] then
    LOG("AppendEventDispatcher again "..tostring(uid) );
    return
  end

  LOG("AppendEventDispatcher "..tostring(uid) );

  local dispatcherInfo = DispatcherInfo(self)
  dispatcherInfo:RegisterEventDispatcher(dispatcher)
  self._dispatcherList[uid] = dispatcherInfo

end

function ClassInfo:AppendBlueFunction(bluefunction)

  local uid = bluefunction:GetFuncId();

  if self._ftnList[uid] then
    LOG("AppendBlueFunction again "..tostring(uid) );
    return
  end

  LOG("AppendBlueFunction "..tostring(uid) );

  local funcInfo = FuncInfo(self)
  funcInfo:RegisterBlueFunction(bluefunction)
  self._ftnList[uid] = funcInfo

end

-- 单独一个类成员被删除回调
function ClassInfo:ToBeDeletedEd(info)
  if info:isType(FuncInfo) then
    local funcInfo = info
    local ftnUid = funcInfo:GetFuncId()
    self._ftnList[ftnUid] = nil;
  elseif info:isType(EventInfo) then
    local evenInfo = info
    local eventId = evenInfo:GetUid()
    self._eventList[eventId] = nil;
  elseif info:isType(DispatcherInfo) then
    local dispatcherInfo = info
    local dispatcherId = dispatcherInfo:GetUid()
    self._dispatcherList[dispatcherId] = nil;
  else
    error("ClassInfo miss matach ")
  end
end


-- 类整体被销毁
function ClassInfo:DeleteClassInfoEd()
  WARNING("DeleteClassInfoEd begin");

  -- 相当于类中的各个成员被单独删除了
  for id, ftnInfo in pairs(self._ftnList) do
    ftnInfo:ToBeDeletedEd();
  end

  for id, eventInfo in pairs(self._eventList) do
    eventInfo:ToBeDeletedEd();
  end

  for id, dispatcherInfo in pairs(self._dispatcherList) do
    dispatcherInfo:ToBeDeletedEd();
  end

  WARNING("DeleteClassInfoEd end");
end




return ClassInfo;