

#include "common.inc"
#include "shadow.inc"

#ifndef LIGHT_INC
#define LIGHT_INC

struct SurfaceInput
{
	float3 albedo;
	float3 specular;
	float gloss;
	float3 emission;
	float3 worldNormal;
	float3 worldPos;
	float atten;
};

struct LightInput
{
	float3 lightColor;
	float3 viewDir;
	float3 lightDir;
	float3 normalDir;
	float atten;
	float3 emission;
	float3 albedo;
	float specular;
	float gloss;
};

float3 BlinLighting(LightInput input)
{
	float3 halfV = normalize(input.lightDir + input.viewDir);
	float dotNL = clamp(dot(input.normalDir.xyz, input.lightDir.xyz), 0.0, 1.0);
	float dotNH = clamp(dot(halfV.xyz, input.normalDir.xyz), 0.0, 1.0);
	float3 diffuse = dotNL * input.lightColor.xyz;
	float3 specular = pow(dotNH, input.gloss * 128.0) * input.specular * input.lightColor.xyz;

	float3 fragColor = (diffuse.xyz * input.albedo + specular.xyz) * input.atten + input.emission.xyz * input.albedo;
	return fragColor;
}
// worldNorm
#ifdef PointLight
float3 LightCalc(SurfaceInput input)
{
	float3 lightDir = LIGHT_POSITION.xyz - input.worldPos.xyz;
	float dis = sqrt(dot(lightDir.xyz, lightDir.xyz));
	float disAtten = clamp(dis * LIGHT_RANGE_INV, 0.0, 1.0);
	float attenation = (1.0 - disAtten) /  ( LIGHT_ATTENUATION.x + disAtten * LIGHT_ATTENUATION.y + disAtten * disAtten * LIGHT_ATTENUATION.z );
	float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);
	lightDir.xyz = normalize(lightDir.xyz);

	LightInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.specular = input.specular;
	lightParam.gloss = input.gloss;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.viewDir.xyz = worldViewDir.xyz;
	lightParam.lightDir.xyz = lightDir.xyz;
	lightParam.atten = attenation * input.atten;
	lightParam.lightColor = LIGHT_COLOR.xyz;
	lightParam.emission = AMBIENT_COLOR.xyz;

	return BlinLighting(lightParam);
}
#elif DirLight
float3 LightCalc(SurfaceInput input)
{
	float3 lightDir = -LIGHT_GIVEN_DIRECTION.xyz;
	float atten = 1.0;
    float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);

	LightInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.specular = input.specular;
	lightParam.gloss = input.gloss;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.emission = input.emission.xyz;
	lightParam.viewDir.xyz = worldViewDir.xyz;
	lightParam.lightDir.xyz = lightDir.xyz;
	lightParam.atten = atten * input.atten;
	lightParam.lightColor = LIGHT_COLOR.xyz;

	return BlinLighting(lightParam);
}
#elif SpotLight
float3 LightCalc(SurfaceInput input)
{
	float3 lightDir = LIGHT_POSITION.xyz - input.worldPos.xyz;
	float dis = sqrt(dot(lightDir.xyz, lightDir.xyz));
	float disAtten = clamp(dis * LIGHT_RANGE_INV, 0.0, 1.0);
	float attenation = (1.0 - disAtten) /  ( LIGHT_ATTENUATION.x + disAtten * LIGHT_ATTENUATION.y + disAtten * disAtten * LIGHT_ATTENUATION.z );
	float attenAngle = 1.0;

	lightDir.xyz = normalize(lightDir.xyz);
	attenAngle = clamp( 1.0 - ( LIGHT_INNER_DIFF_INV.x - dot(lightDir.xyz, -LIGHT_GIVEN_DIRECTION.xyz) ) * LIGHT_INNER_DIFF_INV.y, 0.0, 1.0 );
	attenation *= attenAngle;

	float3 worldViewDir = CAMERA_WORLDPOSITION.xyz - input.worldPos.xyz;
	worldViewDir.xyz = normalize(worldViewDir.xyz);

	LightInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.specular = input.specular;
	lightParam.gloss = input.gloss;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.emission = input.emission.xyz;
	lightParam.viewDir.xyz = worldViewDir.xyz;
	lightParam.lightDir.xyz = lightDir.xyz;
	lightParam.atten = attenation * input.atten;
	lightParam.lightColor = LIGHT_COLOR.xyz;

	return BlinLighting(lightParam);
}
#elif NoLight
float3 LightCalc(in SurfaceInput input)
{
	LightInput lightParam;
	lightParam.albedo = input.albedo;
	lightParam.specular = input.specular;
	lightParam.emission = input.emission.xyz;
	lightParam.gloss = 0.0;
	lightParam.normalDir.xyz = input.worldNormal;
	lightParam.viewDir.xyz = float3(0.0, 0.0, 1.0);
	lightParam.lightDir.xyz = float3(0.0, 0.0, 1.0);
	lightParam.atten = 1.0;
	lightParam.lightColor = float3(0.0, 0.0, 0.0);

	return BlinLighting(lightParam);
}
#endif

#endif
