local TransNode = require "apolloutility.apollonode.trasnnode"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"

local SkeletonNode = TransNode:extend();
local radiusscale = math.pi / 180;



function SkeletonNode:new()
  SkeletonNode.super.new(self);
  self.skeleton = self.node:CreateComponent(apolloengine.Node.CT_SKELETON)
  self.animations = nil;
  self.children = {2,3,0,5,6,0,8,9,0,11,12,0,14,0}
end

function SkeletonNode:SetPoseEstimateBoneNames(bonenames)
  self.bonenamemap = bonenames;
end

function SkeletonNode:SetBoneTwist(bonenames)
  self.bonetwist = bonenames;
end

function SkeletonNode:SetSkeleton(skeletonPath,skeletonType)
  self.skeleton:PushMetadata(apolloengine.SkeletonFileMetaData(skeletonType,0,skeletonPath));
end

function SkeletonNode:AddAnimation(animationPath, animationType, frameCount)
  self.skeleton:PushMetadata(apolloengine.KeyFrameFileMetaData(animationType,1,animationPath));
  self.skeleton:SetAnimationInterval(0, frameCount - 1);
end

function SkeletonNode:CreateResource(skeletonPath,skeletonType,defaultAnimation,animationList)
  self.skeleton:PushMetadata(apolloengine.SkeletonFileMetaData(skeletonType,0,skeletonPath));
  local animationListCnt = #animationList;
  for i = 1, animationListCnt, 1 do
    local animationInfo = animationList[i];
    self.skeleton:PushMetadata(apolloengine.KeyFrameFileMetaData(skeletonType,1,animationInfo[1]));
    self.skeleton:SetAnimationInterval(0,animationInfo[2] - 1);
  end
  self.skeleton:CreateResource();
  if animationListCnt > 0 then
    self.skeleton:SetDefaultAnimation(animationList[1][1]); --设置默认动画
  end

end

function SkeletonNode:BindPose()
  self.Tdir = {}
  self.Trot = {}
  self.Tpos = {}
  for i=1,14,1 do
    self.Trot[i] = self:GetTRot(i);
    self.Tdir[i] = self:GetTDir(i);
    self.Tpos[i] = self:GetTPos(i)
  end
end

function SkeletonNode:GetTPos(boneidx)

  local name = self.bonenamemap[boneidx];

  if name==nil then
    return nil
  end
  
  local joint = self:GetJoint(name);
  local comp = joint.trans;
  local pos = comp:GetWorldPosition();
  return {pos:x(),pos:y(),pos:z()};
end


function SkeletonNode:GetTRot(boneidx)
  if self.children[boneidx] == 0 then
    return nil
  end

  local name = self.bonenamemap[boneidx];

  if name==nil then
    return nil
  end
  local joint = self:GetJoint(name);
  local comp = joint.trans;

  return comp:GetWorldRotation();
end

function SkeletonNode:GetTDir(boneidx)
  if self.children[boneidx] == 0 then
    return nil
  end
  local name1 = self.bonenamemap[boneidx];
  local name2 = self.bonenamemap[self.children[boneidx]];
  if name1==nil or name2==nil then
    return nil
  end
  local joint1 = self:GetJoint(name1);
  local comp1 = joint1.trans;
  
  local joint2 = self:GetJoint(name2);
  local comp2 = joint2.trans;
  
  local pos1 = comp1:GetWorldPosition();
  local pos2 = comp2:GetWorldPosition();
  
  local dir =  pos2 - pos1;
  dir:NormalizeSelf();

  return dir;
end

function SkeletonNode:UpdateSkeleton(pos3d)
  local target = {}
  for i =1 , #self.children do
    local joint = self:GetJoint(self.bonenamemap[i]);
    local trans = joint.trans;

    if self.children[i] ~= 0  then
      local mypos = mathfunction.vector3(pos3d[i][1],-pos3d[i][2],-pos3d[i][3]);
      local chidx = self.children[i]
      local chpos = mathfunction.vector3(pos3d[chidx][1],-pos3d[chidx][2],-pos3d[chidx][3]);
      
      local dir =  chpos - mypos;
      dir:NormalizeSelf();
      local rotation = mathfunction.Quaternion();
      rotation:AxisToAxis(self.Tdir[i],dir);

      
      local jointtwist = nil;
      if self.bonetwist~=nil and self.bonetwist[i]~=nil and  self.bonetwist[i]~='' then
        jointtwist = self:GetJoint(self.bonetwist[i]);
      end
      if joint ~= nil then


        local rotated = self.Trot[i]*rotation
        trans:SetWorldRotation( rotated );
        if jointtwist~=nil then
          local transtwist = jointtwist.trans;
          transtwist:SetWorldRotation( rotated );
        end
        
      end
    end
    target[self.bonenamemap[i]] = trans : GetWorldPosition();
  end
  return target;
end

function SkeletonNode:UpdateTargetsAndKinematics(targets)
  for name, pos in pairs(targets) do    
    self:SetTarget(name, pos);
  end  

  local err = self:UpdateKinematics();
  print("ik loss is :"..err);
  return;
end

function SkeletonNode:SetLoop(loop)
    self.skeleton:Loop(loop);
end

function SkeletonNode:Reset(frame)
if frame then
    self.skeleton:Reset(frame)
else
    self.skeleton:Reset()
end  
end

function SkeletonNode:Play()
    self.skeleton:Play()
end

function SkeletonNode:Stop()
    self.skeleton:Stop()
end


function SkeletonNode:GetFrameCount()
    return self.framecount;
end

function SkeletonNode:AttachNodeToJoint(jointName,node)
    self.skeleton:AttachNodeToJoint(jointName,node);
end

local function _TransNodeCast(rawnode)
  local transnode = {}
  transnode = TransNode:cast(transnode);
  transnode.node = rawnode;
  transnode.position = mathfunction.vector3();
  transnode.trans = rawnode:GetComponent(apolloengine.Node.CT_TRANSFORM);
  return transnode;
end

function SkeletonNode:GetJoint(name)
    local rawnode = self.skeleton:GetJoint(name);
    if rawnode then
      return _TransNodeCast(rawnode);
    end
    ERROR("unkown joint with name '"..name.."'");
    return nil;      
end

function SkeletonNode:GetJointNames()
    return self.skeleton:GetJointNames();
end

function SkeletonNode:GetBoneRoots()
  return self.skeleton:GetBoneRoots();
end

function SkeletonNode:_CreateKinematicsComponent()
  if not self.ik then
    self.ik = self.node:CreateComponent(apolloengine.Node.CT_KINEMATICS);
  end  
end

function SkeletonNode:CreateJoint(joint, lx, ux, ly, uy, lz, uz)
  self:CreateJointV3(
    joint,
    mathfunction.vector3(lx * radiusscale, ly * radiusscale, lz * radiusscale),
    mathfunction.vector3(ux * radiusscale, uy * radiusscale, uz * radiusscale));
end

function SkeletonNode:CreateJointV3(name, rotlimlow, rotlimup)
  self:_CreateKinematicsComponent();
  return self.ik:CreateJoint(name, rotlimlow, rotlimup);
end

function SkeletonNode:SetTarget(name, t)
  return self.ik:SetTarget(name, t);
end

function SkeletonNode:CreateEffector(name)
  return self.ik:CreateEffector(name);
end

function SkeletonNode:SetRootJoint(name)
  return self.ik:SetRootJoint(name);
end

function SkeletonNode:LinkJoints(j1, j2)
  return self.ik:LinkJoints(j1, j2);
end

function SkeletonNode:BuildKinematics()
  self.ik:BuildKinematics();
end

function SkeletonNode:UpdateKinematics()
  return self.ik:UpdateKinematics();
end

return SkeletonNode;