local Object            = require "classic"
local apolloengine      = require "apolloengine"
local apollonode        = require "apolloutility.apollonode"
local mathfunction      = require "mathfunction"
local venuscore         = require "venuscore"
local likeapp           = require "likeapp"
local apolldefined      = require "apolloutility.defiend"
local defined           = require "liveportrait.defined"
local utility           = require "liveportrait.utility"
local wrinkleRenderer   = require "liveportrait.wrinklerenderer"

local portraitRender = Object:extend();

function portraitRender:new()
  self:_GenerateFlipLayer()
  self.wrinkle = nil;
  self.enableWrinkle = false;
end

function portraitRender:Clear()
  self.wrinkle = nil;
end

function portraitRender:SetParams(ts, featurePoints, texCoords, enableProj, headCenter, foreheadCenter, lfaceCenter, lmouthCorner, rfaceCenter, rmouthCorner, triangles)
  self.tex  = self:_GenerateTexture(ts)
  self.size = ts:GetSize()
  self.points = featurePoints
  self.texCoords = texCoords
  self.enableFP = enableProj
  self.headCenter = headCenter
  self.foreheadCenter = foreheadCenter
  self.lfaceCenter = lfaceCenter
  self.lmouthCorner = lmouthCorner;
  self.rfaceCenter = rfaceCenter;
  self.rmouthCorner = rmouthCorner;
  
  self:_GenerateRenderContexts(triangles)
  
end

function portraitRender:UpdateWrinkleTexture(facelmks, fhlmks)  
  if self.wrinkle ~= nil then
    self.wrinkle:WarpImage(self.size, facelmks, fhlmks);
  end
  
  local wrinkle_tex = self.wrinkle:GetWrinkleTex(1);
  local forehead_mask = self.wrinkle:GetWrinkleTex(2);
  local ldecree_mask = self.wrinkle:GetWrinkleTex(3);
  local rdecree_mask = self.wrinkle:GetWrinkleTex(4);  
  
  self.renderNode:SetParameter(self.wrinkleTexSlot, wrinkle_tex);
  self.renderNode:SetParameter(self.foreheadMaskTexSlot, forehead_mask);
  self.renderNode:SetParameter(self.ldecreeMaskTexSlot, ldecree_mask);
  self.renderNode:SetParameter(self.rdecreeMaskTexSlot, rdecree_mask);  
end

function portraitRender:EnableWrinkle(enable, wrinkle_config, wrinkle_tex, forehead_mask, ldecree_mask, rdecree_mask, forehead_factor, decree_factor)
  self.enableWrinkle = enable;
  
  self.foreheadFactor = forehead_factor;
  self.decreeFactor = decree_factor;
  
  if enable == true then   
    if self.wrinkle == nil then
      self.wrinkle = wrinkleRenderer(wrinkle_config, wrinkle_tex, forehead_mask, ldecree_mask, rdecree_mask);
      self.wrinkle:SetRenderSequence(defined.SEQUENCE);
    end
  end  
end

function portraitRender:Reset()
  if self.renderNode ~= nil then
    self.renderNode:SetShow(false)
  end
end

-- Create FBO context based on input fbosize
function portraitRender:SetFboSize(fbosize)
  self.portraitCamera, self.flipedPortraitTex = utility.CreateRenderTarget(fbosize, true, true);
  self.portraitCamera:SetSequence(defined.SEQUENCE); 
  self.portraitCamera:SetClearColor(mathfunction.Color(0.0, 0.0, 0.0, 1.0));

  self.flipCamera, self.portraitTex = utility.CreateRenderTarget(fbosize, false, true);
  self.flipCamera:SetSequence(defined.SEQUENCE + 1); 
  self.flipCamera:SetClearColor(mathfunction.Color(0.0, 0.0, 0.0, 1.0));
  self.flipnode:SetParameter(apolloengine.ShaderEntity.TEXTURE_DIFFUSE,self.flipedPortraitTex)
end

-- Update
function portraitRender:Update(def, featurePoints, headCenter, foreheadCenter, lfaceCenter, lmouthCorner, rfaceCenter, rmouthCorner, modelview)
  self.points = featurePoints
  
  utility.UpdateVertexStream(self.renderNode, self.vtxStream, apolloengine.ShaderEntity.ATTRIBUTE_POSITION, self.points)
  
  local curForeheadPoints  = mathfunction.vector3array();
  curForeheadPoints:PushBack(headCenter);
  curForeheadPoints:PushBack(foreheadCenter);  
  self.renderNode:SetParameter(self.curForeheadSlot, curForeheadPoints);
  
  local lFacePoints  = mathfunction.vector3array();
  lFacePoints:PushBack(lfaceCenter);
  lFacePoints:PushBack(lmouthCorner);  
  self.renderNode:SetParameter(self.curLMouthCornerSlot, lFacePoints);
  
  local rFacePoints  = mathfunction.vector3array();
  rFacePoints:PushBack(rfaceCenter);
  rFacePoints:PushBack(rmouthCorner);  
  self.renderNode:SetParameter(self.curRMouthCornerSlot, rFacePoints);
  
  self.renderNode:SetParameter(self.viewSlot, modelview);

  self:SetShow(true);
end

-- Setshow 
function portraitRender:SetShow(show)
   if self.flipnode ~= nil then
    self.flipnode:SetShow(show)
  end
  
  if self.renderNode ~= nil then
    self.renderNode:SetShow(show);
  end 
end

function portraitRender:GetPortraitTex()
  return self.portraitTex;
end

function portraitRender:_GenerateFlipLayer()
  local materialPath  = apolldefined.flip_material_path;  
  
  self.flipnode = apollonode.QuadNode();  
  self.flipnode:CreateResource(materialPath,true,false);
  self.flipnode:SetSequence(defined.SEQUENCE + 1);
  self.flipnode:SetShow(false);
end

-- Generate Render Contexts
-- 1. Init vertex/indices stream
-- 2. Load material
-- 3. Create RenerNode
function portraitRender:_GenerateRenderContexts(triangles)  
  
  self.vtxStream, self.idxStream = self:_InitStreams(triangles);  
  
  apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.ATTRIBUTE, "ATTRIBUTE_COORDNATE2");  
  local texSlot                 = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE_DIFFUSE");  
  local projSlot                = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "PROJECTION");
  self.viewSlot                 = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "MODELVIEW");
  local mixedSlot               = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "MIXED");
  self.wrinkleTexSlot           = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE_WRINKLE");
  self.foreheadMaskTexSlot      = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE_FOREHEAD_MASK");
  self.ldecreeMaskTexSlot       = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE_LDECREE_MASK");
  self.rdecreeMaskTexSlot       = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "TEXTURE_RDECREE_MASK");
  self.enableWrinkleSlot        = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ENABLE_WRINKLE");  
  local origForeheadSlot        = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ORIGIN_FOREHEAD_CENTER");
  self.curForeheadSlot          = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "CUR_FOREHEAD_CENTER");
  local origLMouthCornerSlot    = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ORIGIN_LMOUTH_CORNER");
  self.curLMouthCornerSlot      = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "CUR_LMOUTH_CORNER");
  local origRMouthCornerSlot    = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "ORIGIN_RMOUTH_CORNER");
  self.curRMouthCornerSlot      = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "CUR_RMOUTH_CORNER");
  local foreheadFactorSlot      = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "FOREHEAD_FACTOR");
  local decreeFactorSlot        = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "DECREE_FACTOR");
   
  local materialPath  = "docs:liveportrait/material/liveportrait.material";
  local renderMode    =  apolloengine.RenderComponent.RM_TRIANGLES;
  
  local projection = mathfunction.Matrix44(
    mathfunction.vector4( 2 / self.size:x(), 0, 0, 0),
    mathfunction.vector4(0, 2 / self.size:y(), 0, 0),
    mathfunction.vector4(0, 0, -2 / 2000, 0),
    mathfunction.vector4(-1, -1, 0, 1)
  );
  local modelview = mathfunction.Matrix44();
  modelview:SetIdentity()
  
  self.renderNode     =  apollonode.RenderNode();
  self.renderNode:CreateResource(materialPath, renderMode, self.vtxStream, self.idxStream);    
  self.renderNode:SetParameter(texSlot, self.tex);
  self.renderNode:SetParameter(projSlot, projection);
  self.renderNode:SetParameter(self.viewSlot, modelview);
  self.renderNode:SetParameter(mixedSlot, mathfunction.vector1(1));
  
  local originForeheadPoints  = mathfunction.vector3array();
  originForeheadPoints:PushBack(self.headCenter);
  originForeheadPoints:PushBack(self.foreheadCenter);
  self.renderNode:SetParameter(origForeheadSlot, originForeheadPoints);
  
  local lFacePoints  = mathfunction.vector3array();
  lFacePoints:PushBack(self.lfaceCenter);
  lFacePoints:PushBack(self.lmouthCorner);  
  self.renderNode:SetParameter(origLMouthCornerSlot, lFacePoints);
  
  local rFacePoints  = mathfunction.vector3array();
  rFacePoints:PushBack(self.rfaceCenter);
  rFacePoints:PushBack(self.rmouthCorner);  
  self.renderNode:SetParameter(origRMouthCornerSlot, rFacePoints);
  
  if self.enableWrinkle == true then
    self.renderNode:SetParameter(self.enableWrinkleSlot, mathfunction.vector1(1));
  else
    self.renderNode:SetParameter(self.enableWrinkleSlot, mathfunction.vector1(0));
  end
  
  self.renderNode:SetParameter(foreheadFactorSlot, mathfunction.vector1(self.foreheadFactor));
  self.renderNode:SetParameter(decreeFactorSlot, mathfunction.vector1(self.decreeFactor));
  
  self.renderNode:SetSequence(defined.SEQUENCE);
  self.renderNode:SetShow(false);
end

-- generate texture based on texture stream
function portraitRender:_GenerateTexture(ts)
  if (ts == nil)
  then
    return nil;
  end
  
  local texture = apolloengine.TextureEntity();
  texture:PushMetadata(  apolloengine.TextureBufferMetadata(
                          apolloengine.TextureEntity.TU_WRITE,
                          1, false,
                          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
                          apolloengine.TextureEntity.TF_LINEAR,
                          apolloengine.TextureEntity.TF_LINEAR,
                          ts));
                      
  texture:CreateResource();  
  return texture;
end

-- Initialize the vertex stream and indicies stream
function portraitRender:_InitStreams(triangles)  
  local vtxstream     = apolloengine.VertexStream();
  
  -- set vertex format
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_POSITION,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            3);
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            3); 
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            2); 
  vtxstream:SetVertexType(  apolloengine.ShaderEntity.ATTRIBUTE_COLOR0,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            apolloengine.VertexBufferEntity.DT_FLOAT,
                            1); 

  for i = 1, self.points:Size() do 
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_POSITION, self.points:Get(i));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE0, self.points:Get(i));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COORDNATE1, self.texCoords:Get(i));
    vtxstream:PushVertexData( apolloengine.ShaderEntity.ATTRIBUTE_COLOR0, mathfunction.vector1(self.enableFP[i]));
  end
  
  local idxStream = apolloengine.IndicesStream();
  
  idxStream:SetIndicesType(apolloengine.IndicesBufferEntity.IT_UINT16);
  idxStream:ReserveBuffer(#triangles);

  for i = 1, #triangles do
    idxStream:PushIndicesData(triangles[i]);
  end

  return vtxstream, idxStream;
end

return portraitRender;
