local Object        = require "classic"
local apolloengine  = require "apolloengine"
local apollonode    = require "apolloutility.apollonode"
local defined       = require "facemorph.defined"
local filter        = require "facemorph.warpfilter"
local RenderPassBase = require "facemorph.renderpass"
local mf            = require "mathfunction"
local masquerade    = require "masquerade"

local WarpPass = RenderPassBase:extend();

function WarpPass:new()  
  WarpPass.super.new(self);

  self.warpTexs = {} -- output from shader  
  self.seqCams = {}
  self.filters = {}
  self.morph = masquerade.FaceMorph()
end

function WarpPass:Clear() 
  WarpPass.super.Clear(self);

  self.warpTexs = {};
end

function WarpPass:ReleaseResource()
  self:Clear();
  
  self.morph = nil;
  for i = defined.SOURCE, defined.TARGET do
    if self.filters[i] ~= nil then 
      self.filters[i]:Clear();
      self.filters[i] = nil;
    end
  end
end

-- Setshow 
function WarpPass:SetShow(show)
  for i = defined.SOURCE, defined.TARGET do 
    if self.filters[i] ~= nil then
      self.filters[i]:SetShow(show);
    end
  end
end

function WarpPass:Update(alpha)
  self.morph:Update(alpha);
  local src2dstMat = self.morph:GetInterHomographyMat();
  self.filters[defined.SOURCE]:SetHomoGraphyMatrix(src2dstMat);
  local dst2srcmat = self.morph:GetInverseInterHomographyMat();
  self.filters[defined.TARGET]:SetHomoGraphyMatrix(dst2srcmat);
end

function WarpPass:GetWarppedLandmarks(index)
  if (index == defined.SOURCE) 
  then 
    return self.morph:GetWarppedSrcPoints();
  else
    return self.morph:GetWarppedDstPoints();
  end
end

-- Please be noted in current algo, the output should be the image size, not the fbo size
-- will improve the restriction 
function WarpPass:SetFboSize(size)  
end

function WarpPass:SetFboSize2(size)
  self.morph:SetSize(size:x(), size:y())
  
  for i = defined.SOURCE, defined.TARGET do  
    local renderSeq = defined.SEQUENCE + i;
    local camera = apollonode.CameraNode(size);    
    camera:Activate(); 
    camera:SetSequence(renderSeq);
    camera:SetClearColor(mf.Color(0.0, 0.0, 0.0, 0.0));
    self.rt, self.tex = self:_CreateRenderTarget(size);
    camera:AttachRenderTarget(self.rt);
    
    if (self.filters[i] == nil)
    then
      self.filters[i] = filter();
    end
    self.filters[i]:SetRenderSequence(renderSeq);
    self.filters[i]:SetSize(size);
    self.seqCams[i] = camera;
    self.warpTexs[i] = self.tex;
  end
end

function WarpPass:Reset()  
  self:SetFboSize2(self.texs[defined.SOURCE]:GetSize());
  self.morph:Reset(self.points[defined.SOURCE], self.points[defined.TARGET]); 
  
  for i = defined.SOURCE, defined.TARGET do 
    self.filters[i]:SetTexture(self.texs[i]);    
    self.filters[defined.SOURCE]:SetHomoGraphyMatrix(mf.Matrix33());
    self.filters[defined.TARGET]:SetHomoGraphyMatrix(mf.Matrix33());
  end 
  self:SetShow(true);
end

function WarpPass:ResetBuffer(index, tex, size, landmarks)
  WarpPass.super.ResetBuffer(self, index, tex, size, landmarks);
  
end

function WarpPass:GetWarpTexture(index)
  if index < defined.SOURCE or index > defined.TARGET 
  then 
    return nil;
  else
    return self.warpTexs[index];
  end
end

function WarpPass:GetTargetTexture(index)
  return self.warpTexs[index];
end

return WarpPass;
