require "venusdebug"
local Object = require "classic"

local EffectAgent = Object:extend()

local DISP_STATE_HIDE = 0
local DISP_STATE_SHOW = 1

local PLAY_STATE_STOP = 0
local PLAY_STATE_START = 1

local EVENT_INVISIBLE = "invisible"
local EVENT_PLAYING = "playing"
local EVENT_PAUSED = "paused"
local EVENT_FIRST_FRAME = "first_frame"
local EVENT_LAST_FRAME = "last_frame"
local EVENT_MIDDLE_FRAME = "middle_frame"

local FADING_IN = 0
local FADING_OUT = 1

local LOGIC_FPS = 20

local function getTriggerFromState(disp, play)
    if disp == DISP_STATE_HIDE and play == PLAY_STATE_STOP then
        return EVENT_INVISIBLE
    end
    if disp == DISP_STATE_SHOW and play == PLAY_STATE_STOP then
        return EVENT_PAUSED
    end
    if disp == DISP_STATE_SHOW and play == PLAY_STATE_START then
        return EVENT_PLAYING
    end
    --the only condition left is START+HIDE, which should become invisible
    return EVENT_INVISIBLE
end

function EffectAgent:new(effectObj, name)
    self.name = name
    self.effectObj = effectObj
    self.curState = EVENT_INVISIBLE
    self.triggerFlag = false

    self.prevDispState = DISP_STATE_HIDE
    self.prevPlayState = PLAY_STATE_STOP
    self.currDispState = DISP_STATE_HIDE
    self.currPlayState = PLAY_STATE_STOP

    self.stateLasting = nil
    self.stateFading = nil
    self.stateFadingMode = nil
    self.curFading = 0
    self.fadingFactor = 1.0

    self.pendingQueue = {}
end

function EffectAgent:GetEffectObj()
    return self.effectObj
end

function EffectAgent:GetState()
    return self.curState
end

function EffectAgent:SetShow(show)
    --local dispState = show and DISP_STATE_SHOW or DISP_STATE_HIDE
    --local trigger = getTriggerFromState(dispState, self.currPlayState)
    --self:Trigger(trigger, {})
    --LOG("SUNTYLOG: EffectAgent:SetShow " .. string.format("%s %s", self.name, tostring(show)))
    if self.currDispState == DISP_STATE_SHOW then
        self.effectObj:SetShow(show)
    end
end

function EffectAgent:Trigger(event, args)

    if event == nil or args == nil then
        return
    end

    if args.delay ~= nil and args.delay > 0 then
        --LOG("SUNTYLOG: EffectAgent:Trigger delay " .. tostring(args.delay))
        local pending = {
            delay = args.delay / LOGIC_FPS,
            event = event,
            args = args
        }
        args.delay = nil
        table.insert(self.pendingQueue, pending)
        return
    end

    --LOG("SUNTYLOG: EffectAgent:Trigger " .. string.format("%s %s", self.name, event))

    if event == EVENT_PLAYING then
        self:Show(args)
        self:Start(args)
    elseif event == EVENT_INVISIBLE then
        self:Stop(args)
        self:Hide(args)
    elseif event == EVENT_PAUSED then
        self:Stop(args)
        self:Show(args)
    elseif event == EVENT_FIRST_FRAME then
        self:SeekToFirst()
        self:Start(args)
        self:Stop(args)
        self:Show(args)
    elseif event == EVENT_LAST_FRAME then
        self:SeekToLast()
        self:Start(args)
        self:Stop(args)
        self:Show(args)
    elseif event == EVENT_MIDDLE_FRAME then
        self:SeekToMiddle()
        self:Start(args)
        self:Stop(args)
        self:Show(args)
    end

    local deducedEvent = getTriggerFromState(self.currDispState, self.currPlayState)

    if (event == EVENT_FIRST_FRAME or event == EVENT_LAST_FRAME or event == EVENT_MIDDLE_FRAME) and
            deducedEvent == EVENT_PAUSED then
        deducedEvent = event
    end

    if deducedEvent ~= self.curState then
        self.curState = deducedEvent
        self.triggerFlag = true
    end
end

function EffectAgent:TestTrigger()
    if self.triggerFlag then
        self.triggerFlag = false
        return self.curState
    end
    return nil
end


function EffectAgent:getAsset(name)
    if self.name == name then
        return self
    end
    return nil
end


--below is what the effectObj should also implement

function EffectAgent:SetRootNode(rootNode)
    self.effectObj:SetRootNode(rootNode)
end

function EffectAgent:SetFade(factor, args)
    self.effectObj:SetFade(factor, args)
end

function EffectAgent:Finished()
    return self.effectObj:Finished()
end

function EffectAgent:SeekToFirst()
    self.effectObj:SeekToFirst()
end

function EffectAgent:SeekToLast()
    self.effectObj:SeekToLast()
end

function EffectAgent:SeekToMiddle()
    self.effectObj:SeekToMiddle()
end

function EffectAgent:Show(args)
    if args.lasting ~= nil and args.lasting > 0 then
        self.stateLasting = args.lasting / LOGIC_FPS
    else
        self.stateLasting = nil
    end
    if args.fading ~= nil and args.fading > 0 then
        self.stateFading = args.fading / LOGIC_FPS
        self.stateFadingMode = FADING_IN
        args.initialFading = 0
        args.deltaFading = LOGIC_FPS / args.fading
        self:SetFade(0, args)
    else
        self.stateFading = nil
        self.stateFadingMode = nil
    end
    self.curFading = 0

    self.prevDispState = self.currDispState
    self.currDispState = DISP_STATE_SHOW
    self.effectObj:Show(args)
end


function EffectAgent:Hide(args)
    if args.lasting ~= nil and args.lasting > 0 then
        self.stateLasting = args.lasting / LOGIC_FPS
    else
        self.stateLasting = nil
    end
    if args.fading ~= nil and args.fading > 0 then
        self.stateFading = args.fading / LOGIC_FPS
        self.stateFadingMode = FADING_OUT
        args.initialFading = 1
        args.deltaFading = -LOGIC_FPS / args.fading
        self:SetFade(1, args)
    else
        self.stateFading = nil
        self.stateFadingMode = nil
    end
    self.curFading = 0

    if self.stateFading == nil or self.stateFading == 0 then
        self.prevDispState = self.currDispState
        self.currDispState = DISP_STATE_HIDE
        self.effectObj:Hide(args)
    end

end


function EffectAgent:Start(args)
    if args.lasting ~= nil and args.lasting > 0 then
        self.stateLasting = args.lasting
    else
        self.stateLasting = nil
    end

    self.prevPlayState = self.currPlayState
    self.currPlayState = PLAY_STATE_START
    self.effectObj:Start(args)
end


function EffectAgent:Stop(args)
    if args.lasting ~= nil and args.lasting > 0 then
        self.stateLasting = args.lasting
    else
        self.stateLasting = nil
    end

    self.prevPlayState = self.currPlayState
    self.currPlayState = PLAY_STATE_STOP
    self.effectObj:Stop(args)
end

function EffectAgent:Update(timeSpan, face, position, rotation, action)

    --LOG("SUNTYLOG: EffectAgent:Update enter")

    for pos, pending in ipairs(self.pendingQueue) do
        pending.delay = pending.delay - timeSpan
        if pending.delay <= 0 then
            self:Trigger(pending.event, pending.args)
            table.remove(self.pendingQueue, pos)
        end
    end

    --if self.currDispState == DISP_STATE_HIDE then
    --    return
    --end

    local factor = 1.0

    if self.stateFadingMode ~= nil then
        --LOG("SUNTYLOG: EffectAgent:Update fading " .. string.format("%d %d", self.curFading, self.stateFading))
        if self.curFading <= self.stateFading then
            if self.stateFadingMode == FADING_IN then
                factor = self.curFading / self.stateFading
            elseif self.stateFadingMode == FADING_OUT then
                factor = (self.stateFading - self.curFading) / self.stateFading
            end
            self.curFading = self.curFading + timeSpan
        else
            if self.stateFadingMode == FADING_OUT then
                --LOG("SUNTYLOG: EffectAgent:Update 1")
                self:Trigger(EVENT_INVISIBLE, {})
            end
            self.stateFadingMode = nil
            self.stateFading = nil
        end
    end

    if self.fadingFactor ~= factor then
        self:SetFade(factor)
        self.fadingFactor = factor
    end

    if self.currPlayState == PLAY_STATE_START then

        self.effectObj:Update(timeSpan, face, position, rotation, action)

        if self.effectObj:Finished() then
            --LOG("SUNTYLOG: EffectAgent:Update 2")
            self:Trigger(EVENT_INVISIBLE, {})
        end

    elseif self.currPlayState == PLAY_STATE_STOP then

        self.effectObj:Update(0, face, position, rotation, action)

        if self.stateLasting ~= nil then
            self.stateLasting = self.stateLasting - timeSpan
            if self.stateLasting <= 0 then
                self.stateLasting = nil
                local trigger = getTriggerFromState(self.prevDispState, self.prevPlayState)
                --LOG("SUNTYLOG: EffectAgent:Update 3 " .. trigger)
                self:Trigger(trigger, {})
            end
        end
    end
end

return EffectAgent
