local apollonode = require "apolloutility.apollonode"
local apolloengine = require "apolloengine"
local venusjson = require "venusjson"
local mathfunction = require "mathfunction"
local Object = require "classic"
local renderqueue = require "apolloutility.renderqueue"
local tongue = require "emoji.tongue"
local lowerteeth = require "emoji.lowerteeth"
local animation = require "emoji.animation"
local eye = require "emoji.eye"
local stationary = require "emoji.stationary"
local face = require "emoji.face"
local bsmodel = require "emoji.bsmodel"

local likeapp = require "likeapp"
local libmasquerade = require "libmasquerade"
local mlbvt = require "machinelearningservice"
local videodecet = require "videodecet"
local vc = require "venuscore"
local videodefined = require"videodecet.defined"


local landmarkNum = 106;
local venusemoji = Object:extend();

function venusemoji:new()
  self.models = {};
  self.show = false;
end


function venusemoji:ParseConfig(modelconfig)
  --load of face--
  local faceConfig = modelconfig.face;
  if faceConfig == nil then
    return true;
  end

  local zPos = modelconfig.zPosition or 0;
  zPos = zPos - 2000;
  
  local bindboxV = mathfunction.Aabbox3d( mathfunction.vector3(-1,-1, zPos-1),
                                          mathfunction.vector3(1,1, zPos));   

  self.renderbefore = modelconfig.renderbefore or false;
  
  if modelconfig.lightdirection then
    local dir = modelconfig.lightdirection;
    self.direct = apollonode.LightNode(apolloengine.LightComponent.LT_DIRECTIONAL);
    self.direct:SetLocalDirection(mathfunction.vector3(dir[1],dir[2],dir[3]));
    if modelconfig.lightcolor then
      local color = modelconfig.lightcolor;
      self.direct:SetColor(mathfunction.vector3(color[1],color[2],color[3]));
    else
      self.direct:SetColor(mathfunction.vector3(1,1,1));
    end    
  end
  if modelconfig.ambientcolor then  
    self.ambient = apollonode.LightNode(apolloengine.LightComponent.LT_AMBIENT);
    local color = modelconfig.ambientcolor;
    self.ambient:SetColor(mathfunction.vector3(color[1],color[2],color[3]));
  end
  if modelconfig.skybox and 6 == #modelconfig.skybox then  
    self.skybox = apolloengine.TextureEntity();      
    for i, p in ipairs(modelconfig.skybox) do        
      --self.skybox:PushMetadata(apolloengine.TextureFileMetadata(p));
      self.skybox:PushMetadata(apolloengine.TextureFileMetadata(
      apolloengine.TextureEntity.TT_TEXTURECUBE_FRONT + i - 1,
      apolloengine.TextureEntity.TU_STATIC,
      apolloengine.TextureEntity.PF_AUTO,
      1, false,
      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,--默认repeat
      apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
      apolloengine.TextureEntity.TF_LINEAR,
      apolloengine.TextureEntity.TF_LINEAR,p));
    end

    self.skybox:CreateResource();
    apolloengine.IMaterialSystem:SetGlobalParameter(apolloengine.ShaderEntity.SKY_BOX, self.skybox);
    apolloengine.ShaderEntity.EVN_SCALE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"ENV_SCALE");
    local env_scale = modelconfig.env_scale or 1;
    apolloengine.IMaterialSystem:SetGlobalParameter(apolloengine.ShaderEntity.EVN_SCALE,mathfunction.vector1(env_scale));
  end
  
  -- face
  if modelconfig["face"] ~= nil then
    self.face_model = face();
    
    local face_config = modelconfig.face;
    self:SetModel(self.face_model, face_config, nil, bindboxV, self.renderbefore);
        
    -- pos related parameters
    self.pos_adjust = mathfunction.vector3(face_config.pos[1],face_config.pos[2],face_config.pos[3]);    
  end 
  
  --eyelash
  if modelconfig["eyelash"] ~= nil then
    local eyelash = bsmodel();
    self:SetModel(eyelash, modelconfig.eyelash, self.face_model, bindboxV, self.renderbefore);
  end 
    
  --extra blendshape model
  if modelconfig["extra"] ~= nil then
    local extra = bsmodel();
    self:SetModel(extra, modelconfig.extra, self.face_model, bindboxV, self.renderbefore);
  end 

  if modelconfig["eyes"] ~= nil then
    self.eyes = {};
    local eyesConfig = modelconfig.eyes;
    --left eye
    if eyesConfig["left"] ~= nil then
      local lefteye = eye();
      self:SetModel(lefteye, eyesConfig.left, self.face_model, bindboxV, self.renderbefore);
      table.insert(self.eyes, lefteye);  
    end
    
    --right eye
    if eyesConfig["right"] ~= nil then
      local righteye = eye();
      self:SetModel(righteye, eyesConfig.right, self.face_model, bindboxV, self.renderbefore);
      table.insert(self.eyes, righteye);  
    end
  end
  
	--lower_teeth
   if modelconfig["lower_teeth"] ~= nil then
    local lower_teeth = lowerteeth();
    self:SetModel(lower_teeth, modelconfig.lower_teeth, self.face_model, bindboxV, self.renderbefore);
  end 
  
  --tongue
  if modelconfig["tongue"] ~= nil then
    local tongue = tongue();
    self:SetModel(tongue, modelconfig.tongue, self.face_model, bindboxV, self.renderbefore);
  end 
  
	--stationary
  if modelconfig["stationary"] ~= nil then
    local stationaryConfig = modelconfig.stationary;
    for meshKey, meshitems in pairs(stationaryConfig) do
      local stationary = stationary();
      self:SetModel(stationary, meshitems, self.face_model, bindboxV, self.renderbefore);
    end
   end
   
	--animation
	if modelconfig["animation"] ~= nil then
    local animationConfig = modelconfig.animation;
    for meshKey, meshitems in pairs(animationConfig) do
      local animation = animation();
      self:SetModel(animation, meshitems, self.face_model, bindboxV, self.renderbefore);
    end
	end

  self.root_node = apollonode.TransNode();
  self.root_node:AttachNode(self.face_model:GetModel());
  
  if self:_InitMasquerade() == false then
    LOG("Failed to init masquerade.");
    self.masqueradeTracking = nil;
    return false;
  end    
  self.bvtInited = false;
  
  return true;
end


function venusemoji:ShowModel(show)
  self.show = show;
  for i = 1, #self.models do
    self.models[i]:SetShow(show);
  end
end

function venusemoji:SetModel(model, config, attach_to_node, bindbox, render_before)
  if (model ~= nil ) then 
    model:CreateModel(config, attach_to_node, bindbox, render_before);  
    table.insert(self.models, model);  
  end
end

function venusemoji:SetPosition(est)
  if(est ~= nil) then
    self.root_node:SetLocalRotation(est.rotation);
    self.root_node:SetLocalPosition(est.position + self.pos_adjust);
  end
end

function venusemoji:UpdateData(input_data)  
  if (input_data == nil) then
    return false;
  end
  
  local weights_dict = input_data.weights_dict;  
  if weights_dict == nil or next(weights_dict) == nil then
    return false;
  end
  
  -- update the weights, from up to down
  self:AdjustWeight(self.face_model, weights_dict);
  return true;
end

function venusemoji:AdjustWeight(model, weights_dict)
  if(model ~= nil) then
    model:AdjustWeight(weights_dict);
    
    local attached_models = model:GetAttachedModels();    
    if (not attached_models or not next(attached_models)) then
      return;
    end
    
    for i = 1, #attached_models do
      local sub_model = attached_models[i];
      self:AdjustWeight(sub_model, weights_dict);
    end
  end
end

function venusemoji:UpdateModel(input_data, def)
  for i = 1, #self.models do
    local model = self.models[i];
    if model ~= nil then
      model:UpdateModel(input_data, def);
    end
  end
end

-- this is a temporary function which will be replaced after the likeapp implment the interface
function venusemoji:ReceiveInputDatas()
  local size = videodecet:GetVideoSize();
  local ts = videodecet:GetVideoFrame();
  local faces = videodecet:GetFaces();
  if #faces > 0 and size and ts then    
    if not self.bvtInited then
      if self:_InitBvt(size[1], size[2]) then
        self.bvtInited = true;
      else
        self.cnnIris = nil;
        self.cnnTongue = nil;
        return false;
      end
    end
    
    local landmarks = {};
    local keyPoints = mathfunction.vector2array();
    local faceKeyPoints = videodecet:GetPixelFacekeypointArray();
    for i=1, landmarkNum do
      keyPoints:PushBack(faceKeyPoints:Get(i));
      landmarks[2*i-1] = faceKeyPoints:Get(i):x();
      landmarks[2*i] = faceKeyPoints:Get(i):y();
    end

    local eyes = self.cnnIris:Run(keyPoints,ts);
    local leftEye = mathfunction.vector2(eyes:Get(20):x(),eyes:Get(20):y());    
    local rightEye = mathfunction.vector2(eyes:Get(40):x(),eyes:Get(40):y());
    local bvtTongue = self.cnnTongue:Run(keyPoints,ts):Get(1):x();
    if leftEye:x() == 0 and leftEye:y() == 0 then
      leftEye = keyPoints:Get(105);
    end
    if rightEye:x() == 0 and rightEye:y() == 0 then
      rightEye = keyPoints:Get(106);
    end

    local pry = faces[1]:GetRotation();
    local euler = mathfunction.vector3(pry[1],pry[3],pry[2]);
    local visibility = likeapp.AI:GetFaceVisibility();
    if not self.masqueradeTracking:Track(ts, euler, landmarks, visibility, leftEye, rightEye, bvtTongue) then
        return nil
    end
    local bsNum = self.masqueradeTracking:GetBlendshapeNumber();
    local masqueradeResult = {};
    masqueradeResult.weights_dict = {};
    for i=1, bsNum do
      local name = self.masqueradeTracking:GetBlendshapeName(i-1);
      local weight = self.masqueradeTracking:GetBlendshapeWeight(i-1);
      masqueradeResult.weights_dict[name] = weight;
    end
    masqueradeResult.lookat = self.masqueradeTracking:GetLookat();
    masqueradeResult.tongue = self.masqueradeTracking:GetTongue();
    return masqueradeResult;
  end

  return nil;
end


function venusemoji:_InitBvt(w, h)
  self.cnnIris = mlbvt.MachinelearningService();
  self.cnnTongue = mlbvt.MachinelearningService();
  
  if _PLATFORM_WINDOWS then  --只有windows需要设置模型路径
    self.cnnIris:SetModelAndParamsWin(
      6,
      vc.IFileSystem:PathAssembly(videodefined.bvt_model_path));
    self.cnnTongue:SetModelAndParamsWin(
      10,
      vc.IFileSystem:PathAssembly(videodefined.bvt_model_path));
  end
  self.cnnIris:SetType(6);
  local res = self.cnnIris:Init(w,h);
  if res ~= 0 then
    LOG("bvt iris init failed: "..res);
    return false;
  end
  self.cnnTongue:SetType(10);
  res = self.cnnTongue:Init(w,h);
  if res ~= 0 then
    LOG("bvt tongue init failed: "..res);
    return false;
  end
  return true;
end

function venusemoji:_InitMasquerade()
  self.masqueradeTracking = libmasquerade.FaceTracking();
  if self.masqueradeTracking:Init(
    "config_face_morphable_3.json",
    "3dmm_60.bin",
    "config_face_template_4.json",
    "blendshape_21_4.bin",
    "user_face_recon.json",
    "config.json") then
    return true;
  end
  return false;
end

return venusemoji;
