local venuscore = require "venuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"


local PluginSystem = require "window.editor.system.pluginsystem"

local StickerObjectBase = venuscore.Object:extend("StickerObjectBase");

function StickerObjectBase:new(root)
  self._root = root
  self.propsPrev = {}
  self.propsCurr = {}
  self.propsNext = {}
end

function StickerObjectBase:setPropNext(prop, value)
  self.propsNext[prop] = value
end

function StickerObjectBase:setPropCurr(prop, value)
  self.propsCurr[prop] = value
end

function StickerObjectBase:getPropCurr(prop)
  return self.propsCurr[prop]
end

function StickerObjectBase:getPropPrev(prop)
  return self.propsPrev[prop]
end

function StickerObjectBase:Subscribe(subscribe)
  subscribe:AddSubscriber(self)
end

function StickerObjectBase:GetProperties()
  return {}
end

function StickerObjectBase:Update(delta)
end

-------------------

local StickerLogger = StickerObjectBase:extend("StickerLogger");

function StickerLogger:new(root)
  StickerLogger.super.new(self, root)
end

function StickerLogger:setPropCurr(prop, value)
  StickerLogger.super.setPropCurr(self, prop, value)
  if prop == "log" then
    LOG(string.format(
      "SUNTYLOG: StickerLogger: prop: %s, value: %s", 
      tostring(prop), 
      tostring(value)
    ))
  end
end

function StickerLogger:GetProperties()
  return { "log" }
end

-------------------

local StickerTimer = StickerObjectBase:extend("StickerTimer");

function StickerTimer:new(root)
  StickerTimer.super.new(self, root)
  self.interval = 1.0
  self.timer = 0.0
end

function StickerTimer:Update(delta)
  self.timer = self.timer + delta
  if self.timer >= self.interval then
    self.timer = self.timer - self.interval
    self:setPropCurr("signal", true)
  else
    self:setPropCurr("signal", false)
  end
end

function StickerTimer:GetProperties()
  return { "signal" }
end

-------------------

--TODO: Subscribe gesture signal source
local StickerGestureDetect = StickerObjectBase:extend("StickerGestureDetect");

function StickerGestureDetect:new()
  StickerGestureDetect.super.new(self)
end

function StickerGestureDetect:Update(delta)
  
end
-------------------

local StickerSignalSourceBase = venuscore.Object:extend("StickerSignalSourceBase");

function StickerSignalSourceBase:new(root)
  self._root = root
  self.subscribers = {}
end

function StickerSignalSourceBase:AddSubscriber(subscriber)
  table.insert(self.subscribers, subscriber)
end

--TODO:
--Dispatch should be overriden by face/gesture sources
--in order to choose the 'right face' to receive the event
function StickerSignalSourceBase:Dispatch(signal)
  local subscribers = self.subscribers
  local count = #subscribers
  for i = 1, count do
    subscribers[i]:setPropCurr("signal", signal)
  end
end

function StickerSignalSourceBase:Update(delta)
end

-------------------

local StickerGestureSource = StickerSignalSourceBase:extend("StickerGestureSource")

function StickerGestureSource:new(root)
  StickerGestureSource.super.new(self, root)
  self.component = root.desc.Node:GetComponent(apolloengine.Node.CT_CV_CLASSIFY)
  if self.component then
    --TODO: subscribe classify component result
  end
end


function StickerGestureSource:Update(delta)

end

-------------------
local StickerBehaviorInst = venuscore.Object:extend("StickerBehaviorInst");

StickerBehaviorInst.StickerObjectBase = StickerObjectBase;
StickerBehaviorInst.StickerTimer = StickerTimer;
StickerBehaviorInst.StickerLogger = StickerLogger;

function StickerBehaviorInst:new(behaviorDesc)

  self.desc = behaviorDesc;

  self.groups = {}
  
  local desc = self.desc;

  for i = 1, #desc.groups do
    local groupDesc = desc.groups[i]
    local groupInst = {
      id = groupDesc.id,
      objects = {},
    }
    for j = 1, #groupDesc.objects do
      local objectId = groupDesc.objects[j].value;
      local objectDesc = self:_FindObjectDesc(desc.objects, objectId)
      local objectInst = objectDesc.type(self)
      groupInst.objects[objectId] = objectInst;
    end

    self.groups[groupDesc.id] = groupInst;
  end

end

function StickerBehaviorInst:_FindObjectDesc(objectsDesc, id)
  local desc = self.desc;
  for i = 1, #objectsDesc do
    if objectsDesc[i].id == id then
      return objectsDesc[i]
    end
  end
  return nil
end

function StickerBehaviorInst:Update(delta)
  --TODO: Update each signal source

  local behaviorDesc = self.desc;
  local transitionsDesc = behaviorDesc.transitions;

  for _, groupInst in pairs(self.groups) do
    for _, objectInst in pairs(groupInst.objects) do
      objectInst:Update(delta)
    end
  end

  for groupId, groupInst in pairs(self.groups) do
    for i = 1, #transitionsDesc do
      local transitionDesc = transitionsDesc[i];
      local triggerDesc = transitionDesc.trigger;
      local actionDesc = transitionDesc.action;

      if self:_Triggered(groupInst, triggerDesc) then
        LOG(string.format("SUNTYLOG: transition %s triggered for group %s", transitionDesc.name, groupId))
        self:_TakeAction(groupInst, actionDesc);
      end
    end

    for objectId, objectInst in pairs(groupInst.objects) do
      objectInst.propsPrev = { unpack(objectInst.propsCurr) }
      for prop, val in pairs(objectInst.propsNext) do
        objectInst:setPropCurr(prop, val)
      end
      objectInst.propsNext = {}
    end
  end
end

function StickerBehaviorInst:_Triggered(groupInst, stateDesc)
  local category = stateDesc.category
  if category == "all" then
    local list = stateDesc.list
    local count = #list
    for i = 1, count do
      local subStateDesc = list[i]
      if not self:_Triggered(groupInst, subStateDesc) then
        return false
      end
    end
    return true
  elseif category == "any" then
    local list = stateDesc.list
    local count = #list
    for i = 1, count do
      local subStateDesc = list[i]
      if self:_Triggered(groupInst, subStateDesc) then
        return true
      end
    end
    return false
  elseif category == "object" then
    local objectInst = groupInst.objects[stateDesc.id]
    local prop = stateDesc.prop
    local result = true
    if stateDesc.prev ~= nil then
      result = result and (objectInst:getPropPrev(prop) == stateDesc.prev)
    end
    if stateDesc.curr ~= nil then
      result = result and (objectInst:getPropCurr(prop) == stateDesc.curr)
    end
    return result
  end
  return false
end

function StickerBehaviorInst:_TakeAction(groupInst, stateDesc)
  local category = stateDesc.category
  if category == "all" then
    local list = stateDesc.list
    local count = #list
    for i = 1, count do
      local subStateDesc = list[i]
      self:_TakeAction(groupInst, subStateDesc)
    end
  elseif category == "random" then
    local list = stateDesc.list
    local count = #list
    if count > 0 then
      --TODO
      local rand = 1
      local subStateDesc = list[rand]
      self:_TakeAction(groupInst, subStateDesc)
    end
  elseif category == "object" then
    local prop = stateDesc.prop
    local nextValue = stateDesc.next
    if prop == nil or nextValue == nil then
      return
    end
    local targetGroupType = stateDesc.targetGroup
    local targetGroups = {}
    if targetGroupType == "local" then
      table.insert(targetGroups, groupInst)
    elseif targetGroupType == "all" then
      targetGroups = self.groups
    end
    for i = 1, #targetGroups do
      local targetGroup = targetGroups[i]
      local objectInst = targetGroup.objects[stateDesc.id]
      if objectInst ~= nil then
        objectInst:setPropNext(prop, nextValue)
      end
    end
  end
end

return StickerBehaviorInst;