local venuscore = require "venuscore"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local cv = require "computervisionfunction"
local facelift = require "beauty.facelift_live"
local facelift_defined = require "beauty.defined_live"
local venusjson = require "venusjson"

local FaceliftBehavior = venuscore.VenusBehavior:extend("FaceliftBehavior")

-- all facial effects library
local libs = {
  "chinscale","eyeangle","eyebrow","eyedistance","eyescale",
  "faceshin","hairline","innercanthus","lipscale","mouthscale",
  "noselength","noseshin","outercanthus","philtrumscale", "roundeye"
}

local lib_map = {
  [-3]={"roundeye"},
  [-2]={"lipscale"},
  [-1]={"eyescale"},
  [0]={"mouthscale"},
  [1]={"faceshin"},
  [17]={"noseshin"},
  [19]={"chinscale"},
  [24]={"hairline"},
  [29]={"philtrumscale"},
  [31]={"noselength"},
  [37]={"innercanthus"},
  [39]={"eyebrow"},
  [41]={"outercanthus"},
  [45]={"eyeangle","eyedistance"}
}

local lib_range = {
  ["roundeye"]={0,1},
  ["chinscale"]={-1,1},
  ["eyeangle"]={-1,1},
  ["eyebrow"]={-1,1},
  ["eyedistance"]={-1,1},
  ["eyescale"]={0,1},
  ["faceshin"]={0,1},
  ["hairline"]={0,1},
  ["innercanthus"]={0,1},
  ["lipscale"]={0,1},
  ["mouthscale"]={0,1},
  ["noselength"]={-1,1},
  ["noseshin"]={0,1},
  ["outercanthus"]={0,1},
  ["philtrumscale"]={-1,1}
}

-- set starting state of switches in venus editor
local editor_lib_effects_on = false;

local min_key_index = -3

local KEYS = {
  Face_0_32 = facelift_defined.face_start,
  Face_1_31 = facelift_defined.face_start + 1,
  Face_2_30 = facelift_defined.face_start + 2,
  Face_3_29 = facelift_defined.face_start + 3,
  Face_5_27 = facelift_defined.face_start + 4,
  Face_7_25 = facelift_defined.face_start + 5,
  Face_10_22 = facelift_defined.face_start + 6,
  Face_12_20 = facelift_defined.face_start + 7,
  Face_14_18 = facelift_defined.face_start + 8,
  Face_84_90 = facelift_defined.face_start + 9,
  Face_87_93 = facelift_defined.face_start + 10,
  Face_82_83 = facelift_defined.face_start + 11,
  Face_49_49 = facelift_defined.face_start + 12,
  Face_16_16 = facelift_defined.face_start + 13,
  Face_add_1 = facelift_defined.face_start + 14,
  Face_add_2 = facelift_defined.face_start + 15,

  Nose_0 = facelift_defined.nose_start,
  Nose_1 = facelift_defined.nose_start + 1,

  Chin_7_25 = facelift_defined.chin_start,
  Chin_10_22 = facelift_defined.chin_start + 1,
  Chin_12_20 = facelift_defined.chin_start + 2,
  Chin_14_18 = facelift_defined.chin_start + 3,
  Chin_16_16 = facelift_defined.chin_start + 4,
  HairLine_0 = facelift_defined.HairLine_start,
  HairLine_1 = facelift_defined.HairLine_start + 1,
  HairLine_2 = facelift_defined.HairLine_start + 2,
  HairLine_3 = facelift_defined.HairLine_start + 3,
  HairLine_4 = facelift_defined.HairLine_start + 4,
  Philtrum_0 = facelift_defined.Philtrum_start,
  Philtrum_1 = facelift_defined.Philtrum_start + 1,

  -- Noselength
  Noselength_80_81 = facelift_defined.Noselength_start,
  Noselength_82_83 = facelift_defined.Noselength_start + 1,
  Noselength_48_50 =  facelift_defined.Noselength_start + 2,
  Noselength_49 = facelift_defined.Noselength_start + 3,
  Noselength_46 =  facelift_defined.Noselength_start + 4,
  Noselength_45 =  facelift_defined.Noselength_start + 5,

  -- Innercanthus
  InnerCanthus_78_79 = facelift_defined.InnerCanthus_start,
  InnerCanthus_DistAlpha = facelift_defined.InnerCanthus_start + 1,
  
  Eyebrow_34_41 = facelift_defined.Eyebrow_start,
  Eyebrow_35_40 = facelift_defined.Eyebrow_start + 1,

  -- Outercanthus
  OuterCanthus_52_61 = facelift_defined.OuterCanthus_start,
  OuterCanthus_57_62 = facelift_defined.OuterCanthus_start + 1,
  OuterCanthus_extra = facelift_defined.OuterCanthus_start + 2,
  OuterCanthus_DistAlpha = facelift_defined.OuterCanthus_start + 3,
  
  -- eye angle and distance 
  EyeAngDis = facelift_defined.EyeAngDis_start, 
  
  Roundeye = -3,
  Lip = -2,
  Eye = -1,
  Mouth = 0
}

local KEYSINORDER = {}
local ATTRIBUTES_EYEAD = { "Ext1", "Ext2", "Dis", "Ang" }
local ATTRIBUTES = { "X", "Y", "R" }
local ATTRIBUTES_MOUTH = {"Inten", "YScale", "RR", "rr"}
local ATTRIBUTES_LIP = { "X", "YUP", "YDOWN", "RUP", "RDOWN" }
local ATTRIBUTES_ROUNDEYE = { "R", "S" }
local face_total_num = 16;

local function GetKeyString(key, attribute)
  return key.."_"..attribute
end

function FaceliftBehavior:readParams()
  LOG("FaceliftBehavior:readParams start");
  local AllParam = facelift:GetParams();
  self["Fall_Off"] = AllParam.FallOff;
  self["FaceFallOff_1"] = AllParam.FaceFallOff[1];
  self["FaceFallOff_2"] = AllParam.FaceFallOff[2];
  self["OutercanthusFallOff"] = AllParam.OutercanthusFallOff
  self["EyescaleFallOff"] = AllParam.EyescaleFallOff
  for key, index in pairs(KEYS) do
    LOG("index: " .. index .. ", key: " .. key);
    if index == -3 then
      self["Roundeye_R"] = AllParam.roundeyeParams[1]
      self["Roundeye_S"] = AllParam.roundeyeParams[2]
    elseif index == -2 then
      self["Lip_X"] = AllParam.lipparams[1];
      self["Lip_YUP"] = AllParam.lipparams[2];
      self["Lip_YDOWN"] = AllParam.lipparams[3];
      self["Lip_RUP"] = AllParam.lipparams[4];
      self["Lip_RDOWN"] = AllParam.lipparams[5];
    elseif index == -1 then
      self["Eye_Inten"] = AllParam.eyeparams[3];
      self["Eye_YScale"] = AllParam.eyeparams[4];
      self["Eye_RR"] = AllParam.eyeparams[1];
      self["Eye_rr"] = AllParam.eyeparams[2];
    elseif index == 0 then
      self["Mouth_Inten"] = AllParam.mouthparams[3];
      self["Mouth_YScale"] = AllParam.mouthparams[4];
      self["Mouth_RR"] = AllParam.mouthparams[1];
      self["Mouth_rr"] = AllParam.mouthparams[2];
    elseif index == facelift_defined.InnerCanthus_start + 1 then
      self[key] = AllParam.innercanthusDistAlpha;
    elseif index == facelift_defined.OuterCanthus_start + 3 then
      self[key] = AllParam.outercanthusDistAlpha;
    elseif index == facelift_defined.EyeAngDis_start then
      self["EyeAngDis_Ext1"] = AllParam.eyeadparams[1];
	    self["EyeAngDis_Ext2"] = AllParam.eyeadparams[2];
      self["EyeAngDis_Dis"] = AllParam.eyeadparams[3];
      self["EyeAngDis_Ang"] = AllParam.eyeadparams[4];
    elseif index < facelift_defined.nose_start then
      local var_key = key.."_X"
      --LOG("new Init: " .. index .. " " .. var_key .. " " .. AllParam.FaceParam[key_param_table[index]][1]);
      self[var_key] = AllParam.FaceParam[index][1];
      var_key = key.."_Y"
      --LOG("new Init: " .. index .. " " .. var_key .. " " .. AllParam.FaceParam[key_param_table[index]][2]);
      self[var_key] = AllParam.FaceParam[index][2];
      var_key = key.."_R"
      --LOG("new Init: " .. index .. " " .. var_key .. " " .. AllParam.FaceParamRadius[key_param_table[index]][1]);
      self[var_key] = AllParam.FaceParamRadius[index][1];
    elseif index < facelift_defined.chin_start then
      local var_key = key.."_X"
      self[var_key] = AllParam.noseparams[index - facelift_defined.nose_start + 1][1];
      var_key = key.."_Y"
      self[var_key] = 0;
      var_key = key.."_R"
      self[var_key] = AllParam.noseparams[index - facelift_defined.nose_start + 1][2];
    elseif index < facelift_defined.HairLine_start then
      local var_key = key.."_X"
      self[var_key] = AllParam.ChinParam[index - facelift_defined.chin_start + 1][1];
      var_key = key.."_Y"
      self[var_key] = AllParam.ChinParam[index - facelift_defined.chin_start + 1][2];
      var_key = key.."_R"
      self[var_key] = AllParam.ChinParamRadius[index - facelift_defined.chin_start + 1][1];
    elseif index < facelift_defined.Philtrum_start then
      --forehead
      local var_key = key.."_X";
      self[var_key] = AllParam.hairline_param[index - facelift_defined.HairLine_start + 1][1];
      LOG("VAR KEY " .. var_key .. " : " .. self[var_key]);
      var_key = key.."_Y";
      self[var_key] = AllParam.hairline_param[index - facelift_defined.HairLine_start + 1][2];
      var_key = key.."_R";
      self[var_key] = AllParam.hairline_param_radius[index - facelift_defined.HairLine_start + 1][1];
    elseif index < facelift_defined.Noselength_start then
      local var_key = key.."_X"
      self[var_key] = 0
      var_key = key.."_Y"
      self[var_key] = AllParam.philtrumparams[index - facelift_defined.Philtrum_start + 1][1];
      var_key = key.."_R"
      self[var_key] = AllParam.philtrumparams[index - facelift_defined.Philtrum_start + 1][2];
    -- noselength: index 29 - 34
    elseif index < facelift_defined.InnerCanthus_start then
      local var_key = key.. "_X"
      self[var_key] = AllParam.noselengthParam[index - facelift_defined.Noselength_start + 1][1];
      var_key = key.. "_Y"
      self[var_key] = AllParam.noselengthParam[index - facelift_defined.Noselength_start + 1][2];
      var_key = key.. "_R"
      self[var_key] = AllParam.noselengthParamRadius[index - facelift_defined.Noselength_start + 1][1];
    -- innercanthus: index 35
    elseif index < facelift_defined.Eyebrow_start - 1 then
      local var_key = key.."_X"
      self[var_key] = AllParam.innercanthusParam[index - facelift_defined.InnerCanthus_start + 1][1];
      var_key = key.."_Y"
      self[var_key] = AllParam.innercanthusParam[index - facelift_defined.InnerCanthus_start + 1][2];
      var_key = key.."_R"
      self[var_key] = AllParam.innercanthusParamRadius[index - facelift_defined.InnerCanthus_start + 1][1];
    elseif index < facelift_defined.OuterCanthus_start then
      local var_key = key.."_X"
      self[var_key] = AllParam.EyebrowParam[index - facelift_defined.Eyebrow_start + 1][1];
      var_key = key.."_Y"
      self[var_key] = AllParam.EyebrowParam[index - facelift_defined.Eyebrow_start + 1][2];
      var_key = key.."_R"
      self[var_key] = AllParam.EyebrowParamRadius[index - facelift_defined.Eyebrow_start + 1][1];
    -- outercanthus: index 38 - 40
    elseif index < facelift_defined.EyeAngDis_start then
      local var_key = key.."_X"
      self[var_key] = AllParam.outercanthusParam[index - facelift_defined.OuterCanthus_start + 1][1];
      var_key = key.."_Y"
      self[var_key] = AllParam.outercanthusParam[index - facelift_defined.OuterCanthus_start + 1][2];
      var_key = key.."_R"
      self[var_key] = AllParam.outercanthusParamRadius[index - facelift_defined.OuterCanthus_start + 1][1];
    end
  end
  LOG("FaceliftBehavior:readParams end");
end

function FaceliftBehavior:initLibraryControlUICoef(on)
  
  for i=1,#libs do
    self[libs[i]] = on;
    self[libs[i].."_coef"] = 1;
  end
end

function FaceliftBehavior:new()
  -- LOG("behavior new used!");
  self:readParams();
  self.inited = false;
  self.EnableFaceLift = 1;
  self.Image = "";
  self.Video_Image = 1;
end

function FaceliftBehavior:_OnAwake()
  LOG("behavior onawake used!");
  self:readParams();
  self:initLibraryControlUICoef(editor_lib_effects_on);
  self.bg_node = self.Node:GetNodeByName("bg")
  self.bg_render = self.bg_node:GetComponent(apolloengine.Node.CT_RENDER)
  self:SwitchVideo()
end

local apollonode = require "apolloutility.apollonode"
function FaceliftBehavior:_OnUpdate(def)
  -- LOG("FaceliftBehavior:_OnUpdate");
  if self.Camera == nil or self.Recognition == nil then
    return
  end
  if not self.inited then
    self:Init()
    self.inited = true
  end
  
  if self.Video_Image == 1 then
    local lmks, pry = self:GetLandmarks()
    local resolution = self.Camera.CameraResolution
    local rt = self.Camera:GetAttachedRenderTarget()
    
    local entity= apolloengine.TextureEntity();
    entity:PushMetadata(apolloengine.TextureReferenceMetadata("DEVICE_CAPTURE"));
    entity:CreateResource();
    -- if(lmks ~= nil) then
    --     LOG("lmks size: " .. #lmks);
    -- end
    -- if(pry ~= nil) then
    --   LOG("pry size: " .. #pry);
    -- end
    -- LOG("resolution x: " .. resolution:x());
    -- LOG("resolution y: " .. resolution:y());
    
    facelift:EditorUpdate(lmks, pry, entity, resolution)
  else
    facelift:EditorImageUpdate();
  end
  -- LOG("FaceliftBehavior:_OnUpdate end");
end

function FaceliftBehavior:_OnDestroy()
end

function FaceliftBehavior:Init()
  LOG("behavior init used!");
  local resolution = self.Camera.CameraResolution;
  local aspect_ratio = resolution:y() / resolution:x();
  local face = facelift:EditorInitialize(self.Camera, aspect_ratio)
  for i=1,#libs do
    facelift:_Switch(libs[i],editor_lib_effects_on);
  end
  self.Node:AttachNode(face.node)
end

function FaceliftBehavior:ChangeValue(key_index, attribute, value)
  facelift:UpdateParaValue(key_index, attribute, value);
end

function FaceliftBehavior:GetLandmarks()
  local results = self.Recognition:GetResult();
  if results==nil or results[cv.RecognitionComponent.cvFace]==nil or results[cv.RecognitionComponent.cvForeheadDetection] == nil then
      LOG("NO LANDMARK RESULT");
      return nil;
  end

  local facerets = results[cv.RecognitionComponent.cvFace][1];
  local forehead_rets = results[cv.RecognitionComponent.cvForeheadDetection][1];
  --LOG("Num cv forehead: " .. #results[cv.RecognitionComponent.cvForeheadDetection][1])
  if facerets==nil then
      return nil;
  end

  local advFacerets = results[cv.RecognitionComponent.cvAdvancedLandmark];
  if advFacerets ~= nil and advFacerets[1] ~= nil and advFacerets[1][107 * 2 - 1] >= 0 then
    facerets[85 * 2 - 1] = advFacerets[1][(106 + 71) * 2 - 1];
    facerets[85 * 2] = advFacerets[1][(106 + 71) * 2];
    facerets[86 * 2 - 1] = advFacerets[1][(106 + 74) * 2 - 1];
    facerets[86 * 2] = advFacerets[1][(106 + 74) * 2];
    facerets[87 * 2 - 1] = advFacerets[1][(106 + 77) * 2 - 1];
    facerets[87 * 2] = advFacerets[1][(106 + 77) * 2];
    facerets[88 * 2 - 1] = advFacerets[1][(106 + 79) * 2 - 1];
    facerets[88 * 2] = advFacerets[1][(106 + 79) * 2];
    facerets[89 * 2 - 1] = advFacerets[1][(106 + 81) * 2 - 1];
    facerets[89 * 2] = advFacerets[1][(106 + 81) * 2];
    facerets[90 * 2 - 1] = advFacerets[1][(106 + 84) * 2 - 1];
    facerets[90 * 2] = advFacerets[1][(106 + 84) * 2];
    facerets[91 * 2 - 1] = advFacerets[1][(106 + 87) * 2 - 1];
    facerets[91 * 2] = advFacerets[1][(106 + 87) * 2];
    facerets[92 * 2 - 1] = advFacerets[1][(106 + 131) * 2 - 1];
    facerets[92 * 2] = advFacerets[1][(106 + 131) * 2];
    facerets[93 * 2 - 1] = advFacerets[1][(106 + 129) * 2 - 1];
    facerets[93 * 2] = advFacerets[1][(106 + 129) * 2];
    facerets[94 * 2 - 1] = advFacerets[1][(106 + 127) * 2 - 1];
    facerets[94 * 2] = advFacerets[1][(106 + 127) * 2];
    facerets[95 * 2 - 1] = advFacerets[1][(106 + 125) * 2 - 1];
    facerets[95 * 2] = advFacerets[1][(106 + 125) * 2];
    facerets[96 * 2 - 1] = advFacerets[1][(106 + 123) * 2 - 1];
    facerets[96 * 2] = advFacerets[1][(106 + 123) * 2];
    facerets[97 * 2 - 1] = advFacerets[1][(106 + 88) * 2 - 1];
    facerets[97 * 2] = advFacerets[1][(106 + 88) * 2];
    facerets[98 * 2 - 1] = advFacerets[1][(106 + 93) * 2 - 1];
    facerets[98 * 2] = advFacerets[1][(106 + 93) * 2];
    facerets[99 * 2 - 1] = advFacerets[1][(106 + 96) * 2 - 1];
    facerets[99 * 2] = advFacerets[1][(106 + 96) * 2];
    facerets[100 * 2 - 1] = advFacerets[1][(106 + 99) * 2 - 1];
    facerets[100 * 2] = advFacerets[1][(106 + 99) * 2];
    facerets[101 * 2 - 1] = advFacerets[1][(106 + 104) * 2 - 1];
    facerets[101 * 2] = advFacerets[1][(106 + 104) * 2];
    facerets[102 * 2 - 1] = advFacerets[1][(106 + 115) * 2 - 1];
    facerets[102 * 2] = advFacerets[1][(106 + 115) * 2];
    facerets[103 * 2 - 1] = advFacerets[1][(106 + 112) * 2 - 1];
    facerets[103 * 2] = advFacerets[1][(106 + 112) * 2];
    facerets[104 * 2 - 1] = advFacerets[1][(106 + 109) * 2 - 1];
    facerets[104 * 2] = advFacerets[1][(106 + 109) * 2];
    --LOG("USE ADV FACE LANDMARKS")
  end

  local lmk = {}
  for i = 1, 106 do
    lmk[i] = {facerets[i * 2 - 1], facerets[i * 2]}
  end
  
  local pry = {facerets[109 * 2 - 1], facerets[111 * 2 - 1], facerets[110 * 2 - 1]}
  
  for i = 1, 23 do
    lmk[i + 106] = {forehead_rets[i * 2 - 1], forehead_rets[i * 2]}
  end
  return lmk, pry
end


FaceliftBehavior:MemberRegister("Camera",
  venuscore.ScriptTypes.ReferenceType(
    apolloengine.CameraComponent:RTTI(),    
    nil,
    function(thiz, value)
      thiz.Camera = value
    end
))

FaceliftBehavior:MemberRegister("Recognition",
  venuscore.ScriptTypes.ReferenceType(
    cv.RecognitionComponent:RTTI(),    
    nil,
    function(thiz, value)
      thiz.Recognition = value
    end
))

FaceliftBehavior:MemberRegister("Fall_Off",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    nil,
    function(thiz, value)
      thiz["Fall_Off"] = value;
      facelift:_SetFallOff(value);
    end
  ))
  
FaceliftBehavior:MemberRegister("FaceFallOff_1",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    nil,
    function(thiz, value)
      thiz["FaceFallOff_1"] = value;
      facelift:_SetFaceFallOff1(value);
    end
  ))

FaceliftBehavior:MemberRegister("FaceFallOff_2",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    nil,
    function(thiz, value)
      thiz["FaceFallOff_2"] = value;
      facelift:_SetFaceFallOff2(value);
    end
  ))

FaceliftBehavior:MemberRegister("OutercanthusFallOff",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    nil,
    function(thiz, value)
      thiz["OutercanthusFallOff"] = value;
      facelift:_SetOutercanthusFallOff(value);
    end
  ))

FaceliftBehavior:MemberRegister("EyescaleFallOff",
  venuscore.ScriptTypes.FloatType(
    0, 1,
    nil,
    function(thiz, value)
      thiz["EyescaleFallOff"] = value;
      facelift:_SetEyescaleFallOff(value);
    end
  ))
  
function RegisterEditorLibrarySwitch(lib)
  -- Register Venus Editor UI for turning lib on/off and changing temporary coef value 
  FaceliftBehavior:MemberRegister(lib,  
  venuscore.ScriptTypes.ComboType(
    {
      { key = "On", value = true },
      { key = "Off", value = false },
    },
    nil,
    function(thiz, value)
      thiz[lib] = value;
      if thiz.inited then
        facelift:_Switch(lib, value);
      end
    end
  ));
  FaceliftBehavior:MemberRegister(lib.."_coef",
  venuscore.ScriptTypes.FloatType(
    lib_range[lib][1], lib_range[lib][2],
    nil,
    function(thiz, value)
      thiz[lib.."_coef"] = value;
      facelift:_SetTempCoef(lib, value);
    end
  ))
end
for key, index in pairs(KEYS) do
  KEYSINORDER[index-min_key_index+1] = key;
end

for i = 1, #KEYSINORDER do
  local key = KEYSINORDER[i]
  local index = KEYS[key]
  local attributes = {};
  LOG("KEYS id: " .. index)
  local libs = lib_map[i+min_key_index-1];
  if (libs~=nil) then
    for j = 1, #libs do
      RegisterEditorLibrarySwitch(libs[j])
    end
  end
  if (index==facelift_defined.EyeAngDis_start) then
    attributes = ATTRIBUTES_EYEAD
    for _, attribute in ipairs(attributes) do
      local var_key = key.."_"..attribute 
      if(attribute == "Ext1") then
        FaceliftBehavior:MemberRegister(var_key,
          venuscore.ScriptTypes.FloatType(
            0, 1,
            nil,
            function(thiz, value)
              thiz[var_key] = value
              thiz:ChangeValue(index, attribute, value)
            end
          )
        )
      elseif(attribute == "Ext2") then
        FaceliftBehavior:MemberRegister(var_key,
          venuscore.ScriptTypes.FloatType(
            0, 1,
            nil,
            function(thiz, value)
              thiz[var_key] = value
              thiz:ChangeValue(index, attribute, value)
            end
          )
        )
      elseif(attribute == "Dis") then
        FaceliftBehavior:MemberRegister(var_key,
          venuscore.ScriptTypes.FloatType(
            -1, 1,
            nil,
            function(thiz, value)
              thiz[var_key] = value
              thiz:ChangeValue(index, attribute, value)
            end
          )
        )
      else
        FaceliftBehavior:MemberRegister(var_key,
          venuscore.ScriptTypes.FloatType(
            -20, 20,
            nil,
            function(thiz, value)
              thiz[var_key] = value
              thiz:ChangeValue(index, attribute, value)
            end
          )
        )
      end
    end
  elseif (index == 0 or index == -1) then -- mouth
    attributes = ATTRIBUTES_MOUTH;
    for _, attribute in ipairs(attributes) do
      local var_key = key.."_"..attribute
      if(attribute == "Inten") then
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          -1, 1,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      elseif(attribute == "YScale") then
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          0, 3,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      else
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          0, 2,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      end
    end
  elseif(index == -2) then -- lip { "X", "Y", "RUP", "RDOWN" }
    attributes = ATTRIBUTES_LIP;
    local attribute = "X";
    local var_key = key.."_"..attribute;
    local max_num = 1.5;
    local min_num = -0.1;
    FaceliftBehavior:MemberRegister(var_key,
      venuscore.ScriptTypes.FloatType(
      min_num, max_num,
      nil,
      function(thiz, value)
        thiz[var_key] = value
        thiz:ChangeValue(index, attribute, value)
      end
      ))
    local attribute = "YUP";
    local var_key = key.."_"..attribute;
    FaceliftBehavior:MemberRegister(var_key,
      venuscore.ScriptTypes.FloatType(
        min_num, max_num,
        nil,
        function(thiz, value)
          thiz[var_key] = value
          thiz:ChangeValue(index, attribute, value)
          end
        ))
    local attribute = "YDOWN";
    local var_key = key.."_"..attribute;
    FaceliftBehavior:MemberRegister(var_key,
      venuscore.ScriptTypes.FloatType(
        min_num, max_num,
        nil,
        function(thiz, value)
          thiz[var_key] = value
          thiz:ChangeValue(index, attribute, value)
          end
        ))
    local attribute = "RUP";
    local var_key = key.."_"..attribute;
    local max_num = 2.0;
    local min_num = 0.0;
    FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          min_num, max_num,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
            end
          ))  
    local attribute = "RDOWN";
    local var_key = key.."_"..attribute;
    FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          min_num, max_num,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
            end
          ))
  elseif (index == -3) then
    attributes = ATTRIBUTES_ROUNDEYE;
    for _, attribute in ipairs(attributes) do
      local var_key = key.."_"..attribute
      if(attribute == "R") then
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          0, 3,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      elseif(attribute == "S") then
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          0, 1,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      end
    end
  elseif (index==facelift_defined.InnerCanthus_start + 1) then
    FaceliftBehavior:MemberRegister(key,
    venuscore.ScriptTypes.FloatType(
      0, 1,
      nil,
      function(thiz, value)
        thiz[key] = value
        thiz:ChangeValue(index, "DistAlpha", value)
      end
    ))
  elseif (index==facelift_defined.OuterCanthus_start + 3) then
    FaceliftBehavior:MemberRegister(key,
    venuscore.ScriptTypes.FloatType(
      0, 1,
      nil,
      function(thiz, value)
        thiz[key] = value
        thiz:ChangeValue(index, "DistAlpha", value)
      end
    ))
  else
    attributes = ATTRIBUTES;
    for _, attribute in ipairs(attributes) do
      local var_key = key.."_"..attribute
      local max_num = 0.2;
      local min_num = -0.1;
      if(index <= 0) then
        max_num = 1.5;
      end
      if (attribute == "X" or attribute == "Y") then
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          min_num, max_num,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      else
        FaceliftBehavior:MemberRegister(var_key,
        venuscore.ScriptTypes.FloatType(
          0, 2,
          nil,
          function(thiz, value)
            thiz[var_key] = value
            thiz:ChangeValue(index, attribute, value)
          end
        ))
      end
    end
  end
end

function FaceliftBehavior:SetImage(value)  
  -- LOG("FaceliftBehavior::SetImage");
  local tex = apolloengine.TextureEntity();
  tex:PushMetadata(apolloengine.TextureFileMetadata(
          apolloengine.TextureEntity.TU_STATIC,
          apolloengine.TextureEntity.PF_AUTO,1, false,
          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
          apolloengine.TextureEntity.TW_CLAMP_TO_EDGE,
          apolloengine.TextureEntity.TF_LINEAR,
          apolloengine.TextureEntity.TF_LINEAR,
          self.Image));
  tex:SetKeepSource(true);
  tex:CreateResource();
  
  
  self.bg_render:SetParameter(apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM, "_MainTex"), tex)  
  
  local ts = tex:GetSourceStream()

  -- local results = self.Recognition:GetResult();
  -- if results==nil or results[cv.RecognitionComponent.cvFace]==nil then
  --   LOG("NO LANDMARK RESULT");
  --   return nil;
  -- end

  -- local facerets = results[cv.RecognitionComponent.cvFace][1];
  -- local forehead_rets = results[cv.RecognitionComponent.cvForeheadDetection][1];
  
  -- local lmks = {}
  -- for k = 1, 106 do
  --   lmks[k] = {facerets[k * 2 - 1], facerets[k * 2]}
  -- end
  -- local pry = {facerets[109 * 2 - 1], facerets[111 * 2 - 1], facerets[110 * 2 - 1]}

  -- for k = 1, 23 do
  --   lmks[k + 106] = {forehead_rets[k * 2 - 1], forehead_rets[k * 2]}
  -- end
  local file_name = self.Image:match("(.*)%.")
  local config = venusjson.LaodJsonFile(file_name..".json");
  local points = config.value0.feature_points
  local lmks = {}
  for k = 1, #points/2 do
    lmks[k] = {points[k*2-1], points[k*2]}
  end
  local pyr = config.value0.pitch_yaw_roll
  local pry = {pyr[1], pyr[3], pyr[2]}
  local resolution = ts:GetSize();
  facelift:EditorUpdate(lmks, pry, tex, resolution)
end

function FaceliftBehavior:SwitchVideo()
  if not self.bg_node then
    return
  end
  if self.Video_Image == 1 then
    self.bg_node.Active = false
  else
    self.bg_node.Active = true
    if self.Image ~= "" then
      self:SetImage()
    end
  end    
end

FaceliftBehavior:MemberRegister("EnableFaceLift",  
venuscore.ScriptTypes.ComboType(
  {
    {
      key = "Open",
      value = 1
    },
    {
      key = "Close",
      value = 0
    },
  },
  nil,
  function(thiz, value)
    thiz.EnableFaceLift = value
    if thiz.inited then
      facelift:UpdateCoeff(value)
    end
  end
));


FaceliftBehavior:MemberRegister("Image",  
  venuscore.ScriptTypes.FilePathType(
    {"jpg", "png","tga","jpeg","bmp","JPG","PNG","TGA","JPEG"},
    nil,
    function(thiz, value)
      thiz.Image = value
      thiz:SetImage()
    end
  ));

FaceliftBehavior:MemberRegister("Video_Image",  
venuscore.ScriptTypes.ComboType(
  {
    {
      key = "Video",
      value = 1
    },
    {
      key = "Image",
      value = 2
    },
  },
  nil,
  function(thiz, value)
    thiz.Video_Image = value
    thiz:SwitchVideo()
  end
));

return FaceliftBehavior;