local vc = require "libvenuscore"
local BD = require "bluecore.bluedefined"
local BU = require "bluecore.blueutility"
local ArgInfo = require "bluecore.core.arg_info"

local FuncInfo = vc.Object:extend("FuncInfo");

function FuncInfo:new(classInfo)
  self.owner = classInfo ;
  self._RefactorList = setmetatable({}, { __mode = 'v' })
end

--  如果是直接注册的blueprint 代表是蓝图类 而不是prefab类 或者是端内运行的情况
-- register from bluefunction instance
function FuncInfo:RegisterBlueFunction(blueFtn)

  self._inputInfos = {}
  self._outputInfos = {}

  -- function name
  self._funcName = blueFtn:GetName();
  self._funcUid = blueFtn:GetFuncId();

  local inputInfos = blueFtn:GetInputInfo();
  local outputInfos = blueFtn:GetOutputInfo();

  -- input argument
  if inputInfos ~= nil then
    for _, reflectInfo in pairs(inputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterBlueFunPinInfo(reflectInfo);
      table.insert(self._inputInfos, argInfo);
    end
  end

  -- output argument
  if outputInfos ~= nil then
    for _, reflectInfo in pairs(outputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterBlueFunPinInfo(reflectInfo);
      table.insert(self._outputInfos, argInfo);
    end
  end

  blueFtn:AddRefactor(self);
end


function FuncInfo:RegisterPrefab(prefabJson)

  self._inputInfos = {}
  self._outputInfos = {}

  -- function name
  self._funcName = prefabJson.funcName ;
  self._funcUid =  prefabJson.funcUid ;

  local inputInfos = prefabJson.inputInfos;
  local outputInfos = prefabJson.outputInfos;

  -- input argument
  if inputInfos ~= nil then
    for _, reflectInfo in pairs(inputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterPrefab(reflectInfo);
      table.insert(self._inputInfos, argInfo);
    end
  end

  -- output argument
  if outputInfos ~= nil then
    for _, reflectInfo in pairs(outputInfos) do
      local argInfo = ArgInfo()
      argInfo:RegisterPrefab(reflectInfo);
      table.insert(self._outputInfos, argInfo);
    end
  end


end

function FuncInfo:GetInputInfo()
  return  self._inputInfos ;
end

function FuncInfo:GetOutputInfo()
  return  self._outputInfos
end

function FuncInfo:GetName()
  return self._funcName ;
end

function FuncInfo:GetFuncId()
  return self._funcUid ;
end

function FuncInfo:AddRefactor(callback)

  if _KRATOSEDITOR then

    local cb = callback ;
    -- 检查节点是否实现了指定的接口
    -- notify change
    if (cb._OnInfoChange == nil ) then error("_OnInfoChange not implement") end
    if (cb._OnEditorPinDelete == nil) then error("_OnEditorPinDelete not implement") end
    if (cb._OnEditorPinAdd == nil) then error("_OnEditorPinAdd not implement") end
    -- FIXME(hhl) 函数删除
    -- add or remote from refactorList/callbackList
    if (cb._OnDeserializePost == nil ) then error("_OnDeserializePost not implement") end
    if (cb._OnCreateWithEditor == nil ) then error("_OnCreateWithEditor not implement") end
    if (cb._OnDeleteEd == nil ) then error("_OnDeleteEd not implement") end
    -- 当调用节点被删除的时候, 从引用列表中移除(FIXME(hhl)也可以不调用 RemoveRefactor ,这个table value是weaker的)

    -- 加入到该函数的引用列表,当函数的定义/声明修改的时候,进行refactor
    table.insert(self._RefactorList, callback);

  end -- 非编辑器启动不加入refactor列表
end


function FuncInfo:InfoChange(reflectIdx, type, val, isInput)
  --  更新自己的信息

  local argInfo = isInput and self._inputInfos or self._outputInfos
  argInfo = argInfo[reflectIdx]

  if type == BD.FuncInfoProp.Name then
    argInfo:ChangeName(val)
  elseif type == BD.FuncInfoProp.BaseType then
    argInfo:ChangeRtti(val, BU:GetDefaultByRtti(val));
  elseif type == BD.FuncInfoProp.Value then
    argInfo:ChangeDefault(val);
  end

  -- 通知引用该函数的节点
  for _, cb in pairs(self._RefactorList) do
    cb:InfoChange(reflectIdx, type, val, isInput);
  end

end

function FuncInfo:EditorPinDelete(isInput, idx)

  if isInput then
    -- _inputInfos是数组 移除前面的后面自动移前
    table.remove(self._inputInfos, idx)
  else
    table.remove(self._outputInfos, idx)
  end

  for _, cb in pairs(self._RefactorList) do
    cb:EditorPinDelete(isInput, idx);
  end

end


function FuncInfo:EditorPinAdd(isInput, reflectInfo)

  local argInfo = ArgInfo()
  argInfo:RegisterBlueFunPinInfo(reflectInfo);
  if isInput then
    table.insert(self._inputInfos,  argInfo);
  else
    table.insert(self._outputInfos, argInfo);
  end

  for _, cb in pairs(self._RefactorList) do
    cb:EditorPinAdd(isInput, reflectInfo);
  end

end


function FuncInfo:RemoveRefactor(callback)
  if _KRATOSEDITOR then
    for idx, cb in pairs(self._RefactorList) do
      if cb == callback then
        self._RefactorList[idx] = nil
      end
    end
  end
end

function FuncInfo:ToBeDeletedEd()
  for _, cb in pairs(self._RefactorList) do
    cb:ToBeDeletedEd();
  end
  self.owner:ToBeDeletedEd(self)
  self.owner = nil
end

function FuncInfo:NameChanged(oldName, newName)
  self._funcName = newName
  for _, cb in pairs(self._RefactorList) do
    cb:NameChanged(oldName, newName);
  end
end

function FuncInfo:GetRefactorList()
  return self._RefactorList;
end

function FuncInfo:Dump()
  LOG("----name:"..tostring(self._funcName))
  LOG("---- uid:"..tostring(self._funcUid))
  for i, argInfo in pairs( self._inputInfos) do
    LOG("---- arg "..tostring(i)..":")
    argInfo:Dump();
  end
  for i, argInfo in pairs( self._outputInfos) do
    LOG("---- ret "..tostring(i)..":")
    argInfo:Dump();
  end
end

return FuncInfo