require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"

local RoundEye = {}

function RoundEye:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function RoundEye:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
    self.coef = self.coef_temp;
    -- self.circlecoef = self.coef_temp;
    -- ERROR("self.circlecoef set to " .. self.circlecoef)
	end  
end

function RoundEye:_InitParams(defined)
  beautydefined = defined
  --大眼参数
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_LEFT_EYE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_LEFT_EYE");
  apolloengine.ShaderEntity.UNIFORM_RIGHT_EYE = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_RIGHT_EYE");
  apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_RADIUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_ROUNDEYE_RADIUS");
  apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_EYECOEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_ROUNDEYE_EYECOEF");
end

function RoundEye:SetParams(config, shaderParams)
  shaderParams.lefteyes = mathfunction.vector2(0,0);
  shaderParams.righteyes = mathfunction.vector2(0,0);
  shaderParams.radiuses = mathfunction.vector2(0,0);
end

function RoundEye:_InitQuad(quadnode, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_LEFT_EYE, shaderParams.lefteyes);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_RIGHT_EYE, shaderParams.righteyes);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_RADIUS, shaderParams.radiuses);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_EYECOEF, mathfunction.vector1(0.0));
end

function RoundEye:_DrawQuad(quadnode, keypoints, rate, config) 
	
	--左眼
	local c1 = beautyutils._ArrayToVec2(keypoints[75]);
	local pt2 = beautyutils._ArrayToVec2(keypoints[55]);
	local r1 = beautyutils._GetVector2dLength(pt2 - c1) ;
	local lefteyes = c1;
	self.r1 = r1

	--右眼 
	local c2 = beautyutils._ArrayToVec2(keypoints[78]);
	pt2 = beautyutils._ArrayToVec2(keypoints[60]);
	local r2 = beautyutils._GetVector2dLength(pt2 - c2);
	local righteyes = c2;
	self.r2 = r2

	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_LEFT_EYE,lefteyes);
	quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_RIGHT_EYE,righteyes);
		
  local radiuses = mathfunction.vector2(self.r1 * (self.coef*rate+1) * config.RoundeyeRadius, 
  self.r2 * (self.coef * rate+1) * config.RoundeyeRadius);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_RADIUS,radiuses);

  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_EYECOEF,mathfunction.vector1(config.RoundeyeScale * self.coef * rate))
end

function RoundEye:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);

  local coef = self.coef
  local radius = config.RoundeyeRadius
  local scale = config.RoundeyeScale
  
  local indexRange = self:_getIndexRange(#keypoints)
  for i = 1, #indexRange do
    local index = indexRange[i] + 1
    local newpoint = beautyutils.ScaleWarp(keypoints[index], keypoints[75], self.r1 * (coef * rate + 1) * radius, 0.7 * scale * coef * rate, aspect_ratio);
    local tmp = {newpoint[1] - keypoints[index][1], newpoint[2] - keypoints[index][2]}
    newpoint = beautyutils.ScaleWarp(keypoints[index], keypoints[78], self.r2 * (coef * rate + 1) * radius, 0.7 * scale * coef * rate, aspect_ratio);
    tmp = {tmp[1] + newpoint[1] - keypoints[index][1], tmp[2] + newpoint[2] - keypoints[index][2]}
    keypointsDisplacement[index][1] = keypointsDisplacement[index][1] + tmp[1]
    keypointsDisplacement[index][2] = keypointsDisplacement[index][2] + tmp[2]
  end
  
end

function RoundEye:_getIndexRange(num)
  local min = 30
  local max = 79
  local range= {0, 1, 2, 104, 105}
  for i = min, max do
    table.insert(range, i)
  end
  
  if num >= 240 then
    min = 0
    max = 43
    for i = min, max do
      table.insert(range, i + 106)
    end
  end
  return range
end

function RoundEye:UpdateParaValue(quadnode, config, rate, attribute, value)
  
  if(attribute == "R") then
    config.circle_eye_param_radius = value;
  elseif (attribute == "S") then
    config.circle_eye_param_scale = value;
  end
  
  local radiuses = mathfunction.vector2(self.r1 * (self.coef*rate+1) * config.RoundeyeRadius, 
  self.r2 * (self.coef * rate+1) * config.RoundeyeRadius);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_RADIUS,radiuses);

  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_EYECOEF,mathfunction.vector1(config.RoundeyeScale * self.coef * rate))
end

function RoundEye:UpdateCoeff(quadnode, value, rate, config)
  if(value == 1) then
    self.coef = 1.0; -- full warping
  else
    self.coef = 0.0; -- no warping
  end
  LOG("Round Eye coef set to: " .. self.coef);
  
  local radiuses = mathfunction.vector2(self.r1 * (self.coef*rate+1) * config.RoundeyeRadius, 
  self.r2 * (self.coef * rate+1) * config.RoundeyeRadius);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_RADIUS,radiuses);

  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_ROUNDEYE_EYECOEF,mathfunction.vector1(config.RoundeyeScale * self.coef * rate))
end

function RoundEye:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
end

function RoundEye:GetParams(config)
  return {config.RoundeyeRadius, config.RoundeyeScale};
end

function RoundEye:Release()

end
  
return RoundEye