require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local NoseLength = {}

function NoseLength:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function NoseLength:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function NoseLength:_InitParams(defined)
  beautydefined = defined
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_PTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_NOSELENGTH_PTS");
  apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_COEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_NOSELENGTH_COEF");
  apolloengine.ShaderEntity.NOSELENGTH_PARAM = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"NOSELENGTH_PARAM");
  apolloengine.ShaderEntity.NOSELENGTH_PARAMRADIUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"NOSELENGTH_PARAMRADIUS");
end

function NoseLength:SetParams(config, shaderParams)
  shaderParams.nosepointArray = mathfunction.vector4array();
  shaderParams.noseParamArray = mathfunction.vector2array();
  shaderParams.noseParamRadiusArray = mathfunction.vector2array();

  for i = 1, #beautydefined.NoselengthKey, 1 do
    shaderParams.noseParamArray:PushBack(beautyutils._ArrayToVec2(config.NoselengthParam[i]));
    if type(beautydefined.NoselengthKey[i])=="table" then
      shaderParams.noseParamArray:PushBack(beautyutils._ArrayToVec2(config.NoselengthParam[i]));
    end
  end
  
  for i = 1, #beautydefined.NoselengthKey, 1 do
    shaderParams.noseParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.NoselengthParamRadius[i]));
    if type(beautydefined.NoselengthKey[i])=="table" then
      shaderParams.noseParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.NoselengthParamRadius[i]));
    end
  end
  self:_InitLens(config)
end

function NoseLength:_InitLens(config)
  self.lens = {}
  local index2 = beautydefined.NoselengthDir+1;
  for i = 1, #beautydefined.NoselengthKey do
    if type(beautydefined.NoselengthKey[i])=="table" then
      local index1 = beautydefined.NoselengthKey[i][1]+1;
      self.lens[#self.lens + 1] = Len(index1, index2, beautydefined.NoseLengthParamScript[i][1], beautydefined.NoseLengthParamScript[i][2], beautydefined.NoseLengthParamRadiusScript[i])
      index1 = beautydefined.NoselengthKey[i][2]+1;
      self.lens[#self.lens + 1] = Len(index1, index2, beautydefined.NoseLengthParamScript[i][1], beautydefined.NoseLengthParamScript[i][2], beautydefined.NoseLengthParamRadiusScript[i])
    else 
      local index1 = beautydefined.NoselengthKey[i]+1;
      self.lens[#self.lens + 1] = Len(index1, index2, beautydefined.NoseLengthParamScript[i][1], beautydefined.NoseLengthParamScript[i][2], beautydefined.NoseLengthParamRadiusScript[i])
    end
  end
end

function NoseLength:_InitQuad(quadnode, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_PTS, shaderParams.nosepointArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.NOSELENGTH_PARAM, shaderParams.noseParamArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.NOSELENGTH_PARAMRADIUS, shaderParams.noseParamRadiusArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_COEF, mathfunction.vector1(0.0));
end

function NoseLength:_DrawQuad(quadnode, keypoints, rate)
  local nosePointArray = mathfunction.vector4array();
  for i = 1, #beautydefined.NoselengthKey, 1 do
    local index2 = beautydefined.NoselengthDir+1;
    
    if type(beautydefined.NoselengthKey[i])=="table" then
      local index1 = beautydefined.NoselengthKey[i][1]+1;
      nosePointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
      index1 = beautydefined.NoselengthKey[i][2]+1;
      nosePointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
    else 
      local index1 = beautydefined.NoselengthKey[i]+1;
      nosePointArray:PushBack(beautyutils._ArrayToVec4(keypoints[index1], keypoints[index2]));
    end

  end
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_COEF,mathfunction.vector1(self.coef));
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_PTS,nosePointArray);
end

function NoseLength:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
  local coef = self.coef
  local tmpDis = {}
  for i = 1, #self.lens do
    local srcIdx = self.lens[i]:GetSrcIdx()
    local dstIdx = self.lens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.lens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)  
    local tmp = self.lens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx] = tmp
  end
  
  for k, v in pairs(tmpDis) do
    keypointsDisplacement[k][1] = keypointsDisplacement[k][1] + v[1]
    keypointsDisplacement[k][2] = keypointsDisplacement[k][2] + v[2]
  end
  
  for i = 1, #beautydefined.NoseLengthWarpMap do
    local src = beautydefined.NoseLengthWarpMap[i][1] + 1
    local dst = beautydefined.NoseLengthWarpMap[i][2] + 1
    keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.NoseLengthWarpCoef[i]
    keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.NoseLengthWarpCoef[i]
  end
end

function NoseLength:UpdateParaValue(quadnode, config, key_index, attribute, value)
  if (attribute == "X") then
    config.NoselengthParam[key_index - beautydefined.Noselength_start + 1][1] = value;
  elseif (attribute == "Y") then
    config.NoselengthParam[key_index - beautydefined.Noselength_start + 1][2] = value;
  elseif (attribute == "R") then
    config.NoselengthParamRadius[key_index - beautydefined.Noselength_start + 1][1] = value;
  end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.NOSELENGTH_PARAM, shaderParams.noseParamArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.NOSELENGTH_PARAMRADIUS, shaderParams.noseParamRadiusArray);
end

function NoseLength:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0;
  else
    self.coef = 0.0;
  end
  LOG("Nose Length coef set to: " .. self.coef);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_NOSELENGTH_COEF,mathfunction.vector1(self.coef));
end

function NoseLength:GetParams(config)
  local noselengthParam = beautyutils.Clone(config.NoselengthParam);
  local noselengthParamRadius = beautyutils.Clone(config.NoselengthParamRadius);
  LOG("noselengthParam size: " .. #noselengthParam);
  return noselengthParam, noselengthParamRadius
end

function NoseLength:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --LOG("Nose Length coef: " .. self.coef)
end
  
function NoseLength:Release()
  self.lens = {}
end  
  
return NoseLength