require "venusdebug"
local apolloengine = require "apolloengine"
local mathfunction = require "mathfunction"
local beautydefined
local beautyutils = require "beauty.utils"
local Len = require "beauty.len"

local InnerCanthus = {}

function InnerCanthus:_Switch(on)
	-- venus editor on/off facial effect's switch
	self.on = on;
	if self.on then
		self.coef = self.coef_temp;
	else
		self.coef = 0;
	end
end

function InnerCanthus:_SetTempCoef(coef)
	-- venus editor temporary params value, used when switched on
	self.coef_temp = coef;
	if self.on then
		self.coef = self.coef_temp;
	end  
end

function InnerCanthus:_InitParams(defined)
  beautydefined = defined
  self.coef = self.coef or 0
  self.coef_temp = 1.0;
  -- shader params
  apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_PTS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_INNERCANTHUS_PTS");
  apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_COEF = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"UNIFORM_INNERCANTHUS_COEF");
  apolloengine.ShaderEntity.INNERCANTHUS_PARAM = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"INNERCANTHUS_PARAM");
  apolloengine.ShaderEntity.INNERCANTHUS_PARAMRADIUS = apolloengine.IMaterialSystem:NewParameterSlot(apolloengine.ShaderEntity.UNIFORM,"INNERCANTHUS_PARAMRADIUS");
end

function InnerCanthus:SetParams(config, shaderParams)
  shaderParams.innercanthusPointArray = mathfunction.vector4array();
  shaderParams.innercanthusParamArray = mathfunction.vector2array();
  shaderParams.innercanthusParamRadiusArray = mathfunction.vector2array();

  for i = 1, #beautydefined.InnercanthusKey, 1 do
    shaderParams.innercanthusParamArray:PushBack(beautyutils._ArrayToVec2(config.InnercanthusParam[1]));
    shaderParams.innercanthusParamRadiusArray:PushBack(beautyutils._ArrayToVec2(config.InnercanthusParamRadius[1]));
  end
  self:_InitLens(config)
end

function InnerCanthus:_InitLens(config)
  self.lens = {}
  for i = 1, #beautydefined.InnercanthusKey do
    local index1 = beautydefined.InnercanthusKey[i][1] + 1
    local index2 = beautydefined.InnercanthusDir[i] + 1
    self.lens[#self.lens + 1] = Len(index1, index2, -config.InnercanthusParam[1][1], config.InnercanthusParam[1][2], config.InnercanthusParamRadius[1][1])
  end
end

function InnerCanthus:_InitQuad(quadnode, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_PTS, shaderParams.innercanthusPointArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.INNERCANTHUS_PARAM, shaderParams.innercanthusParamArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.INNERCANTHUS_PARAMRADIUS, shaderParams.innercanthusParamRadiusArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_COEF, mathfunction.vector1(0.0));
end

function InnerCanthus:_DrawQuad(quadnode, keypoints, rate, config)
  -- param: args = {rate, config}
  local innercanthusPointArray = mathfunction.vector4array();
  for i = 1, #beautydefined.InnercanthusKey, 1 do
    local target = keypoints[beautydefined.InnercanthusDir[i]+1];
    local alpha = config.InnercanthusDistAlphaParam;

    local p_1 = keypoints[beautydefined.InnercanthusKey[i][1]+1];
    local p_2 = keypoints[beautydefined.InnercanthusKey[i][2]+1];
    local source = {(1-alpha) * p_1[1] + alpha * p_2[1], (1-alpha) * p_1[2] + alpha * p_2[2]};

    innercanthusPointArray:PushBack(beautyutils._ArrayToVec4(source, target));
  end
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_COEF,mathfunction.vector1(self.coef * rate));
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_PTS,innercanthusPointArray);
end

function InnerCanthus:_getWarpedKeyPointsDisplacement(keypoints, keypointsDisplacement, aspect_ratio, config, rate)
  if self.coef == 0 then
    return;
  end
  local facedir = {keypoints[75][1] - keypoints[78][1], keypoints[75][2] - keypoints[78][2]};
  facedir[2] = facedir[2] * aspect_ratio
  facedir[1] = facedir[1] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  facedir[2] = facedir[2] / math.sqrt(facedir[1] * facedir[1] + facedir[2] * facedir[2]);
  
    local coef = self.coef * rate
  local tmpDis = {}
  for i = 1, #self.lens do
    local srcIdx = self.lens[i]:GetSrcIdx()
    local dstIdx = self.lens[i]:GetDstIdx()
    local src = keypoints[srcIdx]
    local dst = keypoints[dstIdx]
    self.lens[i]:InitParams(src, dst, facedir, coef, aspect_ratio)  
    local tmp = self.lens[i]:TransformWarp(src, aspect_ratio)
    tmpDis[srcIdx] = tmp
  end
  
  for k, v in pairs(tmpDis) do
    keypointsDisplacement[k][1] = keypointsDisplacement[k][1] + v[1]
    keypointsDisplacement[k][2] = keypointsDisplacement[k][2] + v[2]
  end
  
  if #keypoints >= 240 then
    for i = 1, #beautydefined.AdvInnercanthusWarpMap do
      local src = beautydefined.AdvInnercanthusWarpMap[i][1] + 107
      local dst = beautydefined.AdvInnercanthusWarpMap[i][2] + 1
      keypointsDisplacement[src][1] = keypointsDisplacement[src][1] + tmpDis[dst][1] * beautydefined.AdvInnercanthusWarpCoef[i]
      keypointsDisplacement[src][2] = keypointsDisplacement[src][2] + tmpDis[dst][2] * beautydefined.AdvInnercanthusWarpCoef[i]
    end
  end
end

function InnerCanthus:UpdateParaValue(quadnode, config, key_index, attribute, value)
  if (attribute == "X") then
    config.InnercanthusParam[key_index - beautydefined.InnerCanthus_start + 1][1] = value;
  elseif (attribute == "Y") then
    config.InnercanthusParam[key_index - beautydefined.InnerCanthus_start + 1][2] = value;
  elseif (attribute == "R") then
    config.InnercanthusParamRadius[key_index - beautydefined.InnerCanthus_start + 1][1] = value;
  elseif (attribute == "DistAlpha") then
    config.InnercanthusDistAlphaParam = value;
  end
  local shaderParams = {}
  self:SetParams(config, shaderParams)
  quadnode:SetParameter(apolloengine.ShaderEntity.INNERCANTHUS_PARAM, shaderParams.innercanthusParamArray);
  quadnode:SetParameter(apolloengine.ShaderEntity.INNERCANTHUS_PARAMRADIUS, shaderParams.innercanthusParamRadiusArray);
end

function InnerCanthus:UpdateCoeff(quadnode, value, rate)
  if(value == 1) then
    self.coef = 1.0;
  else
    self.coef = 0.0;
  end
  LOG("Nose Length coef set to: " .. self.coef);
  quadnode:SetParameter(apolloengine.ShaderEntity.UNIFORM_INNERCANTHUS_COEF,mathfunction.vector1(self.coef * rate));
end

function InnerCanthus:GetParams(config)
  local innercanthusParam = beautyutils.Clone(config.InnercanthusParam);
  local innercanthusParamRadius = beautyutils.Clone(config.InnercanthusParamRadius);
  local InnercanthusDistAlphaParam = config.InnercanthusDistAlphaParam;
  LOG("innercanthusParam size: " .. #innercanthusParam);
  return innercanthusParam, innercanthusParamRadius, InnercanthusDistAlphaParam 
end

function InnerCanthus:SetCoef(coef)
  if coef == nil then
    coef = 0
  end
  self.coef = coef;
  --LOG("Innercanthus coef: " .. self.coef)
end

function InnerCanthus:Release()
  self.lens = {}
end
  
return InnerCanthus