## SDDFInterface

The Discovery Framework Core

This package contains all of the code for iOS and Android to load and find Discoverable interfaces at runtime. It is used in conjunction with the SDDFCore package, which generates the Discoverable mappings file.

## Overriding Singletons on Android

If you have a situation where you have a shared implementation of a Discoverable contract, accessed via UniqueDiscovery, and it's implemented in a shared module, but you don't want to change the code in the shared module but do want different functionality, you can override the functionality according to the method below.

To clarify, if you have a situation like this:

```
// Module A:

public interface SingletonContract {
    public void entryPointMethod();
}


// Module B (depends on Module A):

public class SingletonImplementation implements SingletonContract, Discoverable {
    @Override
    public void entryPointMethod() {
        // some shared functionality
    }
}

// Module C (depends on Module B):

    // some code somewhere
    SingletonContract singleton = UniqueDiscovery.of(SingletonContract.class).value();


// Module D (depends on C and produces an APK)
// - includes SingletonImplementation

// Module E (depends on C and produces a different APK)
// - includes SingletonImplementation

```

The above matches the case of the Comixology Bespoke app, which lives in the same package as the KindleAndroidReader app. Both of those are defined as modules within the package, and they both depend on another module, ThirdPartyCore. When a UniqueDiscovery contract is implemented and found in ThirdPartyCore, it is then forced into both KindleAndroidReader and Comixology. If either team wants to modify the functionality, they have to test both applications thoroughly, or introduce risks and possibly unwanted changes or bugs.

To work around this, we can try to subclass the shared implementation, and override the functionality like so:

```
// Module E (depends on C and produces a different APK)

public class ModuleSpecificSingletonOverride extends SingletonImplementation implements SingletonContract, Discoverable {

    // ***** NOTE THAT IT WAS CRUCIAL THAT THIS CLASS IMPLEMENTS BOTH THE CONTRACT AND DISCOVERABLE, otherwise the DiscoveryTool won't find it

    @Override
    public void entryPointMethod() {
        // some specialized functionality

        super.entryPointMethod();

        // some more specialized functionality
    }
}
```

But if a new implementation of SingletonContract is defined in one of the APK producing modules (for instance, in the Comixology Bespoke app), an exception at runtime will be thrown, because then there will be two implementations defined for the same interface contract, and UniqueDiscovery only allows one, as it's meant to be used for Singletons.

### A solution

To work around this, the DiscoveryTool allows for a configuration file to be specified, which allows for classes to be excluded. Use the -c option to specify the file, with a format like this:

```
// Comments are acceptable
// Format is:
// $TYPE $name
// Valid types are: typeExclusion, packageExclusion, contractExclusion, packagePrefix

typeExclusion SingletonImplementation
```

Now, the shared implementation of SingletonImplementation won't be included when the discovery mappings are generated, and the only implementation that will be loaded at runtime will be the ModuleSpecificSingletonOverride.

### Caveat

In order to get this to work, you'll have to modify the build rules that call DiscoveryTool. In KindleAndroidReader, once you've created a DiscoveryTool config file, ensure it's named 'dependencies_config.txt', put it into your module's root directory, and then make it an input file to the generate tasks file in your APK's build.gradle. Something along these lines:

```
afterEvaluate { project ->
    android.applicationVariants.all { variant ->
        //get variant flavor name e.g kfa, kfc, fos4, fos5
        def flavorName = variant.productFlavors.name[0]

        //get generateClasses task name for the variant
        def generateClassesTaskName = 'generateClasses' + variant.javaCompile.name + '-' + flavorName + '-' + variant.buildType.name

        // Ensure that we can override singletons in the Discovery system
        tasks.named(generateClassesTaskName).configure {
            inputs.file "${project.projectDir}/dependencies_config.txt"   // <--------- the really important part!!!!
        }
```


Copyright © 2018 Amazon.com. All Rights Reserved.
